/* Copyright (C) 2004 - 2006  db4objects Inc.  http://www.db4o.com

This file is part of the db4o open source object database.

db4o is free software; you can redistribute it and/or modify it under
the terms of version 2 of the GNU General Public License as published
by the Free Software Foundation and as clarified by db4objects' GPL 
interpretation policy, available at
http://www.db4o.com/about/company/legalpolicies/gplinterpretation/
Alternatively you can write to db4objects, Inc., 1900 S Norfolk Street,
Suite 350, San Mateo, CA 94403, USA.

db4o is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. */
namespace Db4objects.Db4o.Internal.CS
{
	public sealed class ServerMessageDispatcher : Sharpen.Lang.Thread
	{
		private string i_clientName;

		private bool i_loggedin;

		private long i_lastClientMessage;

		private readonly Db4objects.Db4o.Internal.LocalObjectContainer i_mainStream;

		private Db4objects.Db4o.Internal.Transaction i_mainTrans;

		private int i_pingAttempts = 0;

		private int i_nullMessages;

		private bool i_rollbackOnClose = true;

		private bool i_sendCloseMessage = true;

		private readonly Db4objects.Db4o.Internal.CS.ObjectServerImpl i_server;

		private Db4objects.Db4o.Foundation.Network.ISocket4 i_socket;

		private Db4objects.Db4o.Internal.LocalObjectContainer i_substituteStream;

		private Db4objects.Db4o.Internal.Transaction i_substituteTrans;

		private Db4objects.Db4o.Foundation.Hashtable4 _queryResults;

		private Db4objects.Db4o.Internal.Config4Impl i_config;

		internal readonly int i_threadID;

		internal ServerMessageDispatcher(Db4objects.Db4o.Internal.CS.ObjectServerImpl aServer
			, Db4objects.Db4o.Internal.LocalObjectContainer aStream, Db4objects.Db4o.Foundation.Network.ISocket4
			 aSocket, int aThreadID, bool loggedIn)
		{
			SetDaemon(true);
			i_loggedin = loggedIn;
			i_lastClientMessage = Sharpen.Runtime.CurrentTimeMillis();
			i_server = aServer;
			i_config = (Db4objects.Db4o.Internal.Config4Impl)i_server.Configure();
			i_mainStream = aStream;
			i_threadID = aThreadID;
			SetName("db4o message server " + aThreadID);
			i_mainTrans = aStream.NewTransaction();
			try
			{
				i_socket = aSocket;
				i_socket.SetSoTimeout(((Db4objects.Db4o.Internal.Config4Impl)aServer.Configure())
					.TimeoutServerSocket());
			}
			catch (System.Exception e)
			{
				i_socket.Close();
				throw (e);
			}
		}

		public void Close()
		{
			lock (this)
			{
				if (IsClosed())
				{
					return;
				}
				CloseSubstituteStream();
				SendCloseMessage();
				RollbackMainTransaction();
				CloseSocket();
				RemoveFromServer();
			}
		}

		private void SendCloseMessage()
		{
			try
			{
				if (i_sendCloseMessage)
				{
					Write(Db4objects.Db4o.Internal.CS.Messages.Msg.CLOSE);
				}
			}
			catch (System.Exception e)
			{
			}
		}

		private void RollbackMainTransaction()
		{
			if (i_mainStream != null && i_mainTrans != null)
			{
				i_mainTrans.Close(i_rollbackOnClose);
			}
		}

		private void RemoveFromServer()
		{
			try
			{
				i_server.RemoveThread(this);
			}
			catch (System.Exception e)
			{
			}
		}

		private void CloseSocket()
		{
			try
			{
				i_socket.Close();
			}
			catch (System.Exception e)
			{
			}
			i_socket = null;
		}

		private bool IsClosed()
		{
			return i_socket == null;
		}

		private void CloseSubstituteStream()
		{
			if (i_substituteStream != null)
			{
				if (i_substituteTrans != null)
				{
					i_substituteTrans.Close(i_rollbackOnClose);
					i_substituteTrans = null;
				}
				try
				{
					i_substituteStream.Close();
				}
				catch (System.Exception e)
				{
				}
				i_substituteStream = null;
			}
		}

		private Db4objects.Db4o.Internal.LocalObjectContainer GetStream()
		{
			if (i_substituteStream != null)
			{
				return i_substituteStream;
			}
			return i_mainStream;
		}

		internal Db4objects.Db4o.Internal.Transaction GetTransaction()
		{
			if (i_substituteTrans != null)
			{
				return i_substituteTrans;
			}
			return i_mainTrans;
		}

		public override void Run()
		{
			while (i_socket != null)
			{
				try
				{
					if (!MessageProcessor())
					{
						break;
					}
				}
				catch (System.Exception e)
				{
					if (i_mainStream == null || i_mainStream.IsClosed())
					{
						break;
					}
					if (!i_socket.IsConnected())
					{
						break;
					}
					i_nullMessages++;
				}
				if (i_nullMessages > 20 || PingClientTimeoutReached())
				{
					if (i_pingAttempts > 5)
					{
						GetStream().LogMsg(33, i_clientName);
						break;
					}
					if (null == i_socket)
					{
						break;
					}
					Write(Db4objects.Db4o.Internal.CS.Messages.Msg.PING);
					i_pingAttempts++;
				}
			}
			Close();
		}

		private bool PingClientTimeoutReached()
		{
			return (Sharpen.Runtime.CurrentTimeMillis() - i_lastClientMessage > i_config.TimeoutPingClients
				());
		}

		private bool MessageProcessor()
		{
			Db4objects.Db4o.Internal.CS.Messages.Msg message = Db4objects.Db4o.Internal.CS.Messages.Msg
				.ReadMessage(GetTransaction(), i_socket);
			if (message == null)
			{
				i_nullMessages++;
				return true;
			}
			i_lastClientMessage = Sharpen.Runtime.CurrentTimeMillis();
			i_nullMessages = 0;
			i_pingAttempts = 0;
			if (!i_loggedin)
			{
				if (Db4objects.Db4o.Internal.CS.Messages.Msg.LOGIN.Equals(message))
				{
					string userName = ((Db4objects.Db4o.Internal.CS.Messages.MsgD)message).ReadString
						();
					string password = ((Db4objects.Db4o.Internal.CS.Messages.MsgD)message).ReadString
						();
					Db4objects.Db4o.User found = i_server.GetUser(userName);
					if (found != null)
					{
						if (found.password.Equals(password))
						{
							i_clientName = userName;
							i_mainStream.LogMsg(32, i_clientName);
							int blockSize = i_mainStream.BlockSize();
							int encrypt = i_mainStream.i_handlers.i_encrypt ? 1 : 0;
							Write(Db4objects.Db4o.Internal.CS.Messages.Msg.LOGIN_OK.GetWriterForInts(GetTransaction
								(), new int[] { blockSize, encrypt }));
							i_loggedin = true;
							SetName("db4o server socket for client " + i_clientName);
						}
						else
						{
							Write(Db4objects.Db4o.Internal.CS.Messages.Msg.FAILED);
							return false;
						}
					}
					else
					{
						Write(Db4objects.Db4o.Internal.CS.Messages.Msg.FAILED);
						return false;
					}
				}
				return true;
			}
			if (message.ProcessAtServer(this))
			{
				return true;
			}
			ProcessSpecialMsg(message);
			return true;
		}

		public bool ProcessSpecialMsg(Db4objects.Db4o.Internal.CS.Messages.Msg message)
		{
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.PING.Equals(message))
			{
				WriteOK();
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.OBJECTSET_FINALIZED.Equals(message))
			{
				int queryResultID = ((Db4objects.Db4o.Internal.CS.Messages.MsgD)message).ReadInt(
					);
				QueryResultFinalized(queryResultID);
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.CLOSE.Equals(message))
			{
				Write(Db4objects.Db4o.Internal.CS.Messages.Msg.CLOSE);
				GetTransaction().Commit();
				i_sendCloseMessage = false;
				GetStream().LogMsg(34, i_clientName);
				return false;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.IDENTITY.Equals(message))
			{
				RespondInt((int)GetStream().GetID(GetStream().Identity()));
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.CURRENT_VERSION.Equals(message))
			{
				long ver = 0;
				lock (GetStream())
				{
					ver = GetStream().CurrentVersion();
				}
				Write(Db4objects.Db4o.Internal.CS.Messages.Msg.ID_LIST.GetWriterForLong(GetTransaction
					(), ver));
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.RAISE_VERSION.Equals(message))
			{
				long minimumVersion = ((Db4objects.Db4o.Internal.CS.Messages.MsgD)message).ReadLong
					();
				Db4objects.Db4o.Internal.ObjectContainerBase stream = GetStream();
				lock (stream)
				{
					stream.RaiseVersion(minimumVersion);
				}
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.GET_THREAD_ID.Equals(message))
			{
				RespondInt(i_threadID);
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.SWITCH_TO_FILE.Equals(message))
			{
				SwitchToFile(message);
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.SWITCH_TO_MAIN_FILE.Equals(message))
			{
				SwitchToMainFile();
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.USE_TRANSACTION.Equals(message))
			{
				UseTransaction(message);
				return true;
			}
			return true;
		}

		private void WriteOK()
		{
			Write(Db4objects.Db4o.Internal.CS.Messages.Msg.OK);
		}

		private void QueryResultFinalized(int queryResultID)
		{
			_queryResults.Remove(queryResultID);
		}

		public void MapQueryResultToID(Db4objects.Db4o.Internal.CS.LazyClientObjectSetStub
			 stub, int queryResultID)
		{
			if (_queryResults == null)
			{
				_queryResults = new Db4objects.Db4o.Foundation.Hashtable4();
			}
			_queryResults.Put(queryResultID, stub);
		}

		public Db4objects.Db4o.Internal.CS.LazyClientObjectSetStub QueryResultForID(int queryResultID
			)
		{
			return (Db4objects.Db4o.Internal.CS.LazyClientObjectSetStub)_queryResults.Get(queryResultID
				);
		}

		private void SwitchToFile(Db4objects.Db4o.Internal.CS.Messages.Msg message)
		{
			lock (i_mainStream.i_lock)
			{
				string fileName = ((Db4objects.Db4o.Internal.CS.Messages.MsgD)message).ReadString
					();
				try
				{
					CloseSubstituteStream();
					i_substituteStream = (Db4objects.Db4o.Internal.LocalObjectContainer)Db4objects.Db4o.Db4oFactory
						.OpenFile(fileName);
					i_substituteTrans = i_substituteStream.NewTransaction();
					i_substituteStream.ConfigImpl().SetMessageRecipient(i_mainStream.ConfigImpl().MessageRecipient
						());
					WriteOK();
				}
				catch (System.Exception e)
				{
					CloseSubstituteStream();
					Write(Db4objects.Db4o.Internal.CS.Messages.Msg.ERROR);
				}
			}
		}

		private void SwitchToMainFile()
		{
			lock (i_mainStream.i_lock)
			{
				CloseSubstituteStream();
				WriteOK();
			}
		}

		private void UseTransaction(Db4objects.Db4o.Internal.CS.Messages.Msg message)
		{
			int threadID = ((Db4objects.Db4o.Internal.CS.Messages.MsgD)message).ReadInt();
			Db4objects.Db4o.Internal.CS.ServerMessageDispatcher transactionThread = i_server.
				FindThread(threadID);
			if (transactionThread != null)
			{
				Db4objects.Db4o.Internal.Transaction transToUse = transactionThread.GetTransaction
					();
				if (i_substituteTrans != null)
				{
					i_substituteTrans = transToUse;
				}
				else
				{
					i_mainTrans = transToUse;
				}
				i_rollbackOnClose = false;
			}
		}

		private void RespondInt(int response)
		{
			Write(Db4objects.Db4o.Internal.CS.Messages.Msg.ID_LIST.GetWriterForInt(GetTransaction
				(), response));
		}

		public void Write(Db4objects.Db4o.Internal.CS.Messages.Msg msg)
		{
			msg.Write(GetStream(), i_socket);
		}

		public Db4objects.Db4o.Foundation.Network.ISocket4 Socket()
		{
			return i_socket;
		}
	}
}
