/* Distributed Checksum Clearinghouse
 *
 * Copyright (c) 2004 by Rhyolite Software
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND RHYOLITE SOFTWARE DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL RHYOLITE SOFTWARE
 * BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Rhyolite Software DCC 1.2.66-1.9 $Revision$
 */

#include "dcc_defs.h"


/* because tv.tv_sec is not a time_t on all systems
 * and to be thread safe on WIN32 */
const struct tm *
dcc_localtime(time_t secs, struct tm *result)
{
#ifdef HAVE_LOCALTIME_R
	return localtime_r(&secs, result);
#else
	dcc_localtime_lock();
	result = localtime(&secs);
	dcc_localtime_unlock();
	return result;
#endif
}



int
dcc_get_secs(char *s, char **end, int min, int max, int def)
{
	static time_t epoch = 0;
	char *p;
	int secs;

	if (*s == '\0' || *s == ',') {
		secs = def;
		p = s;
	} else if (min > 0
		   && !CSTRCMP(s, "never")) {
		p = s+STRZ("never");
		if (!epoch)
			epoch = time(0);
		secs = epoch;
	} else if ((secs = strtoul(s, &p, 0)) >= DCC_MAX_SECS/60) {
		return -1;
	} else if (*p == '\0' || *p == ',') {
		;
	} else if (!CSTRCMP(p, "seconds")) {
		p += STRZ("seconds");
	} else if (!CSTRCMP(p, "s")) {
		p += STRZ("s");

	} else if (!CSTRCMP(p, "minutes")) {
		p += STRZ("minutes");
		secs *= 60;
	} else if (!CSTRCMP(p, "minute")) {
		p += STRZ("minute");
		secs *= 60;
	} else if (!CSTRCMP(p, "m")) {
		p += STRZ("m");
		secs *= 60;

	} else if (secs >= DCC_MAX_SECS/(60*60)) {
		return -1;
	} else if (!CSTRCMP(p, "hours")) {
		p += STRZ("hours");
		secs *= 60*60;
	} else if (!CSTRCMP(p, "hour")) {
		p += STRZ("hour");
		secs *= 60*60;
	} else if (!CSTRCMP(p, "h")) {
		p += STRZ("h");
		secs *= 60*60;

	} else if (secs >= DCC_MAX_SECS/(24*60*60)) {
		return -1;
	} else if (!CSTRCMP(p, "days")) {
		p += STRZ("days");
		secs *= 24*60*60;
	} else if (!CSTRCMP(p, "day")) {
		p += STRZ("day");
		secs *= 24*60*60;
	} else if (!CSTRCMP(p, "d")) {
		p += STRZ("d");
		secs *= 24*60*60;

	} else if (secs >= DCC_MAX_SECS/(7*24*60*60)) {
		return -1;
	} else if (!CSTRCMP(p, "weeks")) {
		p += STRZ("weeks");
		secs *= 7*24*60*60;
	} else if (!CSTRCMP(p, "week")) {
		p += STRZ("week");
		secs *= 7*24*60*60;
	} else if (!CSTRCMP(p, "w")) {
		p += STRZ("w");
		secs *= 7*24*60*60;

	} else {
		return -1;
	}

	if (secs > max)
		return -1;
	if (secs < min && secs != 0)
		return -1;

	if (*p != '\0') {
		if (*p != ',' || !end)
			return -1;
		++p;
	}
	if (end)
		*end = p;
	return secs;
}
