/* Distributed Checksum Clearinghouse
 *
 * database lister
 *
 * Copyright (c) 2005 by Rhyolite Software, LLC
 *
 * This agreement is not applicable to any entity which sells anti-spam
 * solutions to others or provides an anti-spam solution as part of a
 * security solution sold to other entities, or to a private network
 * which employs the DCC or uses data provided by operation of the DCC
 * but does not provide corresponding data to other users.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * Parties not eligible to receive a license under this agreement can
 * obtain a commercial license to use DCC and permission to use
 * U.S. Patent 6,330,590 by contacting Commtouch at http://www.commtouch.com/
 * or by email to nospam@commtouch.com.
 *
 * A commercial license would be for Distributed Checksum and Reputation
 * Clearinghouse software.  That software includes additional features.  This
 * free license for Distributed ChecksumClearinghouse Software does not in any
 * way grant permision to use Distributed Checksum and Reputation Clearinghouse
 * software
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND RHYOLITE SOFTWARE, LLC DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL RHYOLITE SOFTWARE, LLC
 * BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Rhyolite Software DCC 1.3.42-1.84 $Revision$
 */

#include "srvr_defs.h"
#include "dcc_xhdr.h"
#include "dcc_ck.h"

static DCC_EMSG dcc_emsg;

static int verbose;
#define VERBOSE_HASH 3
static u_char no_hash;
static u_char no_data;
static u_char matching;

static struct {
    DCC_CK_TYPES type;
    DCC_SUM	sum;
    u_char	type_only;
} search_cksums[16];
static int num_search_cksums;

static struct {
    DCC_TS	lo;
    DCC_TS	hi;
} search_ts[16];
static int num_search_ts;

DCC_SRVR_ID search_ids[16];
static int num_search_ids;

static DB_PTR page_offset;
static DB_PTR dbaddr;
static int max_pathlen;

static DB_HADDR hash_fsize;
static char dcc_db_nm[] = DB_DCC_NAME;
static char grey_db_nm[] = DB_GREY_NAME;
static	DCC_PATH hash_nm;
static char *def_argv[2];
static const char *homedir;

static const DB_VERSION_BUF version_buf = DB_VERSION_STR;

static const char hash_magic[sizeof(HASH_ENTRY)] = DB_HASH_MAGIC;


static int save_cksum(DCC_EMSG, DCC_WF *, DCC_CK_TYPES, DCC_SUM, DCC_TGTS);
static void list_cleaned(const DB_PARMS *);
static void list_flod(void);
static int fd_hash = -1;
static int fd_db = -1;
static struct stat hash_sb, db_sb;
static DB_PTR list_db(void);
static u_char open_db(void);
static void open_hash(void);
static void list_hash(void);


static void NRATTRIB
usage(void)
{
	dcc_logbad(EX_USAGE, "usage: [-vVHD] [-G on | off] [-h homedir]"
		   " [-C '[type] [h1 h2 h3 h4]']\n"
		   "   [-I server-Id] [-A dbptr] [-L pathlen] [-P pages]"
		   " [-T timestamp]\n"
		   "   [file file2 ...]");
}



int NRATTRIB
main(int argc, char **argv)
{
	int file_num;
	u_char print_version = 0;
	DCC_CK_TYPES type;
	char tbuf[80];
	const char *cp, *cp0;
	struct timeval tv1, tv2;
	int usecs;
	struct tm tm;
	char *p;
	int i;

	dcc_syslog_init(0, argv[0], 0);
	dup2(1, 2);			/* put error messages in context */

	while ((i = getopt(argc, argv, "vVHDG:h:C:I:A:L:P:T:")) != EOF) {
		switch (i) {
		case 'v':
			++verbose;
			break;

		case 'V':
			fprintf(stderr, DCC_VERSION"\n");
			print_version = 1;
			break;

		case 'H':
			no_hash = 1;
			break;

		case 'D':
			no_data = 1;
			break;

		case 'G':
			if (!strcasecmp(optarg, "on")) {
				grey_on = 1;
			} else if (!strcasecmp(optarg, "off")) {
				grey_on = 0;
			} else {
				usage();
			}
			break;

		case 'h':
			homedir = optarg;
			break;

		case 'C':
			if (num_search_cksums >= DIM(search_cksums)) {
				dcc_error_msg("too many -C checksums");
				break;
			}
			matching = 1;
			cp0 = optarg;
			cp = dcc_parse_word(0, tbuf, sizeof(tbuf),
					    optarg, "checksum type", 0, 0);
			if (!cp)
				exit(1);
			if (!strcasecmp(tbuf, "hex")) {
				/* ignore "hex" */
				cp0 = cp;
				cp = dcc_parse_word(0, tbuf, sizeof(tbuf),
						    cp, "checksum type",
						    0, 0);
				if (!cp)
					dcc_logbad(EX_USAGE,
						   "unrecognized checksum"
						   " \"-C %s\"", optarg);
			}
			if (*cp == '\0') {
				/* allow bare checksum type */
				type = dcc_str2type(tbuf);
				if (type == DCC_CK_INVALID)
					dcc_logbad(EX_USAGE,
						   "unrecognized checksum type"
						   " \"-C %s\"", optarg);
				search_cksums[num_search_cksums].type = type;
				memset(search_cksums[num_search_cksums].sum, 0,
				       sizeof(DCC_SUM));
				search_cksums[num_search_cksums].type_only = 1;
				++num_search_cksums;
				break;
			}
			/* allow missing checksum type */
			strtoul(tbuf, &p, 16);
			if (*p == '\0') {
				if (0 >= dcc_parse_hex_ck(dcc_emsg, 0,
							"-", DCC_CK_FLOD_PATH,
							cp0, 0, save_cksum))
					dcc_logbad(EX_USAGE, "%s", dcc_emsg);
			} else {
				type = dcc_str2type(tbuf);
				if (type == DCC_CK_FLOD_PATH)
					dcc_logbad(EX_USAGE,
						   "unrecognized checksum type"
						   " \"-C %s\"", optarg);
				if (0 >= dcc_parse_hex_ck(dcc_emsg, 0,
							tbuf, type,
							cp, 0, save_cksum))
					dcc_logbad(EX_USAGE, "%s", dcc_emsg);
			}
			break;

		case 'I':
			if (num_search_ids >= DIM(search_ids)) {
				dcc_error_msg("too many -I IDs");
				break;
			}
			search_ids[num_search_ids] = strtoul(optarg, &p, 0);
			if (search_ids[num_search_ids] > DCC_SRVR_ID_MAX
			    || *p != '\0')
				dcc_logbad(EX_USAGE,
					   "invalid server-ID \"-I %s\"",
					   optarg);
			++num_search_ids;
			matching = 1;
			break;

		case 'A':
			dbaddr = strtoul(optarg, &p, 16);
			if (*p != '\0')
				dcc_logbad(EX_USAGE,
					   "invalid database address \"%s\"",
					   optarg);
			matching = 1;
			break;

		case 'L':
			max_pathlen = strtoul(optarg, &p, 0);
			if (*p != '\0')
				dcc_logbad(EX_USAGE,
					   "invalid path length \"%s\"",
					   optarg);
			matching = 1;
			break;

		case 'P':
			page_offset = strtoul(optarg, &p, 0);
			if (*p != '\0')
				dcc_logbad(EX_USAGE,
					   "invalid number of pages \"%s\"",
					   optarg);
			matching = 1;
			break;

		case 'T':
			if (num_search_ts >= DIM(search_ts)) {
				dcc_error_msg("too many -T timestamps");
				break;
			}
			memset(&tm, 0, sizeof(tm));
			i = sscanf(optarg, "%d/%d/%d %d:%d:%d.%d%c",
				   &tm.tm_year, &tm.tm_mon, &tm.tm_mday,
				   &tm.tm_hour, &tm.tm_min, &tm.tm_sec,
				   &usecs, tbuf);
			if (i < 6 || i > 7
			    || tm.tm_mon <= 0)
				dcc_logbad(EX_USAGE,"bad timestamp \"%s\"",
					   optarg);
			--tm.tm_mon;
			tm.tm_year += 100;
			tv1.tv_sec = DCC_TIMEGM(&tm);
			if (tv1.tv_sec < 0)
				dcc_logbad(EX_USAGE, "invalid timestamp \"%s\"",
					   optarg);
			tv2.tv_sec = tv1.tv_sec;
			if (i == 7) {
				if (usecs >= DCC_USECS)
					dcc_logbad(EX_USAGE,
						   "invalid microseconds"
						   " in \"%s\"",
						   optarg);
				tv1.tv_usec = usecs;
				tv2.tv_usec = usecs;
			} else {
				tv1.tv_usec = 0;
				tv2.tv_usec = DCC_USECS-1;
			}
			dcc_timeval2ts(search_ts[num_search_ts].lo, &tv1, 0);
			dcc_timeval2ts(search_ts[num_search_ts].hi, &tv2, 0);
			++num_search_ts;
			matching = 1;
			break;

		default:
			usage();
		}
	}
	argc -= optind;
	argv += optind;
	def_argv[0] = grey_on ? grey_db_nm : dcc_db_nm;
	if (argc == 0) {
		if (print_version)
			exit(EX_OK);
		argv = def_argv;
		argc = 1;
	}

	dcc_clnt_unthread_init();

	if (!dcc_cdhome(dcc_emsg, homedir))
		dcc_error_msg("%s", dcc_emsg);

	flod_mmap_path_set();

	if (matching) {
		if (no_data && no_hash)
			dcc_logbad(EX_USAGE,
				   "patterns need data or hash table");
		if (!no_data && !no_hash)
			no_hash = 1;
	}

	if (dbaddr != 0 && page_offset != 0)
		dcc_logbad(EX_USAGE, "-P and -A are incompatible");

	for (file_num = 1; *argv != 0; ++argv, ++file_num) {
		if (fd_db >= 0)
			close(fd_db);
		if (fd_hash >= 0)
			close(fd_hash);

		BUFCPY(db_nm, *argv);
		snprintf(hash_nm, sizeof(hash_nm), "%s"DB_HASH_SUFFIX, db_nm);

		if (file_num != 1)
			fputc('\n', stdout);
		if (verbose || argc > 1)
			printf("  %s\n", db_nm);

		open_hash();
		if (!open_db())
			continue;

		db_csize = list_db();
		if (!db_csize)
			continue;

		if (fd_hash >= 0)
			list_hash();
	}

	exit(EX_OK);
}



static int
save_cksum(DCC_EMSG emsg UATTRIB, DCC_WF *wf UATTRIB,
	   DCC_CK_TYPES type, DCC_SUM sum, DCC_TGTS tgts UATTRIB)
{
	search_cksums[num_search_cksums].type = type;
	memcpy(search_cksums[num_search_cksums].sum, sum, sizeof(DCC_SUM));
	search_cksums[num_search_cksums].type_only = 0;
	++num_search_cksums;
	return 1;
}



#define RCD_PAT "%-27s %-8.8s %-10.10s %7s "L_H8PAT"\n"
#define RCD_PAT1(s) RCD_PAT, s,  "", "", ""

static DB_HDR hdr_buf;

static enum {NO_LB,			/* no label */
	WHITE_LB,			/* whitelist section labelled */
	DATE_LB				/* normal section labelled */
} last_lb = NO_LB;
static u_char printed_rcd;
static int rcds, white_rcds, sums, white_sums;


static u_char
open_db(void)
{
	int i;

	fd_db = open(db_nm, O_RDONLY, 0);
	if (fd_db < 0) {
		dcc_error_msg("open(%s): %s", db_nm, ERROR_STR());
		return 0;
	}

	i = read_db(dcc_emsg, &hdr_buf, sizeof(hdr_buf),
		    fd_db, 0, db_nm);
	if (i != sizeof(hdr_buf)) {
		if (i < 0)
			dcc_error_msg("%s", dcc_emsg);
		else
			dcc_error_msg("found only %d bytes of magic in %s",
				      i, db_nm);
		return 0;
	}

	if (memcmp(hdr_buf.p.version, version_buf,
		   sizeof(hdr_buf.p.version))) {
		dcc_error_msg("%s contains the wrong magic \"%.*s\"",
			      db_nm, ISZ(version_buf), hdr_buf.p.version);
	}
	if (0 > fstat(fd_db, &db_sb)) {
		dcc_error_msg("stat(%s): %s", db_nm, ERROR_STR());
		return 0;
	}
	if ((DB_PTR)db_sb.st_size < hdr_buf.p.db_csize) {
		dcc_error_msg("%s says it contains "L_DPAT
			      " bytes instead of "OFF_DPAT,
			      db_nm, hdr_buf.p.db_csize, db_sb.st_size);
	}

	if (db_sb.st_size == sizeof(hdr_buf)) {
		dcc_error_msg("%s contains no checksums",db_nm);
		return 0;
	}

	return 1;
}



static void
list_db_entry(DB_PTR rcd_link, const DB_RCD *rcd)
{
	const DB_RCD_CK *rcd_ck;
	DB_PTR rcd_prev;
	DCC_TGTS tgts;
	DCC_CK_TYPES type;
	char ts_buf[40], id_buf[30];
	char tgts_buf[20];
	char ck_buf[sizeof(DCC_SUM)*3+2];
	u_char rpt_match, kept;
	int i;

	/* usually skip padding */
	if (rcd->fgs_num_cks == 0) {
		if (verbose > 1) {
			printf(RCD_PAT1("    page padding"), rcd_link);
			printed_rcd = 1;
		}
		return;
	}

	rpt_match = 0;

	/* skip until the desired first address */
	if (dbaddr != 0) {
		if (rcd_link < dbaddr)
			return;
		rpt_match = 1;
	}

	/* if we have target server-IDs, display only their reports */
	if (num_search_ids > 0) {
		for (i = 0; i < num_search_ids; ++i) {
			if (search_ids[i] == DB_RCD_ID(rcd)) {
				rpt_match = 1;
				goto got_id;
			}
		}
		return;
got_id:;
	}

	/* if we have target checksums, display only reports containing them */
	if (num_search_cksums > 0) {
		for (i = 0; i < num_search_cksums; ++i) {
			for (rcd_ck = rcd->cks;
			     rcd_ck < &rcd->cks[DB_NUM_CKS(rcd)];
			     ++rcd_ck) {
				type = search_cksums[i].type;
				if ((DB_CK_TYPE(rcd_ck) == type
				     || type == DCC_CK_FLOD_PATH)
				    && (search_cksums[i].type_only
					|| !memcmp(search_cksums[i].sum,
						   rcd_ck->sum,
						   sizeof(DCC_SUM)))) {
					rpt_match = 1;
					goto got_ck;
				}
			}
		}
		return;
got_ck:;
	}

	if (num_search_ts > 0
	    && DB_RCD_ID(rcd) != DCC_ID_WHITE) {
		for (i = 0; i < num_search_ts; ++i) {
			if (!DCC_TS_OLDER_TS(rcd->ts,
					     search_ts[i].lo)
			    && !DCC_TS_NEWER_TS(rcd->ts,
						search_ts[i].hi)) {
				rpt_match = 1;
				goto got_ts;
			}
		}
		return;
got_ts:;
	}

	if (max_pathlen != 0
	    && DB_RCD_ID(rcd) != DCC_ID_WHITE) {
		DCC_FLOD_PATH_ID *id;
		DCC_SRVR_ID psrvr;
		int pathlen = 0;

		for (rcd_ck = rcd->cks;
		     rcd_ck < &rcd->cks[DB_NUM_CKS(rcd)]
		     && pathlen < max_pathlen;
		     ++rcd_ck) {
			if (DB_CK_TYPE(rcd_ck) != DCC_CK_FLOD_PATH)
				break;
			id = (DCC_FLOD_PATH_ID *)&rcd_ck->sum;
			for (i = 0; i < DCC_NUM_FLOD_PATH; ++i, ++id) {
				psrvr = ((id->hi<<8) | id->lo);
				if (psrvr == DCC_ID_INVALID)
					break;
				++pathlen;
			}
		}
		if (pathlen < max_pathlen)
			return;
		rpt_match = 1;
	}

	++rcds;
	if (DB_RCD_ID(rcd) == DCC_ID_WHITE) {
		++white_rcds;
		if (last_lb != WHITE_LB) {
			last_lb = WHITE_LB;
			strcpy(ts_buf, "\n"DCC_XHDR_ID_WHITE);
		} else {
			ts_buf[0] = '\0';
		}
	} else {
		if (last_lb != DATE_LB) {
			last_lb = DATE_LB;
			if (rpt_match || verbose > 0)
				putchar('\n');
		}
		if (rpt_match || verbose > 0)
			dcc_ts2str(ts_buf, sizeof(ts_buf), rcd->ts);
	}

	/* display separator between whitelist and ordinary entries
	 * along with the timestamp and the rest of the first line
	 * of a report */
	if (rpt_match
	    || verbose >= 2
	    || (verbose > 0 && DB_RCD_ID(rcd) != DCC_ID_WHITE)) {
		if (last_lb == DATE_LB) {
			tgts = DB_TGTS_RCD_RAW(rcd);
			printf(RCD_PAT, ts_buf,
			       (tgts == 0)
			       ? "deleted"
			       : dcc_tgts2str(tgts_buf, sizeof(tgts_buf),
					      tgts, grey_on),
			       dcc_srvr_id2str(id_buf, sizeof(id_buf),
					       rcd->srvr_id_auth),
			       DB_RCD_TRIMMED(rcd) ? "trimmed"
			       : DB_RCD_SUMRY(rcd) ? "summary"
			       : DB_RCD_DELAY(rcd) ? "delayed"
			       : "",
			       rcd_link);
		} else {
			printf(RCD_PAT1(ts_buf), rcd_link);
		}
		printed_rcd = 1;
	}

	/* display a report */
	for (rcd_ck = rcd->cks;
	     rcd_ck < &rcd->cks[DB_NUM_CKS(rcd)];
	     ++rcd_ck) {
		++sums;
		/* always count whitelist entries,
		 * but display only as requested */
		if (DB_RCD_ID(rcd) == DCC_ID_WHITE) {
			++white_sums;
			if (verbose < 2 && !rpt_match)
				continue;
		} else {
			if (verbose < 1 && !rpt_match)
				continue;
		}

		/* decode the special checksum that is a path */
		if (DB_CK_TYPE(rcd_ck)== DCC_CK_FLOD_PATH) {
			if (DB_RCD_ID(rcd) == DCC_ID_WHITE) {
				int lno;
				memcpy(&lno, rcd_ck->sum, sizeof(lno));
				printf("     line number %d file #%d\n",
				       lno, rcd_ck->sum[sizeof(lno)]);

			} else {
				DCC_SRVR_ID psrvr;
				DCC_FLOD_PATH_ID *path_id, *path_id_lim;
				const char *s;

				path_id=(DCC_FLOD_PATH_ID *)rcd_ck->sum;
				path_id_lim = path_id+DCC_NUM_FLOD_PATH;
				s = "     path: ";
				do {
					psrvr = ((path_id->hi<<8)
						 | path_id->lo);
					if (psrvr == DCC_ID_INVALID)
					    break;
					printf("%s%d", s, psrvr);
					s = "<-";
				} while (++path_id < path_id_lim);
				printf("%s\n", s);
			}
			continue;
		}

		kept = (!DB_TEST_NOKEEP(hdr_buf.p.nokeep_cks,
					DB_CK_TYPE(rcd_ck))
			|| DB_RCD_ID(rcd) == DCC_ID_WHITE);

		printf(" %c%-12.12s %-10.10s %-31s",
		       DB_CK_OBS(rcd_ck)
		       ? '*'
		       : DB_CK_DUP(rcd_ck)
		       ? '-'
		       : ' ',
		       DB_TYPE2STR(DB_CK_TYPE(rcd_ck)),
		       !kept
		       ? "" : dcc_tgts2str(tgts_buf, sizeof(tgts_buf),
					   DB_TGTS_CK(rcd_ck), grey_on),
		       dcc_ck2str(ck_buf, sizeof(ck_buf),
				  DB_CK_TYPE(rcd_ck), rcd_ck->sum));
		rcd_prev = DB_PTR_EX(rcd_ck->prev);
		if (rcd_prev == DB_PTR_NULL)
			printf(" %8s", "");
		else if (DB_PTR_IS_BAD_FULL(rcd_prev))
			printf(" bogus "L_H8PAT, rcd_prev);
		else
			printf(" "L_H8PAT, rcd_prev);
		if (db_hash_len != 0
		    && kept)
			printf(" %x", db_hash(DB_CK_TYPE(rcd_ck),
					      rcd_ck->sum));
		putchar('\n');
	}
}



static DB_PTR				/* size of contents or 0=failed */
list_db(void)
{
	DB_RCD rcd;
	int rcd_len;
	DB_PTR rcd_lim, rcd_link;

	if (verbose > 0) {
		list_cleaned(&hdr_buf.p);
		list_flod();
	}

	if (no_data)
		return hdr_buf.p.db_csize;

	last_lb = NO_LB;
	printed_rcd = 0;
	rcds = 0;
	white_rcds = 0;
	sums = 0;
	white_sums = 0;

	/* list the records in the database */
	if (dbaddr != 0) {
		if ((DB_PTR)db_sb.st_size <= dbaddr) {
			page_offset = 0;
		} else {
			page_offset = ((db_sb.st_size - dbaddr
					+ hdr_buf.p.page_size -1)
				       / hdr_buf.p.page_size);
		}
	}
	if (page_offset == 0) {
		rcd_link = DB_PTR_BASE;
	} else {
		rcd_link = db_sb.st_size / hdr_buf.p.page_size;
		if (rcd_link < page_offset)
			rcd_link = 0;
		else
			rcd_link -= page_offset;
		rcd_link *= hdr_buf.p.page_size;
		if (rcd_link < DB_PTR_BASE)
			rcd_link = DB_PTR_BASE;
	}
	rcd_lim = ((verbose > 2)
		   ? (DB_PTR)db_sb.st_size : (DB_PTR)hdr_buf.p.db_csize);
	read_rcd_invalidate();
	while (rcd_link < rcd_lim) {
		rcd_len = read_rcd(dcc_emsg, &rcd, fd_db, rcd_link, db_nm);
		if (rcd_len <= 0) {
			if (rcd_len == 0)
				break;
			/* ignore fragmentary reports at the end */
			if (rcd_link > hdr_buf.p.db_csize - DB_RCD_HDR_LEN) {
				printf(RCD_PAT1("    page padding"), rcd_link);
				printed_rcd = 1;
				break;
			}
			dcc_error_msg("%s", dcc_emsg);
			return 0;
		}


		list_db_entry(rcd_link, &rcd);
		rcd_link += rcd_len;
	}

	if (verbose || matching) {
		/* print address after the last record,
		 * but only if we printed a record */
		if (printed_rcd)
			printf(RCD_PAT1(""), rcd_link);
		putchar('\n');
	}
	if (!matching) {
		printf("%7d records containing %d checksums\n",
		       rcds, sums);
		if (!grey_on)
			printf("%7d non-whitelist records containing"
			       " %d checksums\n",
			       rcds-white_rcds, sums-white_sums);
	}
	return hdr_buf.p.db_csize;
}



const char *
secs2str(char *buf, u_int buf_len, u_int32_t secs)
{
	const char *dim;

	if (!buf_len)
		return 0;
	if (secs == 0) {
		STRLCPY(buf, "never", buf_len);
	} else {
		if (secs % (24*60*60) == 0) {
			dim = " day";
			secs /= (24*60*60);
		} else if (secs % (60*60) == 0) {
			dim = " hour";
			secs /= (60*60);
		} else {
			dim = "";
		}
		snprintf(buf, buf_len, "%d%s%s",
			 secs, dim, (secs > 1 && dim) ? "s" : "");
	}
	return buf;
}



/* display the expiration information in the database header */
static void
list_cleaned(const DB_PARMS *parms)
{
#define CLEANED_PAT	"     %12s %c %17.17s  %8s  %8s  %5s"
#define CLEANED_PAT_VB3	"     %12s %c %17.17s  %8s  %8s  %5s %8s"
	DCC_CK_TYPES type;
	char allsecs_buf[20];
	char spamsecs_buf[20];
	char clean_thold[20];
	char flod_thold_buf[20];

	printf("     %s%s%spage size %#-8x  s/n %s\n",
	       (parms->flags & DB_PARM_FG_GREY)
	       ? "Greylist  " : "",
	       !(parms->flags & DB_PARM_FG_SELF_CLEAN) ? ""
	       : !(parms->flags & DB_PARM_FG_SELF_CLEAN2) ? "self-cleaned  "
	       : "twice self-cleaned  ",
	       !(parms->flags & DB_PARM_FG_CLEARED) ? ""
	       : "cleared  ",
	       parms->page_size, dcc_ts2str_err(parms->sn));

	if (verbose > 3)
		printf("     db_csize="L_HPAT"  version \"%.*s\"\n",
		       parms->db_csize, ISZ(parms->version), parms->version);

	if (parms->flags & DB_PARM_FG_GREY)
		printf(CLEANED_PAT,
		       "", ' ', "expired    ", "window", "pass", "threshold");
	else
		printf(CLEANED_PAT,
		       "", ' ', "expired    ", "", "bulk  ", "threshold");
	for (type = DCC_CK_TYPE_FIRST; type <= DCC_CK_TYPE_LAST; ++type) {
		if ((type == DCC_CK_SRVR_ID
		     || DB_TEST_NOKEEP(parms->nokeep_cks, type))
		    && verbose < 3)
			continue;
		secs2str(allsecs_buf, sizeof(allsecs_buf),
			 parms->ex_secs[type].all);
		secs2str(spamsecs_buf, sizeof(spamsecs_buf),
			 parms->ex_secs[type].spam);
		if ((parms->flags & DB_PARM_FG_GREY)
		    && verbose < 3) {
			clean_thold[0] = '\0';
		} else {
			dcc_tgts2str(clean_thold, sizeof(clean_thold),
				     parms->ex_secs[type].clean_thold,
				     grey_on);
		}
		dcc_tgts2str(flod_thold_buf, sizeof(flod_thold_buf),
			     parms->flod_tholds[type], grey_on);
		printf(verbose >= 3 ? "\n"CLEANED_PAT_VB3 : "\n"CLEANED_PAT,
		       DB_TYPE2STR(type),
		       DB_TEST_NOKEEP(parms->nokeep_cks, type) ? '*' : ' ',
		       dcc_ts2str_err(parms->ex_ts[type].all),
		       allsecs_buf, spamsecs_buf, clean_thold, flod_thold_buf);
	}
#undef CLEANED_PAT
}



static void
list_flod(void)
{
	FLOD_MMAP *mp;
	DCC_PATH path;
	char hostname[40], fg_buf[60];
	u_char first;

	/* display the flood map only for default database */
	if (strcmp(fnm2path_err(path, db_nm), DB_NM2PATH_ERR(def_argv[0]))) {
		putchar('\n');
	} else if (!flod_mmap(dcc_emsg, 0, 0, 0, 1)) {
		dcc_error_msg("\n\n%s", dcc_emsg);
	} else if (strcmp(flod_mmaps->magic, FLOD_MMAP_MAGIC)) {
		dcc_error_msg("\n\n%s contains the wrong magic \"%.*s\"",
			      flod_mmap_path,
			      ISZ(flod_mmaps->magic), flod_mmaps->magic);
		if (!flod_unmap(dcc_emsg, 0))
			dcc_error_msg("%s", dcc_emsg);
	} else {
		first = 1;
		fputs("\n\n  ", stdout);
		fputs(flod_mmap_path, stdout);
		printf("  s/n %s   delay position "L_HPAT"\n",
		       dcc_ts2str_err(flod_mmaps->sn),
		       flod_mmaps->delay_pos);
		for (mp = flod_mmaps->mmaps;
		     mp <= LAST(flod_mmaps->mmaps);
		     ++mp) {
			if (mp->rem_hostname[0] == '\0')
				continue;
			if (first) {
				first = 0;
				printf("%32s %5s %9s %s\n",
				       "peer", "", "ID", "position");
			}
			printf("%38s %9d "L_H8PAT"%s\n",
			       dcc_host_portname(hostname, sizeof(hostname),
						 mp->rem_hostname,
						 mp->rem_portname),
			       mp->rem_id,
			       mp->confirm_pos,
			       flod_mmap_fg(fg_buf, sizeof(fg_buf), "  ", mp));
		}
		if (!flod_unmap(dcc_emsg, 0))
			dcc_error_msg("%s", dcc_emsg);
	}
}



static void
open_hash(void)
{
	db_hash_len = 0;
	fd_hash = open(hash_nm, O_RDONLY, 0);
	if (0 > fd_hash) {
		dcc_error_msg("open(%s): %s", hash_nm, ERROR_STR());
		return;
	}
	if (0 > fstat(fd_hash, &hash_sb)) {
		dcc_error_msg("stat(%s): %s", hash_nm, ERROR_STR());
		close(fd_hash);
		fd_hash = -1;
		return;
	}
	hash_fsize = hash_sb.st_size;
	db_hash_len = hash_fsize/sizeof(HASH_ENTRY);
	if ((hash_fsize % sizeof(HASH_ENTRY)) != 0) {
		dcc_error_msg("%s has size %u, not a multiple of %d",
			      hash_nm, hash_fsize, ISZ(HASH_ENTRY));
		db_hash_len = 0;
		close(fd_hash);
		fd_hash = -1;
		return;
	}
	if (db_hash_len < MIN_HASH_ENTRIES) {
		dcc_error_msg("%s has too few records, %d bytes",
			      hash_nm, hash_fsize);
		db_hash_len = 0;
		close(fd_hash);
		fd_hash = -1;
		return;
	}
}



#define HASH_MAP_LEN	(1024*1024)
#define HASH_MAP_NUM	16
typedef struct hash_map {
    struct hash_map *fwd, *bak;
    HASH_ENTRY	*buf;
    DB_HADDR	base;
    DB_HADDR	lim;
    off_t	offset;
    int		size;
} HASH_MAP;
static HASH_MAP hash_maps[HASH_MAP_NUM];
static HASH_MAP *hash_map_newest;


static u_char
hash_munmap(HASH_MAP *mp)
{
	if (!mp->buf)
		return 1;

	if (0 > munmap((void *)mp->buf, mp->size)) {
		dcc_error_msg("munmap(%s,%d): %s",
			      hash_nm, mp->size, ERROR_STR());
		return 0;
	}
	mp->buf = 0;
	return 1;
}



static u_char
hash_map_clear(void)
{
	HASH_MAP *mp;
	int i;

	mp = hash_maps;
	for (i = 0; i < DIM(hash_maps); ++i, ++mp) {

		if (i == DIM(hash_maps)-1)
			mp->fwd = hash_maps;
		else
			mp->fwd = mp+1;
		if (i == 0)
			mp->bak = LAST(hash_maps);
		else
			mp->bak = mp-1;
	}
	hash_map_newest = hash_maps;

	for (mp = hash_maps; mp <= LAST(hash_maps); ++mp) {
		if (!hash_munmap(mp))
			return 0;
	}

	return 1;
}



static void
hash_map_ref(HASH_MAP *mp)
{
	if (hash_map_newest != mp) {
		mp->fwd->bak = mp->bak;
		mp->bak->fwd = mp->fwd;
		mp->fwd = hash_map_newest;
		mp->bak = hash_map_newest->bak;
		mp->fwd->bak = mp;
		mp->bak->fwd = mp;
		hash_map_newest = mp;
	}
}



static HASH_ENTRY *
haddr_mmap(DB_HADDR haddr)
{
	HASH_MAP *mp;
	void *p;
	int i;

	for (i = 0, mp = hash_map_newest;
	     i < DIM(hash_maps);
	     ++i, mp = mp->fwd) {
		if (!mp->buf)
			continue;
		if (haddr >= mp->base
		    && haddr < mp->lim) {
			hash_map_ref(mp);
			return mp->buf + (haddr - mp->base);
		}
	}

	mp = hash_map_newest->bak;
	hash_munmap(mp);

	mp->base = haddr -  haddr%HASH_MAP_LEN;
	mp->offset = mp->base*sizeof(HASH_ENTRY);
	mp->size = hash_fsize - mp->offset;
	if (mp->size > HASH_MAP_LEN*ISZ(HASH_ENTRY))
		mp->size = HASH_MAP_LEN*ISZ(HASH_ENTRY);
	mp->lim = mp->base + mp->size/sizeof(HASH_ENTRY);
	p = mmap(0, mp->size, PROT_READ, MAP_SHARED, fd_hash, mp->offset);
	if (p != MAP_FAILED) {
		mp->buf = p;
		hash_map_ref(mp);
		return mp->buf + (haddr - mp->base);
	}
	dcc_error_msg("mmap(%s,%d,%d): %s",
		      hash_nm, (int)mp->size, (int)mp->offset,
		      ERROR_STR());
	return 0;
}



static void
list_hash(void)
{
	HASH_ENTRY *entry = 0;
	int collisions, chains, chain_lens, max_chain_len, chain_len;
	DB_HADDR fwd, bak, rcd_num;
	DB_HADDR db_hash_used_stored, db_hash_len_stored;
	DB_PTR rcd_link, db_csize_stored;
	DCC_CK_TYPES type;
	DB_RCD rcd;
	int rcd_len;
	int i;

	if (verbose >= VERBOSE_HASH)
		printf("\n %s\n", hash_nm);

	if (!hash_map_clear())
		return;

	if (no_hash)
		return;

	db_hash_used_stored = 0;
	db_hash_used = DB_HADDR_MIN;
	collisions = 0;
	chains = 0;
	chain_lens = 0;
	max_chain_len = 1;
	for (rcd_num = 0; rcd_num < db_hash_len; ++rcd_num) {
		entry = haddr_mmap(rcd_num);
		if (!entry)
			break;

		/* deal with the special, first entries */
		if (rcd_num == DB_HADDR_MAGIC) {
			if (memcmp(entry, hash_magic, sizeof(entry))) {
				dcc_error_msg("%s contains the wrong magic",
					      hash_nm);
			}
			if (verbose > VERBOSE_HASH) {
				printf("     magic: \"%.*s\"\n",
				       ISZ(entry), (char*)entry);
			}
			continue;
		}

		fwd = DB_HADDR_EX(entry->fwd);
		bak = DB_HADDR_EX(entry->bak);

		if (rcd_num == DB_HADDR_FREE) {
			if ((DB_HADDR_INVALID(fwd) && fwd != DB_HADDR_FREE)
			    || (DB_HADDR_INVALID(bak) && bak != DB_HADDR_FREE)
			    || DB_HPTR_EX(entry->rcd) != DB_PTR_NULL
			    || !HE_IS_FREE(entry)) {
				dcc_error_msg("%s free list broken", hash_nm);
			}
			if (verbose == VERBOSE_HASH) {
				printf("    first free: %x\n",
				       fwd);
			} else if (verbose > VERBOSE_HASH) {
				printf("      free: %6x %6x\n",
				       fwd, bak);
			}
			continue;
		}

		if (rcd_num == DB_HADDR_SIZES) {
			if (verbose > VERBOSE_HASH) {
				printf("     sizes: %6x %6x\n",
				       fwd, bak);
			}
			db_hash_len_stored = DB_HADDR_EX(entry->HASH_STORE_LEN);
			if (db_hash_len != db_hash_len_stored
			    && (db_hash_len_stored != 0
				|| verbose > VERBOSE_HASH)) {
				dcc_error_msg("%s has %d entries but claims %d",
					      hash_nm,
					      HASH_LEN_EXT(db_hash_len),
					      HASH_LEN_EXT(db_hash_len_stored));
			}
			db_hash_used_stored = DB_HADDR_EX(entry
							->HASH_STORE_USED);
			if (db_hash_used_stored > db_hash_len) {
				dcc_error_msg("%s contains only %d"
					      " entries but %d used",
					      hash_nm,
					HASH_LEN_EXT(db_hash_len),
					HASH_LEN_EXT((db_hash_used_stored)));
			}
			if (db_hash_used_stored == db_hash_len) {
				dcc_error_msg("%s overflows with %d entries",
					      hash_nm,
					      HASH_LEN_EXT(db_hash_len));
			}
			db_csize_stored=DB_HPTR_EX(entry->HASH_STORE_DB_CSIZE);
			if (db_csize_stored != db_csize) {
				dcc_error_msg("%s claims %s contains "L_DPAT
					      " bytes instead of "L_DPAT,
					      hash_nm, db_nm,
					      db_csize_stored, db_csize);
			}
			continue;
		}

		/* deal with a free entry */
		if (HE_IS_FREE(entry)) {
			if (verbose >= VERBOSE_HASH)
				printf("    %6x: %6x %6x\n",
				       rcd_num, fwd, bak);
			continue;
		}

		/* deal with a used entry */
		rcd_link = DB_HPTR_EX(entry->rcd);
		++db_hash_used;
		if (DB_PTR_IS_BAD_FULL(rcd_link))
			dcc_error_msg("bad hash table data link at"
				      " %x to "L_HPAT,
				      rcd_num, rcd_link);
		if (DB_HADDR_INVALID(fwd) && fwd != DB_HADDR_NULL)
			dcc_error_msg("bad hash forward link at %x to %x",
				      rcd_num, fwd);
		if (DB_HADDR_INVALID(bak) && bak != DB_HADDR_NULL)
			dcc_error_msg("bad hash back link at %x to %x",
				      rcd_num, bak);
		if (verbose >= VERBOSE_HASH)
			printf("    %6x: %6x %6x "L_H8PAT" %s\n",
			       rcd_num, fwd, bak, rcd_link,
			       DB_TYPE2STR(HE_TYPE(entry)));

		if (bak != DB_HADDR_NULL) {
			++collisions;
		} else {
			++chains;
			bak = rcd_num;
			chain_len = 1;
			while (!DB_HADDR_INVALID(fwd)) {
				if (++chain_len > 100) {
					dcc_error_msg("possible hash chain loop"
						      " starting at %x"
						      " continuing through %x",
						      rcd_num, fwd);
					break;
				}
				entry = haddr_mmap(fwd);
				if (!entry)
					break;
				if (HE_IS_FREE(entry)
				    || DB_HADDR_EX(entry->bak) != bak) {
					dcc_error_msg("broken hash chain"
						      " starting at %x at %x",
						      rcd_num, fwd);
					break;
				}
				bak = fwd;
				fwd = DB_HADDR_EX(entry->fwd);
			}
			chain_lens += chain_len;
			if (max_chain_len < chain_len)
				max_chain_len = chain_len;
		}

		if (matching) {
			if (num_search_cksums > 0) {
				for (i = 0; i < num_search_cksums; ++i) {
					type = search_cksums[i].type;
					if (type == HE_TYPE(entry)
					    || type == DCC_CK_FLOD_PATH)
					    break;
				}
				if (i >= num_search_cksums)
					continue;
			}
			rcd_len = read_rcd(dcc_emsg, &rcd,
					   fd_db, rcd_link, db_nm);
			if (rcd_len <= 0) {
				if (rcd_len == 0)
					dcc_error_msg("bogus hash table data"
						      " link at %x to "L_HPAT,
						      rcd_num, rcd_link);
				else
					dcc_error_msg("%s", dcc_emsg);
				break;
			}
			list_db_entry(rcd_link, &rcd);
		}

	}

	hash_map_clear();

	if (db_hash_used_stored != db_hash_used) {
		if (db_hash_used_stored == 0)
			dcc_error_msg("%s was not cleanly closed but has"
				      " %d entries",
				      hash_nm,
				      HASH_LEN_EXT(db_hash_used));
		else
			dcc_error_msg("%s claims to have filled"
				      " %d entries but has %d",
				      hash_nm,
				      HASH_LEN_EXT(db_hash_used_stored),
				      HASH_LEN_EXT(db_hash_used));
	}

	if (verbose >= VERBOSE_HASH)
		putchar('\n');
	printf("%7d hash entries total  %d or %d%% used  %d free"
	       "  %.2f%% collisions\n",
	       HASH_LEN_EXT(db_hash_len),
	       HASH_LEN_EXT(db_hash_used),
	       (HASH_LEN_EXT(db_hash_used)*100) / HASH_LEN_EXT(db_hash_len),
	       HASH_LEN_EXT(db_hash_len) - HASH_LEN_EXT(db_hash_used),
	       collisions*100.0/HASH_LEN_EXT(db_hash_len));
	if (chains != 0)
		printf("%7d hash chains  max length %d  average length %.1f\n",
		       chains, max_chain_len, chain_lens*1.0/chains);
}
