/* Distributed Checksum Clearinghouse
 *
 * Copyright (c) 2005 by Rhyolite Software, LLC
 *
 * This agreement is not applicable to any entity which sells anti-spam
 * solutions to others or provides an anti-spam solution as part of a
 * security solution sold to other entities, or to a private network
 * which employs the DCC or uses data provided by operation of the DCC
 * but does not provide corresponding data to other users.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * Parties not eligible to receive a license under this agreement can
 * obtain a commercial license to use DCC and permission to use
 * U.S. Patent 6,330,590 by contacting Commtouch at http://www.commtouch.com/
 * or by email to nospam@commtouch.com.
 *
 * A commercial license would be for Distributed Checksum and Reputation
 * Clearinghouse software.  That software includes additional features.  This
 * free license for Distributed ChecksumClearinghouse Software does not in any
 * way grant permision to use Distributed Checksum and Reputation Clearinghouse
 * software
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND RHYOLITE SOFTWARE, LLC DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL RHYOLITE SOFTWARE, LLC
 * BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Rhyolite Software DCC 1.3.42-1.46 $Revision$
 */

#include "dcc_clnt.h"
#include "dcc_xhdr.h"


static void
print_srvr(const DCC_SRVR_CLASS *class,
	   const DCC_SRVR_ADDR *ap,
	   u_char print_name,
	   u_char have_rtt_adj)
{
	char srvr_nm[MAXHOSTNAMELEN];
	char a1[DCC_SU2STR_SIZE+1+5];
	DCC_SOCKU su;
	const char *addr;
	int addr_len, name_len, i, j;

	printf("# %1s", class->act_inx == ap - class->addrs ? "*" : "");
	if (print_name) {
		dcc_ip2su(&su, &ap->ip);
		dcc_su2name(srvr_nm, sizeof(srvr_nm), &su);
	} else {
		srvr_nm[0] = '\0';
	}
	addr = dcc_ap2str_opt(a1, sizeof(a1), class, ap - class->addrs, '-');
	addr_len = strlen(addr);
	name_len = strlen(srvr_nm);
	i = 22 - (name_len-25);
	if (i < 1)
		i = 1;
	else if (i > 22)
		i = 22;
	j = 25 - (addr_len-22);
	if (j < 1)
		j = 1;
	else if (j > 25)
		j = 25;
	printf("%-*s %-*s", i, addr, j, srvr_nm);
	if (ap->srvr_id != DCC_ID_INVALID) {
		i = 16 - ((addr_len+name_len) - (22+25));
		if (i < 1)
			i = 1;
		else if (i > 16)
			i = 16;
		printf(" %*s ID %d", i, ap->brand, ap->srvr_id);
		if (ap->srvr_pkt_vers != DCC_PKT_VERSION)
			printf("\n#     protocol version %d",
			       ap->srvr_pkt_vers);
	}
	putchar('\n');

	if (ap->rtt >= DCC_RTT_BAD) {
		fputs("#      not answering\n", stdout);
		return;
	}
	if (ap->total_xmits == 0) {
		printf("#     %22s", "");
	} else {
		printf("#     %3.0f%% of %2d requests ok",
		       (ap->total_resps*100.0)/ap->total_xmits,
		       ap->total_xmits);
	}
	if (have_rtt_adj) {
		if (ap->srvr_pkt_vers < DCC_PKT_VERSION
		    && ap->srvr_id != DCC_ID_INVALID) {
			i = printf("%8.2f%+d+%d ms RTT",
				   ap->rtt/1000.0,
				   class->nms[ap->nm_inx].rtt_adj/1000,
				   DCC_RTT_VERS_ADJ/1000);
		} else {
			i = printf("%8.2f%+d ms RTT",
				   ap->rtt/1000.0,
				   class->nms[ap->nm_inx].rtt_adj/1000);
		}
	} else {
		i = printf("%8.2f ms RTT",
			   ap->rtt/1000.0);
	}
	i = (i >= 22) ? 1 : (22-i);
	printf("  %*s  %4d ms queue wait\n",
	       i, "", ap->srvr_wait/1000);
}



/* dump the /var/dcc/map file */
void
dcc_print_info(const char *map_nm,	/* or null for temporary file */
	       const DCC_CLNT_INFO *info,
	       u_char grey,
	       u_char names,
	       u_char show_passwd)
{
#define dcc_clnt_info ??? error using dcc_clnt_info
	DCC_PATH path;
	char date_buf[40];
	int port, nm_inx;
	struct tm tm;
	time_t now;
	const DCC_SRVR_CLASS *class;
	const DCC_SRVR_ADDR *ap,*ap_prev, *ap_next;
	u_char printed_addr[DCC_MAX_SRVR_ADDRS];
	char sustr[DCC_SU2STR_SIZE];
	u_char have_rtt_adj;
	u_char need_blank_line;
	int i;

	now = time(0);
	class = grey ? &info->grey : &info->dcc;
	if (map_nm) {
		strftime(date_buf, sizeof(date_buf), "%x %X %Z",
			 dcc_localtime(now, &tm));
		printf("# %-s  %s%s\n",
		       date_buf, grey ? "GreyList " : "",
		       fnm2path_err(path, map_nm));
		fputs("# ", stdout);
		if (class->resolve > now || dcc_clnt_debug) {
			strftime(date_buf, sizeof(date_buf), "%X",
				 dcc_localtime(class->resolve, &tm));
			printf("Re-resolve names after %s  ", date_buf);
		}
		if (class->measure > now || dcc_clnt_debug) {
			strftime(date_buf, sizeof(date_buf), "%X",
				 dcc_localtime(class->measure, &tm));
			printf("Check RTTs after %s", date_buf);
		}
		putchar('\n');

		fputs("# ", stdout);
		i = 0;
		for (ap = class->addrs; ap <= LAST(class->addrs); ++ap) {
			if (ap->rtt != DCC_RTT_BAD
			    && ap->ip.family != AF_UNSPEC)
				++i;
		}
		if (i > 1 || dcc_clnt_debug)
			printf("%6.2f ms threshold, %4.2f ms average    ",
			       class->thold_rtt/1000.0,
			       class->avg_thold_rtt/1000.0);
		printf("%d total, %d working servers\n",
		       class->num_addrs, i);
		if (class->fail_exp != 0) {
			int fail_time = class->fail_time - now;
			if (fail_time > 0
			    && fail_time <= DCC_MAX_FAIL_SECS) {
				printf("# skipping asking %s server"
				       " %d seconds more\n",
				       grey ? "Greylist" : "DCC", fail_time);
			}
		}

		i = now/DCCPROC_COST - info->dccproc_last/DCCPROC_COST;
		if (i > DCCPROC_MAX_CREDITS*2)
			i = DCCPROC_MAX_CREDITS*2;
		else if (i < 0)
			i = 0;
		i += info->dccproc_c;
		if (i > DCCPROC_MAX_CREDITS)
			i = DCCPROC_MAX_CREDITS;
		else if (i < -DCCPROC_MAX_CREDITS)
			i = -DCCPROC_MAX_CREDITS;
		if (!grey && (i < DCCPROC_MAX_CREDITS
			      || !DCC_IS_TIME(now, info->dccproc_dccifd_try,
					      DCCPROC_TRY_DCCIFD))) {
			strftime(date_buf, sizeof(date_buf), "%X",
				 dcc_localtime(info->dccproc_last, &tm));
			printf("# %d dccproc balance since %s",
			       i, date_buf);
			if (!DCC_IS_TIME(now, info->dccproc_dccifd_try,
					 DCCPROC_TRY_DCCIFD)) {
				strftime(date_buf, sizeof(date_buf), "%X",
					 dcc_localtime(info->dccproc_dccifd_try,
						       &tm));
				printf("; do not try to start dccifd until %s",
				       date_buf);
			}
			putchar('\n');
		}

		if (!grey) {
			fputs((info->flags & DCC_INFO_FG_IPV6)
			      ? DCC_INFO_USE_IPV6 : DCC_INFO_USE_IPV4, stdout);
			if (info->flags & DCC_INFO_FG_SOCKS)
				fputs("   "DCC_INFO_USE_SOCKS, stdout);
			if (info->src.family != AF_UNSPEC) {
				printf("   src=%s",
				       dcc_ip2str(sustr, sizeof(sustr),
						  &info->src));
			}
			putchar('\n');
		}
	}

	have_rtt_adj = 0;
	for (nm_inx = 0; nm_inx < DIM(class->nms); ++nm_inx) {
		if (class->nms[nm_inx].hostname[0] == '\0')
			continue;
		if (class->nms[nm_inx].rtt_adj != 0) {
			have_rtt_adj = 1;
			break;
		}
	}
	if (!have_rtt_adj) {
		for (ap = class->addrs; ap<=LAST(class->addrs); ++ap) {
			if (ap->srvr_pkt_vers < DCC_PKT_VERSION
			    && ap->srvr_id != DCC_ID_INVALID) {
				have_rtt_adj = 1;
				break;
			}
		}
	}

	memset(printed_addr, 0, sizeof(printed_addr));

	/* convert each non-null hostname */
	need_blank_line = 1;
	for (nm_inx = 0; nm_inx < DIM(class->nms); ++nm_inx) {
		if (class->nms[nm_inx].hostname[0] == '\0')
			continue;

		/* First display the main line for a host */
		if (class->nms[nm_inx].defined == 0)
			need_blank_line = 1;
		if (!need_blank_line && nm_inx != 0) {
			for (ap = class->addrs; ap<=LAST(class->addrs); ++ap) {
				if (ap->nm_inx == nm_inx) {
					need_blank_line = 1;
					break;
				}
			}
		}
		if (need_blank_line) {
			need_blank_line = 0;
			putchar('\n');
		}
		i = printf("%s", class->nms[nm_inx].hostname);
		i = (i >= 26) ? 1 : (26-i);
		port = class->nms[nm_inx].port;
		if (port == DCC_GREY2PORT(grey))
			printf(",%-*s", i, "-   ");
		else
			printf(",%-*d", i, ntohs(port));

		if (grey)
			fputs(" Greylist", stdout);

		if (have_rtt_adj) {
			i = printf(" RTT%+d ms",
				   class->nms[nm_inx].rtt_adj/1000);
			i = (i >= 12) ? 1 : (12-i);
			printf("%*s", i, "");
		}

		/* Suppress the password if it does not exist or is secret */
		if (class->nms[nm_inx].clnt_id == DCC_ID_ANON) {
			fputs(" "DCC_XHDR_ID_ANON"\n", stdout);
		} else {
			printf(" %5d "DCC_PASSWD_PAT"\n",
			       class->nms[nm_inx].clnt_id,
			       show_passwd ? class->nms[nm_inx].passwd : "");
		}

		if (class->nms[nm_inx].defined == 0) {
			need_blank_line = 1;
			fputs("#   UNDEFINED\n", stdout);
		}

		/* display operating information for each IP address
		 * kludge sort the IP addresses */
		for (ap_prev = 0, i = 0;
		     i < DCC_MAX_SRVR_ADDRS;
		     ap_prev = ap_next, ++i) {
			ap_next = 0;
			for (ap = class->addrs; ap<=LAST(class->addrs); ++ap) {
				if (ap->nm_inx != nm_inx
				    || ap->ip.family == AF_UNSPEC)
					continue;
				/* find smallest IP address not yet printed */
				if (printed_addr[ap - class->addrs])
					continue;
				if (ap_next
				    && ap->ip.family >= ap_next->ip.family
				    && 0 <= memcmp(&ap->ip.u, &ap_next->ip.u,
						   sizeof(ap->ip.u))
				    && ap->ip.port >= ap_next->ip.port)
					continue;
				if (!ap_prev
				    || ap->ip.family >= ap_prev->ip.family
				    || 0 <= memcmp(&ap->ip.u, &ap_prev->ip.u,
						   sizeof(ap->ip.u))
				    || ap->ip.port >= ap_prev->ip.port)
					ap_next = ap;
			}
			if (!ap_next)
				break;
			print_srvr(class, ap_next, names, have_rtt_adj);
			printed_addr[ap_next - class->addrs] = 1;
			need_blank_line = 1;

		}
	}

	for (ap = class->addrs, i = 0; ap <= LAST(class->addrs); ++ap, ++i) {
		if (ap->ip.family == 0)
			continue;
		if (printed_addr[i])
			continue;
		printf("\n# stray address entry #%d, nm_inx %d:\n",
		       i, ap->nm_inx);
		print_srvr(class, ap, names, have_rtt_adj);
	}

#undef dcc_clnt_info
}
