/* DChub - a Direct Connect hub clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * minibot.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: minibot.c,v 2.1 2003/03/14 13:59:28 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <glib.h>
#include <pthread.h>


#include "dc_com.h"			/* we use the same function as the hub one to communicate */
#include "extprog_toolkit.h"
#include "network.h"

#ifndef MSG_NOSIGNAL
#define HAVE_NO_MSG_NOSIGNAL 1
#define MSG_NOSIGNAL 0
#endif

int hub_com;		/* socket fd to use to communicate with hub */
G_LOCK_DEFINE(hub_com);		/* to avoid conflict between the main function and threads */

char *my_name;
gchar *my_hostip=NULL;		/* it is the address to use to perform search or download */

/**************************************************/
/* this is a list of search to periodically start */
/**************************************************/
typedef struct
{
	time_t last_global_scan;
	gchar *search_result_return_address;		/* it is "my_hostip:my_port" */
	GString *search_query;							/* it is the constant part of the search (file type, search type, search pattern) */
	unsigned int on_start:1;						/* (not yet supported) ==1, when this entry is added, a search is immediatly started */
	unsigned int on_incoming:1;					/* ==1, when a new user enters, a "one user" search is started */
	unsigned int global_periodic:1;				/* (not yet supported) ==1, the scan is periodically started */
} AUTOSEARCH;

GArray *search_to_start=NULL;
G_LOCK_DEFINE(search_to_start);

static void add_pattern_to_search(unsigned short receive_port, char *pattern, unsigned int on_start, unsigned int on_incoming, unsigned int global_periodic)
{
	AUTOSEARCH nw;
	char myaddr[8192];

	sprintf(myaddr,"%s:%hu",my_hostip,receive_port);

	nw.last_global_scan=0;
	nw.search_result_return_address=g_strdup(myaddr);
	nw.search_query=g_string_new(pattern);
	nw.on_start=on_start;
	nw.on_incoming=on_incoming;
	nw.global_periodic=global_periodic;

	G_LOCK(search_to_start);
	search_to_start=g_array_append_val(search_to_start,nw);
	G_UNLOCK(search_to_start);
}

/************************************************************************/
/* check if the given pattern exists inside filename (case insensitive) */
/************************************************************************/
/* output: address of the pattern or NULL */
/******************************************/
const char *my_strcasestr(const GString *pattern, const char *filename)
{
   int max_pos;
   int i;

   max_pos=strlen(filename)-pattern->len;
   if(max_pos<0)             /* filename length < pattern length ? */
      return NULL;            /* no match */

   for(i=0;i<=max_pos;i++)
   {
      if(!strncasecmp(pattern->str,filename+i,pattern->len))
         return filename+i;
   }

   return NULL;               /* no match */
}


/* ----------------------------------------------------------------------------------------- */
/* ----------------------------------------------------------------------------------------- */
/* ----------------------------------------------------------------------------------------- */
/* ---- find users having a buggy clone as client which returns invalid result to query ---- */
/* ----------------------------------------------------------------------------------------- */
/* ----------------------------------------------------------------------------------------- */
/* ----------------------------------------------------------------------------------------- */

/***************************************************************************************************/
/* this function analyzes the result of the "T?T?1?7?azerty" search query to identify buggy clones */
/***************************************************************************************************/
static int junk_autosearch_sr(void *xtra_param,const char *nick, const char *filename, unsigned long long filesize, unsigned int free_slot, unsigned int ttl_slot)
{
	const GString pattern={"azerty",strlen("azerty")};

	printf("%s junk_autosearch_sr: '%s'\n",my_name,filename);

	if	(
		(filesize>1)|| /* the size is above what we want, it is a buggy clone */
		(my_strcasestr(&pattern,filename)==NULL)		/* the wanted string is not in the result */
		)
	{
		G_LOCK(hub_com);
		kick_user(hub_com,my_name,nick,"You are using a buggy clone polluting search results. Don't come back until you have a valid client...");
		G_UNLOCK(hub_com);
	}
	return 0;
}

/**********************************************************************************************************************/
/* junk autosearch searchs for buggy DC clone returning everything but what it is expected when a research is started */
/**********************************************************************************************************************/
/* This function receives the search results of the query "$Search myhost:myip T?T?1?7?azerty|" sent to each   */
/* incoming user.                                                                                              */
/* theoritically, this should returns the list of all video named "*azerty*" with a size of 0 or 1 byte length */
/* which is quite rare. If the received result matchs the query, nothing appends but if it doesn't match,      */
/* the user is kicked and banned and its IP is banned.                                                         */
/***************************************************************************************************************/
/* NOTE: to reduce bandwidth usage, when added, a search is immediatly started and a "one user" search is */
/*       performed each time a user enters the hub                                                        */
/**********************************************************************************************************/
static void *junk_autosearch_fnc(void *dummy)
{
	int incoming_result_fd;
	static TBLDECODE gen_dec={NULL,NULL,NULL,NULL,junk_autosearch_sr,NULL,NULL,NULL,NULL,NULL};

	incoming_result_fd=_x_udp(0);			/* create an UDP socket and found an unused port */
	if(incoming_result_fd==-1)
	{
		fprintf(stderr,"junk_autosearch_fnc: fail to create UDP socket.\n");
	}
	else
	{
		struct sockaddr_in lcl;
		int len_lcl=sizeof(lcl);

		if(getsockname(incoming_result_fd,(void*)&lcl,&len_lcl)!=0)
		{
			perror("getsockname fail");
			shutdown(incoming_result_fd,2);
			close(incoming_result_fd);
		}
		else
		{
			GString *str;

			add_pattern_to_search(ntohs(lcl.sin_port),"T?T?1?7?azerty",TRUE,TRUE,FALSE);

			str=g_string_new("");

			/* loop for ever and process received result */
			while(1)
			{
				int ret;
				char buf[8192];
#if HAVE_NO_MSG_NOSIGNAL
				void (*old_handler)(int);
				old_handler = signal (SIGPIPE, SIG_IGN);
#endif
				ret=recv(incoming_result_fd,buf,sizeof(buf)-1,MSG_NOSIGNAL);
#if HAVE_NO_MSG_NOSIGNAL
				signal (SIGPIPE, old_handler);
#endif
				if(ret!=-1)
				{
					buf[ret]='\0';
					
					/* to know if the result is buggy, we have to check if the size is above 1 */
					/* or if the returned filename does not contain the string "azerty" */

					str=g_string_assign(str,buf);
					generic_decoder(str,&gen_dec,NULL);
				}
			}
		}
	}
	pthread_exit(NULL);
	return(NULL);
}

static void start_auto_search()
{
	pthread_t tid;
	pthread_attr_t thread_attr;
	pthread_attr_init (&thread_attr);
	pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);
	if(pthread_create(&tid,&thread_attr,(void*)junk_autosearch_fnc,NULL)!=0)
	{
		fprintf(stderr,"Fail to create auto search thread.\n");
		exit(1);
	}
}

/**********************************************************/
/* this message is received each time a user enters a hub */
/*****************************************************************************************/
/* when a user enters the hub, we send him all the query having the on_incoming flag set */
/*****************************************************************************************/
static int global_hello_fnc(void *xtra_param, const char *nickname)
{
	int i;
	GString *output;

	output=g_string_new("");

	printf("%s global hello handler: '%s'\n",my_name,nickname);
	G_LOCK(search_to_start);
	for(i=0;i<search_to_start->len;i++)
	{
		AUTOSEARCH *as;

		as=&(g_array_index(search_to_start,AUTOSEARCH,i));
		if(as->on_incoming)
		{
			g_string_sprintf(output,"$UniSearch %s %s %s|",nickname,as->search_result_return_address,as->search_query->str);
			printf("%s global hello handler sends: '%s'\n",my_name,output->str);
			G_LOCK(hub_com);
			send_dc_line(hub_com,output->str,NULL);
			G_UNLOCK(hub_com);
		}
	}
	
	G_UNLOCK(search_to_start);

	g_string_free(output,TRUE);
	return 0;
}

/*********************************************************************************************************/
/* the program receives 3 parameters: its name (has it appears on the hub), the hub com socket number and*/
/* the localhost IP (not 127.0.0.1, the IP used by users to connect to the hub).                         */
/* NOTE: the dc_com.c is the same as the src one but you must provide the DC_INPUT_NO_TIMEOUT else after */
/* 30 seconds idle, you will received a NULL string.                                                     */
/*********************************************************************************************************/
int main(int argc,char **argv)
{
	GString *str;

	if(argc!=4)
	{
		fprintf(stderr,"Usage: %s name fdnumber localhostIP\n",argv[0]);
		exit(1);
	}

	my_name=argv[1];
	sscanf(argv[2],"%u",&hub_com);

	my_hostip=g_strdup(argv[3]);

	g_thread_init(NULL);

	search_to_start=g_array_new(FALSE,FALSE,sizeof(AUTOSEARCH));
	start_auto_search();

	while(1)
	{
		static TBLDECODE global_gen_dec={NULL,NULL,NULL,NULL,NULL,NULL,NULL,global_hello_fnc,NULL,NULL};
		str=get_a_dc_line(hub_com);

		if(str==NULL)
			break;

		printf("%s received: '%s'\n",my_name,str->str);
	
		generic_decoder(str,&global_gen_dec,NULL);

		g_string_free(str,TRUE);

	}
	printf("%s has lost hub connection and leaves\n",my_name);
	return 0;
}

