#! /usr/bin/perl

# $Id: main.pl,v 2.17 2003/07/20 07:50:57 blusseau Exp $

use strict;
use Locale::gettext;
use POSIX;     # Needed for setlocale()

use vars qw(%GLOBAL_VARS);

require "dchub_perl_functions.pl"; # Some usefull functions

my $script_dir="";
my $script_conf_file;
my $last_test_mtime=0;
my $current_file;

my %handlers;
my %modules_mtime;
my %handlers_from_file;

$|=1; # Autoflush

# Add the path of main.pl script to the Perl libraries path
BEGIN {
	my ($libdir) = $0 =~m|(.*)/|; # extract path from $0
	push (@INC,$libdir) unless grep {/$libdir/} @INC;
};

#*****************************************
# load and interpret the scripts.conf file
#*****************************************
# Parameter: 1: force reloading source scripts (0: don't force, 1: force)
sub load_config {
	my $force=shift;
	%GLOBAL_VARS={};
	if (open(CONF,$script_conf_file)) {
	 	while (<CONF>) {
			chomp;
			s/^\s+//g;
			next if ((/^\#/) or (not $_));
			# print "$_\n";
		  SWITCH: {
				s/^set\s+// && do {
					my ($var,$value)=split(/\s+/);
					$value=unquote($value);
					my $str=qq(\$GLOBAL_VARS{$var}="$value";);
					eval $str;
					last SWITCH;
				};
				s/^source\s+// && do {
					source_file($_,$force);
					last SWITCH;
				};
			}
		}
		close(CONF);
	}
}

#****************************
# source a script from a file
#****************************
# parameter: 1: the file to source
#			 2: force the load (0=don't force, 1=force)
sub source_file {
	my $file=shift;
	my $force=shift;
	$file="$script_dir/$file" unless ($file=~/^\//);
	$current_file=$file;
	if ($force == 1 || !grep {/$file/} keys(%INC)) {
		delete $INC{$file} if $INC{$file}; # so we can reload the file with require
		if (! -f $file) {
			print STDERR sprintf("*** "._("Perl Init: Can't source %s").":"._("No such file").".\n",$file);
		} else {
			eval {
				require $file;	# try to source the script
			};
			if ($@) {
				print STDERR sprintf("*** "._("Perl Init: Can't source %s").":\n$@",$file);
			} else {
				my $mtime = (stat("$file"))[9];
				$modules_mtime{$file}=$mtime;
			}
		}
	} else {
		print STDERR sprintf("*** "._("Perl Init: Script %s already loaded").".\n",$file);
	}
	$current_file="";
}

#***********************************************************************
# initialize some variables just after the interpreter has been loaded
#***********************************************************************
# parameter: 1 string: the script dir
sub dchub_perl_init {
	$script_dir=$_[0];
	
	# For localisation
	bindtextdomain("dchub_handlers","$script_dir/i18n");
	textdomain("dchub_handlers");

	my $locale=dchub::db_get("LANGUAGE");
	if (defined $locale) {
		setlocale(LC_ALL, "$locale");
		print "Perl: ".sprintf(_("Using %s as locale for perl handlers"),$locale).".\n";
	}

	$script_conf_file="$script_dir/dchub_scripts.conf";
	load_config(0);
}

#***************************
# add a hook to an handler *
#***************************
# parameter: 1: string: name of the event (ex: myinfo)
#					   (see the Documentation/script for the list of events)
# 			 2: reference to a sub containing the code to be execute when the event is fire
sub add_hook ($&) {
	my $event=shift;
	my $function=shift;
	# print "PUSH:[$event -> $function\n";
	
	push @{$handlers{$event}},$function;
	push @{$handlers_from_file{$current_file}},$function if $current_file;
}

#***********************************************************************
# Catch all invalid method calls
# the function tries to define an handler to call all the hooks
# that are associated with it.
#***********************************************************************
sub AUTOLOAD {
	local *FH;
	our ($AUTOLOAD);

	my $event=$AUTOLOAD;
	# print "EVENT:$event\n";
	if ($event=~s/^.*::dchub_(.*)_handler/$1/) {
		my $funcname="dchub_${event}_handler";
		if (not defined $handlers{$funcname}) {
			my $eval=qq(sub $funcname { my \$array=\$handlers{$event};).
						q(
						  foreach my $refsub (@$array) {
							  &$refsub(@_);
						  }
					  });
			eval $eval;
			no strict 'refs';
			&$funcname(@_);
		}
	} else {
		$event=~s/^main:://;
		print STDERR "*** ".sprintf(_("Unknown Perl function:%s"),$event).".\n";
	}
}

#***********************************************************************
# undefined all autoloaded perl handlers
#***********************************************************************
sub dchub_perl_clear_autoloaded_handlers {
	no strict 'refs';
	foreach my $function (keys %handlers) {
		delete $handlers{$function};
	}
	foreach my $file (keys %handlers_from_file) {
		delete $handlers_from_file{$file};
	}	
	load_config(1); # Force the reload of source scripts
	my $locale=dchub::db_get("LANGUAGE");
	if (defined $locale) {
		setlocale(LC_ALL, "$locale");
		print "Perl: ".sprintf(_("Using %s as locale for perl handlers"),$locale).".\n";
	}
}

#******************************************************************************
#******************************************************************************
#  Event handler entry point
#******************************************************************************
#******************************************************************************
# parameter: 1 hash table (evt_array)
sub dchub_perl_handler {
  my $evt_array = $_[0];	# evt_array is a reference to a hash table (%) containing all the parameters
							# the event array contains at least the following entries
							# "event": the name of the event
							# "nickname": the nickname of the user generating this event
							# "argc": the number of optionnal arguments
							# if optionnal arguments exist, they are named "0", "1", "2", "3, ...
	
  my $hdl_name="dchub_".$$evt_array{"event"}."_handler";

  # Test if some sources has changed (wait 30s between two tests)
  my $now=time();
  if ($now - $last_test_mtime > 30) {
	  foreach my $file (keys(%modules_mtime)) {
		  if (my $mtime = (stat($file))[9]) {
			  if ($mtime > $modules_mtime{$file}) { # Check if the file has been modified
				  print "Perl: ".sprintf(_("Reloading %s script due to file change"),$file).".\n";
				  foreach my $routine (@{$handlers_from_file{$file}}) {
					  foreach my $event (keys(%handlers)) {
						  # Remove the old routine from the event handler
						  $handlers{$event}=[grep { $_ != $routine } @{$handlers{$event}}];
					  }
				  }
				  delete $handlers_from_file{$file};
				  source_file($file,1);
			  }
		  }
	  }
	  $last_test_mtime=$now;
  }
  # call the handle which must deal with this event
  no strict 'refs';
  eval {
      &$hdl_name($evt_array); # Call the sub (AUTOLOAD if undef)
  };
  if ($@) {
  	  print _("*** Perl Script ERROR executing the handler")." $hdl_name:\n$@";
  }
}

#***********************************************************************
# Extended globalchat handler that extract nickname and message
# from global chat. This function also translate some forbidden protocol
# char like | and $
#***********************************************************************
sub dchub_globalchat_handler(\%) {
	my $evt_array = $_[0];
	my $nickname=$evt_array->{"nickname"};
	my $msg=$evt_array->{0};
	$msg=~s/<\Q$nickname\E>\s*//;
	$msg=~s/\|+$//;
	my $array=$handlers{globalchat};
	foreach my $refsub (@$array) {
		&$refsub($evt_array,$nickname,unescape($msg));
	}
}

#***********************************************************************
# Extended privchat handler that extract sender, receiver and message
# from private chat. This function also translate some forbidden protocol
# char like | and $
#***********************************************************************
sub dchub_privchat_handler(\%) {
	my $evt_array = $_[0];
	my $from=$evt_array->{"nickname"};
	my $to=$evt_array->{0};
	my $msg=$evt_array->{1};
	$msg=~s/<\Q$from\E>\s*//;
	$msg=~s/\|+$//;
	my $array=$handlers{privchat};
	foreach my $refsub (@$array) {
		&$refsub($evt_array,$from,$to,unescape($msg));
	}
}

#******************************************************************************
# sub to avoid writing gettext
sub _(@) { gettext(@_); }
