/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * key.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: key.c,v 2.1 2003/02/16 11:02:05 eric Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
	#include <unistd.h>
#endif  /* HAVE_UNISTD_H */
#include <string.h>
#include <errno.h>
#include <glib.h>

#include "dc_com.h"
#include "key.h"

/*************************************************************************/
/* the key is quite easily :) computed from the lock                     */
/* key[x]= ns(lock[x]^lock[x-1])							                      */
/*   ns is a nibble swap (switch the upper 4 bits with the lower 4 bits) */
/* exception:                                                            */
/* key[0] is a bit different                                             */
/* let's name A and B the 2 last bytes of the lock                       */
/* key[0]= ns(lock[0]^A^B^magic)       ; magic is a kind of magic nibble */
/*************************************************************************/
GString *compute_hub_reg_access_key(GString *lck, unsigned char magic)
{
	GString *key;
	int i;
	unsigned int u,l,o,r;
	unsigned char v;

	key=g_string_sized_new((unsigned)(lck->len));

	/* first byte is a special case */
	{
		u=(((unsigned int)(lck->str[0]))&255);
		l=(((unsigned int)(lck->str[lck->len-1]))&255);
		o=(((unsigned int)(lck->str[lck->len-2]))&255);

		/* do xor */
		r=(u^l^o)^magic;		/* don't forget the magic 0x14 */

#if 0
		printf("%02X^%02X^%02X^%02X => %02X\n",
					u,l,o,(unsigned int)magic,r);
#endif
		/* do nibble swap */
		v=(((r<<8)|r)>>4)&255;
		
		switch(v)
		{
			case '\0':		/* \0 is a special value */
			case 5:
						g_string_sprintfa(key,"/%%DCN%03u%%/",((unsigned)v)&255);
						break;

	
			case '$':		/* $ is a special value */
						key=g_string_append(key,"/%DCN036%/");
						break;

			case 96:
						key=g_string_append(key,"/%DCN096%/");
						break;

			/* / is probably a special value but I have never seen it until now */

			default:
						key=g_string_append_c(key,v);
						break;
		}
	}

	/* all other bytes use the same code */
	for(i=1;i<lck->len;i++)
	{
		u=(((unsigned int)(lck->str[i]))&255);
		l=(((unsigned int)(lck->str[i-1]))&255);
		
		/* do xor */
		u=u^l;		

		/* do nibble swap */
		v=(((u<<8)|u)>>4)&255;
		
		switch(v)
		{
			case '\0':		/* \0 is a special value */
			case 5:
						g_string_sprintfa(key,"/%%DCN%03u%%/",((unsigned)v)&255);
						break;

			case '$':		/* $ is a special value */
						key=g_string_append(key,"/%DCN036%/");
						break;

			case 96:
						key=g_string_append(key,"/%DCN096%/");
						break;

			/* / is probably a special value but I have never seen it until now */

			default:
						key=g_string_append_c(key,v);
						break;
		}
	}
	return key;
}

/*************************************************************************/
/* the key is quite easily :) computed from the lock                     */
/* key[x]= ns(lock[x]^lock[x-1])							                      */
/*   ns is a nibble swap (switch the upper 4 bits with the lower 4 bits) */
/* exception:                                                            */
/* key[0] is a bit different                                             */
/* let's name A and B the 2 last bytes of the lock                       */
/* key[0]= ns(lock[0]^A^B^0x05)         ; 0x05 is a kind of magic nibble */
/*************************************************************************/
GString *compute_access_key(GString *lck)
{
	return compute_hub_reg_access_key(lck,0x05);
}

/******************************************/
/* access to dc hub is locked using a key */
/* this function compute and send the key */
/************************************************/
/* medium-level function:                       */
/* same function as previous one except it uses */
/* the given lock instead of taking it from sck */
/* 1) decode the input string                   */
/* 2) compute the "$Key yyyy| reply and send it */
/***********************************************************************/
/* input: socket to use (output to write key)                          */
/*        string "$Lock ....|                                          */
/* output: -1: unable to get $Lock                                     */
/*         -2: no valid string obtained                                */
/*         -3: obtained string is not a string beginning by $Lock      */
/*         -4: ununderstandable $Lock value                            */
/*         -5: lock value to small (never happend, -2 should catch it) */
/*         -6: unable to write $Key to the server                      */
/***********************************************************************/
int do_unlock_access(int sck,GString *lck)
{
	char *t;
	GString *key;

#if 0
	disp_msg(DEBUG_MSG,"unlock_access","received: ",lck->str,NULL);
#endif

	if(lck->len<10)	/* min size is "$Lock xxx|" where xxx is a 3 bytes lock */
	{
#if 0
		disp_msg(ERR_MSG,"unlock_access","Lock too small", lck->str,NULL);
#endif
		return -2;
	}

	if(strncmp(lck->str,"$Lock ",6))
	{
#if 0
		disp_msg(ERR_MSG,"unlock_access","Lock not found",lck->str,NULL);
#endif
		return -3;		/* doesn't start with "$Lock " */
	}

	/* remove "$Lock " */
	lck=g_string_erase(lck,0,6);

	/* inside the Lock, the first thing to know is everything after the first space in xxx */
	/* is ignored */
	t=strchr(lck->str,' ');		/* search a space in xxx */
	if(t==NULL)
	{
#if 0
		disp_msg(ERR_MSG,"unlock_access","Ununderstandable lock",lck->str,NULL);
#endif
		return -4;		/* no space, never seen case */
	}

	if((t-lck->str)<3)
	{
#if 0
		disp_msg(ERR_MSG,"unlock_access","Lock too small", lck->str,NULL);
#endif
		return -5;		/* key too small, min 3 bytes */
	}
	
	lck=g_string_truncate(lck,t-lck->str);
#if 0
	disp_msg(DEBUG_MSG,"unlock_access","using this val: ",lck->str,NULL);
#endif

	key=compute_access_key(lck);

#if 0
	disp_msg(DEBUG_MSG,"unlock_access","key is: ",key->str,NULL);
#endif
	
	/* now, send the key */
	key=g_string_prepend(key,"$Key ");
	key=g_string_append_c(key,'|');

#if 0
	disp_msg(DEBUG_MSG,"unlock_access","sending: ",key->str,NULL);
#endif

	if(write(sck,key->str,(unsigned int)(key->len))!=key->len)
	{
#if 0
		disp_msg(ERR_MSG,"unlock_access","Unable to send key", strerror(errno),NULL);
#endif
		g_string_free(key,TRUE);
		return -6;
	}

#if 0
	disp_msg(DEBUG_MSG,"unlock_access","Ok",NULL);
#endif
	g_string_free(key,TRUE);
	return 0;
}

/******************************************/
/* access to dc hub is locked using a key */
/************************************************/
/* this function is the highest level function, */
/* it does everything using the given socket    */
/* 1) get the "$Lock xxxx|" from the hub        */
/* 2) compute the "$Key yyyy| reply and send it */
/***********************************************************************/
/* input: socket to use (input to get lock and output to write key)    */
/* output: -1: unable to get $Lock                                     */
/*         -2: no valid string obtained                                */
/*         -3: obtained string is not a string beginning by $Lock      */
/*         -4: ununderstandable $Lock value                            */
/*         -5: lock value to small (never happend, -2 should catch it) */
/*         -6: unable to write $Key to the server                      */
/*         if output!=0 and line!=NULL, (*line) is set to the line     */
/*         received from the hub.                                      */
/*         (*line) must be freed when no more used.                    */
/***********************************************************************/
int unlock_access(int sck, GString **line)
{
	GString *lck;
	int a;

	lck=get_a_dc_line(sck);

	if(lck==NULL)
	{
#if 0
		disp_msg(ERR_MSG,"unlock_access","Connection closed",NULL);
#endif
		return -1;
	}

	a=do_unlock_access(sck,lck);
	if((a!=0)&&(line!=NULL))
		(*line)=lck;
	else
		g_string_free(lck,TRUE);

	return a;
}

/*****************************************************************************************************/
/* this function computes the key based on c_lock and compares it to input (containing "$Key xxxx|") */
/*****************************************************************************************************/
/* output: 1=fail, 0=ok */
/************************/
int verify_key(char *c_lock,GString *input)
{
	GString *lck,*key;

	if(strncmp(input->str,"$Key ",5))
		return 1;

	/* compute the wanted key */
	lck=g_string_new(c_lock);
	key=compute_access_key(lck);
	g_string_free(lck,TRUE);

	if(strncmp(input->str+5,key->str,(unsigned int)(key->len)))
	{
#if 0
		int i;
		disp_msg(ERR_MSG,"verify_key","key not ok",NULL);
		disp_msg(ERR_MSG,"verify_key","wanted:",key->str,NULL);
		disp_msg(ERR_MSG,"verify_key","have:  ",input->str+5,NULL);
		
		for(i=0;i<key->len;i++)
		{
			printf("%c(%02X) ",key->str[i],((int)key->str[i])&255);
		}
#endif

		g_string_free(key,TRUE);
		return 1;
	}

	g_string_free(key,TRUE);
	return 0;
}

