/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * passwd.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: passwd.c,v 2.4 2003/04/06 06:43:39 ericprev Exp $
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <regex.h>
#include <glib.h>

#include "gvar.h"
#include "passwd.h"
#include "toolkit.h"
#include "common_defines.h"

/********************************************************/
/* verify if the username do not match the locked patern*/
/*********************************************************/
/* output: 1 if the name is lock else 0                  */
/*********************************************************/
int is_lock_nick(const char *nick)
{
	int i;
	regex_t re;
	char *p;

	G_LOCK(gl_lock_patern);
	if (gl_lock_patern == NULL)
	{
		G_UNLOCK(gl_lock_patern);
		return (0);
	}
	for(i=0;i<gl_lock_patern->len;i++)
	{
		p=g_ptr_array_index(gl_lock_patern,i);
		if (regcomp(&re, p, REG_EXTENDED|REG_NOSUB)!=0)
			continue;
		if (regexec(&re, nick, 0, NULL, 0) == 0) /* regexec return 0 if match */
		{
			G_UNLOCK(gl_lock_patern);
			regfree(&re);
			return (1);
		}
	}
	regfree(&re);
	G_UNLOCK(gl_lock_patern);
	return (0);
}

/*************************************************/
/* get the password associated to the given nick */
/*************************************************************/
/* output: NULL if no password required (*op_type unchanged) */
/*         else, it is the password (must be free()d when no */
/*         more required. (*op_type) is set to the type of   */
/*         "account": | 0: normal user,                      */
/*                    | 1: operator,                         */
/*                    | 2: master of the hub                 */
/*                    | 4: Bot                               */
/*                    |16: hub not a user                    */
/*************************************************************/
char *get_user_password(char *nick, int *op_type, int *ext_type)
{
	char *passwd=NULL;
	char buf[PASSWD_ENTRY];
	
	*ext_type = 0;
	G_LOCK(user_file);
	rewind(user_file);
	while(fread(buf,1,PASSWD_ENTRY,user_file)==PASSWD_ENTRY)
	{
		if((buf[0]=='U')&&
			(!strcasecmp(nick,buf+NICK_PASS_POS)))
		{
			passwd=strdup(buf+PASS_PASS_POS);
			switch(((int)buf[TYPE_PASS_POS])&255)
			{
				case 'O':	*op_type=TYPE_O_PRIV;
								break;
				case 'G':	*op_type=TYPE_G_PRIV;
								break;
				case 'M':	*op_type=TYPE_M_PRIV;
								break;
				case 'b':	*op_type=TYPE_b_PRIV;
								break;
				case 'B':	*op_type=TYPE_B_PRIV;
								break;
				case 'V':	*op_type=TYPE_V_PRIV;
						*ext_type=SHILD_MODE;
								break;
				default:
				case 'N':	*op_type=TYPE_N_PRIV;
								break;
			}
			goto done;
		}
	}

	done:
	G_UNLOCK(user_file);

	if (passwd == NULL)
	{
		if (is_lock_nick(nick))
			passwd = strdup("||||"); /* this password can not be use by the client then it's impossible to log in */
	}
	return passwd;
}

/**********************************************/
/* check if the given nickname has an account */
/**************************************************************/
/* output: 1=yes, 0=no. If op_type is not NULL and output==1, */
/* *op_type contains the type of "account"                    */
/*                   == 0: normal user,                       */
/*                    | OP_PRIV    : operator,                */
/*                    | MASTER_PRIV: master of the hub        */
/*                    | BOT_PRIV   : Bot                      */
/*                    | HUB_PRIV   : hub not a user           */
/**************************************************************/
int account_exist(const char *nick, int *op_type, int *extra)
{
	int ret=0;
	char buf[PASSWD_ENTRY];
	
	G_LOCK(user_file);
	rewind(user_file);
	if (extra != NULL)
		extra = 0;
	while(fread(buf,1,PASSWD_ENTRY,user_file)==PASSWD_ENTRY)
	{
		if((buf[0]=='U')&&
			(!strcasecmp(nick,buf+NICK_PASS_POS)))
		{
			ret=1;
			if(op_type!=NULL)
			{
				switch(((int)buf[TYPE_PASS_POS])&255)
				{
					case 'O':	*op_type=TYPE_O_PRIV;
									break;
					case 'G':	*op_type=TYPE_G_PRIV;
									break;
					case 'M':	*op_type=TYPE_M_PRIV;
									break;
					case 'b':	*op_type=TYPE_b_PRIV;
									break;
					case 'B':	*op_type=TYPE_B_PRIV;
									break;
					case 'V':	*op_type=TYPE_V_PRIV;
							if (extra != NULL)
								*extra = SHILD_MODE;
									break;
					default:
					case 'N':	*op_type=0;
									break;
				}
			}
			goto done;
		}
	}

	done:
	G_UNLOCK(user_file);

	return ret;
}

/**********************************************/
/* check if the given nickname has an account */
/**************************************************************/
/* output: 1=yes, 0=no. If op_type is not NULL and output==1, */
/* *type contains the type of "account"                       */
/*                    =='N': normal user,                     */
/*                    =='O': operator,                        */
/*                    =='M': master of the hub                */
/*                    =='b': Bot                              */
/*                    =='B': Master Bot                       */
/*                    =='V': Vip                              */
/**************************************************************/

int get_account_type(const char *nick, char *type)
{
	int ret=0;
	char buf[PASSWD_ENTRY];
	
	G_LOCK(user_file);
	rewind(user_file);
	while(fread(buf,1,PASSWD_ENTRY,user_file)==PASSWD_ENTRY)
	{
		if((buf[0]=='U')&&
			(!strcasecmp(nick,buf+NICK_PASS_POS)))
		{
			ret=1;
			if(type!=NULL)
				*type = buf[TYPE_PASS_POS];
			goto done;
		}
	}

	done:
	G_UNLOCK(user_file);

	return ret;
}


/******************************************************************/
/* create the list of all registered users with their information */
/******************************************************************/
/* the returned Gstring must be freed when no more useful */
/**********************************************************/
GString *full_user_list(void)
{
	GString *lst;
	char buf[PASSWD_ENTRY];
	int nb=0;
	int usr=0,op=0,master=0,bot=0,mbot=0,gost_op=0,gost=0;

	lst=g_string_new("T:Nickname:Password:\r\n");

	G_LOCK(user_file);
	rewind(user_file);
	while(fread(buf,1,PASSWD_ENTRY,user_file)==PASSWD_ENTRY)
	{
		if(buf[0]=='U')
		{
			g_string_sprintfa(lst,"%c:%.32s:%.16s:\r\n",buf[TYPE_PASS_POS],buf+NICK_PASS_POS,buf+PASS_PASS_POS);
			nb++;
			switch(((int)buf[TYPE_PASS_POS])&255)
			{
				case 'O':	op++;
								break;
				case 'G':	gost_op++;
								break;
				case 'M':	master++;
								break;
				case 'b':	bot++;
								break;
				case 'B':	mbot++;
								break;
				case 'V':	usr++;
								break;
				case 'g':	gost++;
								break;
				default:
				case 'N':	usr++;
								break;
			}
		}
	}

	G_UNLOCK(user_file);

	g_string_sprintfa(lst,"%d account%s exist%s (users: %d, gost: %d, operator: %d, gost_op: %d, bot: %d, master-bot: %d, master: %d)\r\n",
								nb,(nb<2)?"":"s",(nb<2)?"s":"",usr,gost,op,gost_op,bot,mbot,master);
	return lst;
}

/******************************************************************************************************/
/* create the list of all registered users having the given account type ('O','M','b','B','V','g','G' or 'N') */
/******************************************************************************************************/
/* the returned Gstring must be freed when no more useful */
/**********************************************************/
static GString *account_user_list(char account_type)
{
	GString *lst;
	char buf[PASSWD_ENTRY];
	int nb=0;

	lst=g_string_new("T:Nickname:Password:\r\n");

	G_LOCK(user_file);
	rewind(user_file);
	while(fread(buf,1,PASSWD_ENTRY,user_file)==PASSWD_ENTRY)
	{
		if(buf[0]=='U')
		{
			if(buf[TYPE_PASS_POS]==account_type)
			{
				g_string_sprintfa(lst,"%c:%.32s:%.16s:\r\n",buf[TYPE_PASS_POS],buf+NICK_PASS_POS,buf+PASS_PASS_POS);
				nb++;
			}
		}
	}

	G_UNLOCK(user_file);

	g_string_sprintfa(lst,"%d account%s exist%s\r\n", nb,(nb<2)?"":"s",(nb<2)?"s":"");
	return lst;
}

/*********************/
/* User list for gui */
/*********************/

int	gui_account_user_list()
{
	char buf[PASSWD_ENTRY];

	G_LOCK(user_file);
	rewind(user_file);
	while(fread(buf,1,PASSWD_ENTRY,user_file)==PASSWD_ENTRY)
		if(buf[0]=='U')
			printf("_adu:%.32s %c %.16s\n", buf + NICK_PASS_POS,
											buf[TYPE_PASS_POS],
											buf + PASS_PASS_POS);
	G_UNLOCK(user_file);
	return 0;
}



/* same as full_user_list but only MASTER account are returned */
GString *master_user_list(void)
{
	return account_user_list('M');
}

/* same as full_user_list but only OP account are returned */
GString *op_user_list(void)
{
	return account_user_list('O');
}

/***************************************/
/* add a new password to the user list */
/***************************************/
/* output: 0=ok, !=0=error */
/********************************************************************/
/* note: this function doesn't perform any check of the given value */
/********************************************************************/
int add_user_to_ulist(char *nick, char *password, char account_type)
{
	char buf[PASSWD_ENTRY];
	long cur_pos;
	int ret=0;
	
	char nw_buf[PASSWD_ENTRY];

	/* create the new entry */
	nw_buf[0]='U';		/* entry is busy */
	strncpy_max(nw_buf+NICK_PASS_POS,nick,NICK_PASS_LEN);
	strncpy_max(nw_buf+PASS_PASS_POS,password,PASS_PASS_LEN);
	nw_buf[TYPE_PASS_POS]=account_type;

	/* and write it into the file */
	G_LOCK(user_file);
	rewind(user_file);
	cur_pos=ftell(user_file);
	while(fread(buf,1,PASSWD_ENTRY,user_file)==PASSWD_ENTRY)
	{
		if(buf[0]=='F')
		{
			/* empty entry found, replace it */
			fseek(user_file,cur_pos,SEEK_SET);
			fwrite(nw_buf,1,PASSWD_ENTRY,user_file);
			goto done;
		}
		cur_pos=ftell(user_file);
	}

	/* not empty entry found, seek to the latest valid position */
	fseek(user_file,cur_pos,SEEK_SET);
	if(fwrite(nw_buf,1,PASSWD_ENTRY,user_file)!=PASSWD_ENTRY)
		ret=1;

	done:
	G_UNLOCK(user_file);

	if ((ret == 0) && (gl_gui_mode))
		printf("_adu:%s %c %s\n", nick, account_type, password);

	return ret;
}

/************************************/
/* delete a user from the user list */
/************************************/
/* output: 0=ok, !=0=error */
/********************************************************************/
/* note: this function doesn't perform any check of the given value */
/********************************************************************/
int del_user_to_ulist(char *nick)
{
	char buf[PASSWD_ENTRY];
	long cur_pos;
	int ret=0;
	
	/* delete the entry */
	G_LOCK(user_file);
	rewind(user_file);
	cur_pos=ftell(user_file);
	while(fread(buf,1,PASSWD_ENTRY,user_file)==PASSWD_ENTRY)
	{
		if((buf[0]=='U')&&
			(!strcasecmp(nick,buf+NICK_PASS_POS)))
		{
			if (gl_gui_mode)
				printf("_dlu:%s\n", nick);
			/* entry found, clear it */
			buf[0]='F';
			fseek(user_file,cur_pos,SEEK_SET);
			fwrite(buf,1,PASSWD_ENTRY,user_file);
			goto done;
		}
		cur_pos=ftell(user_file);
	}

	/* entry not found */
	ret=1;

	done:
	G_UNLOCK(user_file);

	return ret;
}

/*****************************************/
/* rename the given oldnick into newnick */
/*****************************************/
/* output: 0=ok, !=0=error */
/***************************************************************/
/* note: this function doesn't check if newnick already exists */
/***************************************************************/
int rename_user_to_ulist(char *oldnick,char *newnick)
{
	char buf[PASSWD_ENTRY];
	long cur_pos;
	int ret=0;
	
	/* update the existing entry */
	G_LOCK(user_file);
	rewind(user_file);
	cur_pos=ftell(user_file);
	while(fread(buf,1,PASSWD_ENTRY,user_file)==PASSWD_ENTRY)
	{
		if((buf[0]=='U')&&
			(!strcasecmp(oldnick,buf+NICK_PASS_POS)))
		{
			/* entry found, clear it */
			if (gl_gui_mode)
				printf("_rnu:%s %s\n", oldnick, newnick);
			strncpy_max(buf+NICK_PASS_POS,newnick,NICK_PASS_LEN);
			fseek(user_file,cur_pos,SEEK_SET);
			fwrite(buf,1,PASSWD_ENTRY,user_file);
			goto done;
		}
		cur_pos=ftell(user_file);
	}

	/* entry not found */
	ret=1;

	done:
	G_UNLOCK(user_file);

	return ret;
}

/*********************************************/
/* change the password of the given nickname */
/*********************************************/
/* output: 0=ok, !=0=error */
/***************************/
int chg_user_passwd_to_ulist(char *nick,char *newpasswd)
{
	char buf[PASSWD_ENTRY];
	long cur_pos;
	int ret=0;
	
	/* update the existing entry */
	G_LOCK(user_file);
	rewind(user_file);
	cur_pos=ftell(user_file);
	while(fread(buf,1,PASSWD_ENTRY,user_file)==PASSWD_ENTRY)
	{
		if((buf[0]=='U')&&
			(!strcasecmp(nick,buf+NICK_PASS_POS)))
		{
			/* entry found, clear it */
			if (gl_gui_mode)
				printf("_pwu:%s %s\n", nick, newpasswd);
			strncpy_max(buf+PASS_PASS_POS,newpasswd,PASS_PASS_LEN);
			fseek(user_file,cur_pos,SEEK_SET);
			fwrite(buf,1,PASSWD_ENTRY,user_file);
			goto done;
		}
		cur_pos=ftell(user_file);
	}

	/* entry not found */
	ret=1;

	done:
	G_UNLOCK(user_file);

	return ret;
}

/*****************************************/
/* change the type of the given nickname */
/*****************************************/
/* output: 0=ok, !=0=error */
/***************************/
int chg_user_type_to_ulist(char *nick,char newtype)
{
	char buf[PASSWD_ENTRY];
	long cur_pos;
	int ret=0;
	
	/* update the existing entry */
	G_LOCK(user_file);
	rewind(user_file);
	cur_pos=ftell(user_file);
	while(fread(buf,1,PASSWD_ENTRY,user_file)==PASSWD_ENTRY)
	{
		if((buf[0]=='U')&&
			(!strcasecmp(nick,buf+NICK_PASS_POS)))
		{
			/* entry found, clear it */
			if (gl_gui_mode)
				printf("_tyu:%s %c\n", nick, newtype);
			buf[TYPE_PASS_POS]=newtype;
			fseek(user_file,cur_pos,SEEK_SET);
			fwrite(buf,1,PASSWD_ENTRY,user_file);
			goto done;
		}
		cur_pos=ftell(user_file);
	}

	/* entry not found */
	ret=1;

	done:
	G_UNLOCK(user_file);

	return ret;
}
