/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * format_xml_hublist.c: Copyright (C) Eric Prevoteau <www@a2pb.gotdns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: format_xml_hublist.c,v 1.3 2003/12/28 08:12:38 uid68112 Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#ifdef HAVE_XML2
#include <libxml/tree.h>
#include <libxml/xpath.h>
#include <glib.h>

xmlXPathObjectPtr getNodeSet (xmlDocPtr doc, const xmlChar *xpath)
{
   xmlXPathContextPtr context;
   xmlXPathObjectPtr result=NULL;

   context = xmlXPathNewContext(doc);
   if (context)
	{
      result = xmlXPathEvalExpression(xpath, context);
      if (result)
		{
         if(xmlXPathNodeSetIsEmpty(result->nodesetval))
			{
            xmlXPathFreeObject(result);
            result=NULL;
         }
      }
      xmlXPathFreeContext(context);
   }
   return result;
}

typedef enum
{
	KA_NAME=1,				/* type=string */
	KA_ADDRESS=2,			/* type=string */
	KA_DESCRIPTION=4,		/* type=string */
	KA_USERS=8,				/* type=int */
	KA_COUNTRY=16,			/* type=string */
	KA_SHARED=32,			/* type=bytes (number written in ascii) */
	KA_STATUS=64,			/* type=string */
	KA_MINSHARE=128,		/* type=bytes (number written in ascii) */
	KA_MINSLOT=256,		/* type=int */
	KA_MAXHUBS=512,		/* type=int */
	KA_MAXUSERS=1024,		/* type=int */
	KA_RELIABILITY=2048,	/* type=percent (number written in ascii) */
	KA_RATING=4096,		/* type=string */
	KA_PORT=8192			/* type=int */
} KNOWN_ATTRIBUTES;

typedef enum
{
	XEI_NAME=0,				
	XEI_ADDRESS,			
	XEI_DESCRIPTION,		
	XEI_USERS,				
	XEI_COUNTRY,			
	XEI_SHARED,			
	XEI_STATUS,			
	XEI_MINSHARE,		
	XEI_MINSLOT,		
	XEI_MAXHUBS,		
	XEI_MAXUSERS,		
	XEI_RELIABILITY,	
	XEI_RATING,		
	XEI_PORT,
	NB_XEI
} XML_ENTRY_IDX;

typedef struct
{
	const xmlChar *xpath_pattern;
	KNOWN_ATTRIBUTES flag;

	const char *property_name;
	XML_ENTRY_IDX xei;
} COL_DESCRIPTION;

static COL_DESCRIPTION cd[]={
								{"//Hubs/Columns//Column[@Name=\"Name\"]",KA_NAME,"Name",XEI_NAME},
								{"//Hubs/Columns//Column[@Name=\"Address\"]",KA_ADDRESS,"Address",XEI_ADDRESS},			/* it is the only mandatory column */
								{"//Hubs/Columns//Column[@Name=\"Description\"]",KA_DESCRIPTION,"Description",XEI_DESCRIPTION},
								{"//Hubs/Columns//Column[@Name=\"Users\"]",KA_USERS,"Users",XEI_USERS},
								{"//Hubs/Columns//Column[@Name=\"Country\"]",KA_COUNTRY,"Country",XEI_COUNTRY},
								{"//Hubs/Columns//Column[@Name=\"Shared\"]",KA_SHARED,"Shared",XEI_SHARED},
								{"//Hubs/Columns//Column[@Name=\"Status\"]",KA_STATUS,"Status",XEI_STATUS},
								{"//Hubs/Columns//Column[@Name=\"Minshare\"]",KA_MINSHARE,"Minshare",XEI_MINSHARE},
								{"//Hubs/Columns//Column[@Name=\"Minslots\"]",KA_MINSLOT,"Minslots",XEI_MINSLOT},
								{"//Hubs/Columns//Column[@Name=\"Maxhubs\"]",KA_MAXHUBS,"Maxhubs",XEI_MAXHUBS},
								{"//Hubs/Columns//Column[@Name=\"Maxusers\"]",KA_MAXUSERS,"Maxusers",XEI_MAXUSERS},
								{"//Hubs/Columns//Column[@Name=\"Reliability\"]",KA_RELIABILITY,"Reliability",XEI_RELIABILITY},
								{"//Hubs/Columns//Column[@Name=\"Rating\"]",KA_RATING,"Rating",XEI_RATING},
								{"//Hubs/Columns//Column[@Name=\"Port\"]",KA_PORT,"Port",XEI_PORT},
								{NULL,0}
							};


/*********************************************/
/* check which columns exist in the xml file */
/*********************************************/
static gboolean valid_columns(xmlDocPtr doc, guint32 *ka)
{
	int i;
	xmlXPathObjectPtr result;

	i=0;
	*ka=0;
	while(cd[i].xpath_pattern!=NULL)
	{
#if 0
		fprintf(stderr,"checking '%s'\n",cd[i].xpath_pattern);
#endif
		result=getNodeSet(doc,cd[i].xpath_pattern);
		if(result)
		{
			(*ka)|=cd[i].flag;
			xmlXPathFreeObject(result);
		}
		i++;
	}

	return (((*ka)&KA_ADDRESS)!=0);
}

/****************************************************/
/* populate property array using the given hub node */
/****************************************************/
static void get_node_properties(xmlNodePtr node, xmlChar **xml_entry, guint32 ka)
{
	int i;

	for(i=0;i<NB_XEI;i++)
	{
		if(cd[i].flag&ka)
		{
			xml_entry[i]=xmlGetProp(node,cd[i].property_name);
		}
		else
		{
			xml_entry[i]=NULL;
		}
	}
}

/****************************************************/
/* free the memory used by property array of a node */
/****************************************************/
static void free_node_properties(xmlChar **xml_entry)
{
	int i;

	for(i=0;i<NB_XEI;i++)
	{
		if(xml_entry[i]!=NULL)
		{
			xmlFree(xml_entry[i]);
			xml_entry[i]=NULL;
		}
	}
}

typedef struct
{
	char *source;
	int source_len;
	char *dest;
} CONV_STRING;

static CONV_STRING conv_string[]={
												{"&quote;",7,"\""},
												{"&apos;",6,"'"},
												{"&amp;",5,"&"},
												{"&lt;",4,"<"},
												{"&gt;",4,">"},
												{NULL,0,NULL}
											};
/**************************************************************/
/* input: decode=TRUE: convert HTML like code into ascii code */
/**************************************************************/
static void print_value(guint32 ka, KNOWN_ATTRIBUTES flag, xmlChar **xml_entry, XML_ENTRY_IDX xei, gboolean with_pipe, gboolean decode)
{
	if((ka&flag)&&(xml_entry[xei]!=NULL))
	{
		if((decode==FALSE)||(strchr(xml_entry[xei],'&')==NULL))		/* no decode or no '&' ('&' is the first character of a HTML code */
			printf("%s",xml_entry[xei]);
		else
		{
			char *t;

			t=xml_entry[xei];
			while(*t)
			{
				if(*t!='&')
				{
					fputc(*t,stdout);
					t++;
				}
				else
				{
					int i;
					gboolean found=FALSE;
					i=0;
					while(conv_string[i].source!=NULL)
					{
						if(!strncmp(conv_string[i].source,t,conv_string[i].source_len))
						{
							fputs(conv_string[i].dest,stdout);
							found=TRUE;
							t+=conv_string[i].source_len;
							break;
						}
						i++;
					}
					if(!found)
					{
						fputc(*t,stdout);
						t++;
					}
				}
			}
		}
	}
	if(with_pipe)
		printf("|");
}

static void process_and_display_hubs(xmlDocPtr doc, guint32 ka)
{
	xmlXPathObjectPtr result;

	result=getNodeSet(doc,"//Hubs/Hub[@Address]");		/* retrieve only hub having an address */
	if(result)
	{
		xmlNodeSetPtr nodeset;
		xmlChar *xml_entry[NB_XEI];
		int i;

		nodeset=result->nodesetval;
		for(i=0;i<nodeset->nodeNr;i++)
		{
			get_node_properties(nodeset->nodeTab[i],xml_entry, ka);

			if(strlen(xml_entry[XEI_ADDRESS]))
			{	/* few silly entries have an address field but nothing inside */
				/* output format is: */
				/* name|address[:port]|description|nbusers|country|shared|status|minshare|minslot|maxhub|maxusers|reliability|rating */
				print_value(ka,KA_NAME,xml_entry,XEI_NAME,TRUE,TRUE);
	
				/* no need to test if the address exist but the port may exist */
				printf("%s",xml_entry[XEI_ADDRESS]);
				if((ka&KA_PORT)&&(xml_entry[XEI_PORT]!=NULL))
				{
					if(strlen(xml_entry[XEI_PORT]))
					{
						if(strchr(xml_entry[XEI_ADDRESS],':')==NULL)		/* no port in the address */
						{
							printf(":%s",xml_entry[XEI_PORT]);
						}
					}
				}
				printf("|");
	
				print_value(ka,KA_DESCRIPTION,xml_entry,XEI_DESCRIPTION,TRUE,TRUE);
				print_value(ka,KA_USERS,xml_entry,XEI_USERS,TRUE,FALSE);
				print_value(ka,KA_COUNTRY,xml_entry,XEI_COUNTRY,TRUE,FALSE);
				print_value(ka,KA_SHARED,xml_entry,XEI_SHARED,TRUE,FALSE);
				print_value(ka,KA_STATUS,xml_entry,XEI_STATUS,TRUE,FALSE);
				print_value(ka,KA_MINSHARE,xml_entry,XEI_MINSHARE,TRUE,FALSE);
				print_value(ka,KA_MINSLOT,xml_entry,XEI_MINSLOT,TRUE,FALSE);
				print_value(ka,KA_MAXHUBS,xml_entry,XEI_MAXHUBS,TRUE,FALSE);
				print_value(ka,KA_MAXUSERS,xml_entry,XEI_MAXUSERS,TRUE,FALSE);
				print_value(ka,KA_RELIABILITY,xml_entry,XEI_RELIABILITY,TRUE,FALSE);
				print_value(ka,KA_RATING,xml_entry,XEI_RATING,FALSE,FALSE);
				printf("\r\n");		/* end with a CR-LF has a normal list */
			}
			free_node_properties(xml_entry);
		}

		xmlXPathFreeObject(result);
	}
}

/**********************************************************************/
/* process XML data in memory and print the hubs in a standard format */
/**********************************************************************/
void convert_and_print_xml_hub_list(const char *buffer, int size)
{
	xmlDocPtr doc;
	guint32 known_attributes;		/*  set of KNOWN_ATTRIBUTES flags */

	/* define an alias for the silly windows-1252 encoding */
	xmlAddEncodingAlias("ISO-8859-1","windows-1252");

	/* build the tree in memory */
	doc=xmlParseMemory(buffer,size);
	if(doc==NULL)
	{
		fprintf(stderr,"Unable to process xml hublist\n");
		return;
	}

	/* check if the xml file contains the mandatory columns */
	if(valid_columns(doc,&known_attributes))
	{
		process_and_display_hubs(doc,known_attributes);
	}
	else
	{
		fprintf(stderr,"Unable to process xml hublist (2)\n");
		return;
	}

	/* release xml DOC memory */
	xmlFreeDoc(doc);
}
#else
/**********************************************************************/
/* process XML data in memory and print the hubs in a standard format */
/**********************************************************************/
void convert_and_print_xml_hub_list(const char *buffer, int size)
{
	fprintf(stderr,"xml2 not available, list ignored.\n");
#warning XML2 not available. Support of XML hublists disabled.
}
#endif
