/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * factory.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: factory.c,v 1.3 2003/05/31 13:50:14 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <glib.h>

#include "factory.h"
#include "toolkit.h"

/*******************************/
/* create a new packet builder */
/*******************************/
PACKET_BUILD *ed2k_pb_new(void)
{
	PACKET_BUILD *pb;

	pb=g_malloc(sizeof(PACKET_BUILD));
	pb->gba=g_byte_array_new();

	pb->packet_size_at_pos=-1;
	pb->packet_size_begin_at_pos=-1;
	pb->packet_size_end_at_pos=-1;

	return pb;
}

/*******************************************************/
/* set the packet size guint32 at the current position */
/*******************************************************/
void ed2k_pb_set_size_location(PACKET_BUILD *pb)
{
	guint32 dummy_val=0;

	if(pb->packet_size_at_pos!=-1)
	{
		fprintf(stderr,"Warning: ed2k_pb_set_size_location: function already called for this PB.\n");
	}

	pb->packet_size_at_pos=pb->gba->len;
	g_byte_array_append(pb->gba,(void*)(&dummy_val),sizeof(guint32));	/* keep free space for the size */
}

/***********************************************************/
/* set the location of the first byte to count in the size */
/***********************************************************/
void ed2k_pb_set_size_begin_location(PACKET_BUILD *pb)
{
	pb->packet_size_begin_at_pos=pb->gba->len;
}

/************************************************************/
/* set the location of the first byte to ignore in the size */
/************************************************************/
void ed2k_pb_set_size_end_location(PACKET_BUILD *pb)
{
	pb->packet_size_end_at_pos=pb->gba->len;
}

/*******************************/
/* append a byte to the packet */
/*******************************/
void ed2k_pb_add_byte(PACKET_BUILD *pb, guint8 v8)
{
	g_byte_array_append(pb->gba,&v8,sizeof(guint8));
}

/******************************************/
/* append an array of bytes to the packet */
/******************************************/
void ed2k_pb_add_bytes(PACKET_BUILD *pb, guint8 *v8, int v8_size)
{
	g_byte_array_append(pb->gba,v8,v8_size);
}

/**********************************************************/
/* append a guint16 in little endian format to the packet */
/**********************************************************/
void ed2k_pb_add_lsb_2bytes(PACKET_BUILD *pb, guint16 v16)
{
	v16=GUINT16_TO_BE(v16);
	g_byte_array_append(pb->gba,(void*)&v16,sizeof(guint16));
}

/**********************************************************/
/* append a guint32 in little endian format to the packet */
/**********************************************************/
void ed2k_pb_add_lsb_4bytes(PACKET_BUILD *pb, guint32 v32)
{
	v32=GUINT32_TO_BE(v32);
	g_byte_array_append(pb->gba,(void*)&v32,sizeof(guint32));
}

/********************************************************************************/
/* add the given string prefixed by a guint16 containing its size to the packet */
/********************************************************************************/
void ed2k_pb_add_string(PACKET_BUILD *pb, char *string, int string_len)
{
	if(string_len==-1)
		string_len=strlen(string);

	ed2k_pb_add_lsb_2bytes(pb,string_len);
	g_byte_array_append(pb->gba,string,string_len);
}

/****************************************************************/
/* destroy the packet builder, compute packet size if it exists */
/* and return the built packet (to free with g_free)            */
/* if the size exists and limit is not defined, the begin_pos   */
/* will be the first byte and the end_pos is after the last byte*/
/****************************************************************/
GByteArray *ed2k_pb_end(PACKET_BUILD *pb)
{
	GByteArray *gba;

	if(pb->packet_size_at_pos!=-1)
	{
		guint32 psize;

		if(pb->packet_size_begin_at_pos==-1)
			pb->packet_size_begin_at_pos=0;
		if(pb->packet_size_end_at_pos==-1)
			ed2k_pb_set_size_end_location(pb);

		psize=pb->packet_size_end_at_pos - pb->packet_size_begin_at_pos;

		/* save the size (guint32) in little endian format */
		pb->gba->data[pb->packet_size_at_pos]=psize&255;
		pb->gba->data[pb->packet_size_at_pos+1]=(psize>>8);
		pb->gba->data[pb->packet_size_at_pos+2]=(psize>>16);
		pb->gba->data[pb->packet_size_at_pos+3]=(psize>>24);
	}

	gba=pb->gba;
	g_free(pb);
	return gba;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
guint32 take_lsb_4bytes(guint8 *addr)
{
	return GUINT32_TO_LE(GET_UAA_GUINT32(addr));
}

guint16 take_lsb_2bytes(guint8 *addr)
{
	return GUINT16_TO_LE(GET_UAA_GUINT16(addr));
}

