/* -*- Mode: C++; c-file-style: "stroustrup"; indent-tabs-mode: nil -*- */
/*
 * admin.cc
 *   Administration/Registration CGI for a DDT server.
 *
 * $Id: admin.cc,v 1.22 2002/03/30 22:21:17 benoit Exp $
 *
 * Copyright (c) 2000-2001 Remi Lefebvre <remi@step.polymtl.ca>
 * Copyright (c) 2000 Benoit Joly <benoit@dhis.net>
 *
 * Licensed under the GPLv2
 */

#include <iostream>
#include <string>

#include "CgiForm.h"
#include "DbAdmin.h"

Options *Options::i_ = NULL;

Logger *log;
CgiForm *form;
DbAdmin *db;

// registration
int reg(Options &opts);

// administration
int admin(Options &opts);

void initOptions(int argc, char **argv, Options &opts);

int main(int argc, char **argv)
{
    // parse options and/or config file
    Options *opts = Options::Instance();
    initOptions(argc, argv, *opts);

    // verify provided values
    bool exitRequired = false;
    if (!optinvoked(&opts->dbuser))
    {
        cerr << "please provide a dbuser" << endl;
        exitRequired = true;
    }
    if (!optinvoked(&opts->dbname))
    {
        cerr << "please provide a dbname" << endl;
        exitRequired = true;
    }
    if (!optinvoked(&opts->dbuser))
    {
        cerr << "please provide a dbuser" << endl;
        exitRequired = true;
    }
    if (exitRequired)
    {
        return -1;
    }
    
    try
    {
        // create objects
        log = Logger::Instance();
        log->setIdent(argv[0]);
        log->openSyslog(LOG_USER, LOG_DEBUG);
        
        form = new CgiForm(opts->templatePath, opts->foot);
        db = new DbAdmin(log, opts->dbname, opts->dbuser, opts->dbpass,
                         opts->domain, opts->banned, opts->reserved);
    }
    catch (DdtException &e)
    {
        cerr << e.message() << endl;
        return -1;
    }
    catch (...)
    {
        cerr << "Unknown error" << endl;
        return -1;
    }

    // we are the registration CGI
    int retCode = 0;
    
    form->display(opts->head);
    if (strcmp(argv[0], "register.cgi") == 0)
    {
        retCode = reg(*opts);
    }
    else
    {
        retCode = admin(*opts);
    }
    return retCode;
}

void initOptions(int argc, char **argv, Options &opts)
{
    optTitle("CGI - DDT's Registration/Administration CGIs\n");
    optrega(&opts.dbname, OPT_STRING, 'n', "dbname", "*database name");
    optrega(&opts.dbuser, OPT_STRING, 'u', "dbuser", "*database owner");
    optrega(&opts.dbpass, OPT_STRING, 'p', "dbpass", "*database password");
    optrega(&opts.domain, OPT_STRING, 'd', "domain", "*domain name");
    optrega(&opts.banned, OPT_STRING, 'b', "banned", "*banned names file");
    optrega(&opts.reserved, OPT_STRING, 'r', "reserved", "*reserved file");
    optrega(&opts.head,   OPT_STRING, 'h', "head",   "*header file");
    optrega(&opts.foot,   OPT_STRING, 'f', "foot",   "*footer file");
    optrega(&opts.templatePath,   OPT_STRING, 't', "template",   "*template path");
    optDefaultFile(CONF_FILE);
    opt(&argc, &argv);
}

// register a new account
int reg(Options &opts)
{
    const char *action = cgiGetValue(form->cgi, "action");
    //cerr << "in reg" << endl;

    // if action is not register, print form
    if (!action || strcmp(action, "Register") != 0)
    {
        //cerr << "displaying reg form" << endl;
        form->printRegForm(opts.domain);
        form->quit();
    }

    // hold passwords
    char aPass[9], aPassConfirm[9];
    char uPass[9], uPassConfirm[9];

    // holds information for account to be created
    userAccount newAccount;
    // our action is registers so we proccess the provided values
    
    // get user input
    char fqdn[256];
    sprintf(fqdn, "%s.%s", cgiGetValue(form->cgi, "fqdn"), opts.domain);
    strncpy(newAccount.fqdn, fqdn, sizeof(newAccount.fqdn));
    strncpy(newAccount.contactName, cgiGetValue(form->cgi, "name"),
            sizeof(newAccount.contactName));
    strncpy(newAccount.contactEmail, cgiGetValue(form->cgi, "email"),
            sizeof(newAccount.contactEmail));
    strncpy(newAccount.arch, cgiGetValue(form->cgi, "arch"), 
            sizeof(newAccount.arch));
    strncpy(newAccount.os, cgiGetValue(form->cgi, "OS"), 
            sizeof(newAccount.os));

    strncpy(aPass, cgiGetValue(form->cgi, "apass"), sizeof(aPass));
    strncpy(aPassConfirm, cgiGetValue(form->cgi, "apassconfirm"), 
            sizeof(aPassConfirm));
    strncpy(uPass, cgiGetValue (form->cgi, "upass"), sizeof(uPass));
    strncpy(uPassConfirm, cgiGetValue (form->cgi, "upassconfirm"), 
            sizeof(uPassConfirm));
    
    newAccount.hostStatus = OFFLINE;
    strncpy(newAccount.ipAddress, "169.254.0.0", 
            sizeof(newAccount.ipAddress));

    if (db->hashPassword(aPass, aPassConfirm, newAccount.adminPassword) 
        == -1)
    {
        form->quit(-1, "Admin password confirmation error.");
    }
    
    if (db->hashPassword(uPass, uPassConfirm, newAccount.updatePassword) 
        == -1)
    {
        form->quit(-1, "Update password confirmation error.");
    }

    // All sanity checks are done within this function
    // create the account
    if (!db->addAccount(&newAccount, opts.domain))
    {
        form->quit(-1, "Failed to add account");
    }

    form->printRegisteredInfoPage(newAccount.userAccountId, uPass, aPass);
    form->quit ();
}

// administration tool
int admin(Options &opts)
{
    //cerr << "in admin" << endl;
    char aPass[9], uPass[9];
    char aPassConfirm[9], uPassConfirm[9];
    char adminPassword[MD5_DIGEST_LENGTH*2 + 1];

    // structure to hold account informations
    userAccount accountInfo;

    // if no action called, request authentication
    const char *action = cgiGetValue(form->cgi, "action");
    if (!action)
    {
        form->printAuthForm();
        form->quit(0, "");
    }

    // verify authentication and print forms
    else if (strcmp(action, "Authenticate") == 0)
    {
        int id = atoi(cgiGetValue(form->cgi, "hostid"));
        strncpy (aPass, cgiGetValue(form->cgi, "apass"), sizeof(aPass));

        if (db->checkPasswd(id, aPass, adminPassword) == 0)
        {
            // fill structure with account info
            db->getAcctInfo(id, &accountInfo);

            // print form with given account info
            // print account and dns forms
            form->printAcctForm(adminPassword, &accountInfo);

            // FIXME: DNS stuff will go here.
            form->printDnsForm(adminPassword, id);
            
            // show a button that deletes host entry. USE WITH CAUTION
            form->printDeleteForm(adminPassword, id);
        }
        else
        {
            form->quit(-1, "Authentication Failed");
        }
        
        form->quit(0, "");
    }

    // update account informations
    else if (strcmp(action, "Update Info") == 0)
    {
        int id;
        id = atoi(cgiGetValue(form->cgi, "hostid"));

        // FIXME: there's a segfault around ..
        char *hash = cgiGetValue(form->cgi, "md5hash");
        if (db->checkPasswd(id, NULL, hash) == -1)
        {
            form->quit(-1, "Authentication Failed");
        }

        accountInfo.userAccountId = id;
        
        // get info fom form
        strncpy(accountInfo.contactName, cgiGetValue(form->cgi, "name"), 
                sizeof(accountInfo.contactName));
        strncpy(accountInfo.contactEmail, cgiGetValue(form->cgi, "email"),
                sizeof(accountInfo.contactEmail));
        strncpy(accountInfo.arch, cgiGetValue(form->cgi, "arch"),
                sizeof(accountInfo.arch));
        strncpy(accountInfo.os, cgiGetValue(form->cgi, "OS"),
                sizeof(accountInfo.os));
        
        strncpy(aPass, cgiGetValue (form->cgi, "apass"), sizeof(aPass));
        strncpy(aPassConfirm, cgiGetValue(form->cgi, "apassconfirm"),
                sizeof(aPassConfirm));
        strncpy(uPass, cgiGetValue (form->cgi, "upass"), sizeof(uPass));
        strncpy(uPassConfirm, cgiGetValue(form->cgi, "upassconfirm"),
                sizeof(uPassConfirm));
    

        accountInfo.hostStatus = OFFLINE;
        strncpy(accountInfo.ipAddress, "169.254.0.0", 
                sizeof(accountInfo.ipAddress));

        // hash the provided passwords if they match. should not accept
        // null password.
        if (db->hashPassword(aPass, aPassConfirm,
                             accountInfo.adminPassword) == -1)
        {
            form->quit(-1, "Admin password confirmation error.");
        }
        
        if (db->hashPassword(uPass, uPassConfirm, 
                             accountInfo.updatePassword) == -1)
        {
            form->quit(-1, "Update password confirmation error.");
        }

        if (aPass[0] == '\0') // no password, don't update md5
        {
            accountInfo.adminPassword[0] = '\0';
        }

        if (uPass[0] == '\0') // no password, don't update md5
        {
            accountInfo.updatePassword[0] = '\0';
        }

        // update the database
        if (!db->updateAccount(&accountInfo, opts.domain))
        {
            form->quit(-1, "Failed to update account information.");
        }

        // if a password was provided, check it's confimation and update
        cout << "Succesfully updated database.<br>\n";
        form->quit(0, "");
    }

    // add record
    else if (strcmp(action, "Add Record") == 0)
    {
        // auth
        int id;
        id = atoi(cgiGetValue(form->cgi, "hostid"));

        char *hash = cgiGetValue(form->cgi, "md5hash");
        if (db->checkPasswd(id, NULL, hash) == -1)
        {
            form->quit(-1, "Authentication Failed");
        }
        
        if (db->addDnsRecord(id, cgiGetValue(form->cgi, "dname"),
                             (DnsRecordType) atoi(cgiGetValue(form->cgi,
                                                              "type")),
                             cgiGetValue(form->cgi, "data"), opts.domain)
            == -1)
        {
            form->quit(-1, "Failed to add record");
        }

        cout << "Added record succesfully.<br>\n";
    }

    // remove record
    else if (strcmp(action, "Remove Record") == 0)
    {
        // auth
        int id;
        id = atoi(cgiGetValue(form->cgi, "hostid"));

        // FIXME: there's a segfault around ..
        char *hash = cgiGetValue(form->cgi, "md5hash");
        if (db->checkPasswd(id, NULL, hash) == -1)
        {
            form->quit(-1, "Authentication Failed");
        }
        
        if (db->delDnsRecord(id, cgiGetValue(form->cgi, "dname"),
                             (DnsRecordType) atoi(cgiGetValue(form->cgi,
                                                              "type")),
                             cgiGetValue(form->cgi, "data")) == -1)
        {
            form->quit(-1, "Failed to add record");
        }

        cout << "Removed record succesfully.<br>\n";
    }
    
    else if (strcmp(action, "Delete Host") == 0)
    {
        int id;
        id = atoi (cgiGetValue(form->cgi, "hostid"));

        char *hash = cgiGetValue(form->cgi, "md5hash");
        if (db->checkPasswd(id, NULL, hash) == -1)
        {
            form->quit (-1, "Authentication Failed");
        }
        
        if (db->removeAccount(id) == -1)
        {
            form->quit (-1, "Failed to remove account");
        }
        
        cout << "Host Entry completely deleted from databases.<br>\n";
    }

    // We passed through the CGI with success
    form->quit (0, "");
}
