/* -*- Mode: C++; c-file-style: "stroustrup"; indent-tabs-mode: nil -*- */
/*
 * DdtManager.cc
 *   
 * $Id: DdtManager.cc,v 1.12 2003/04/24 04:44:56 benoit Exp $
 *
 * Copyright (c) 2000-2001 Remi Lefebvre <remi@dhis.net>
 * Copyright (c) 2000 Luca Filipozzi <lfilipoz@dhis.net>
 *
 * DDT comes with ABSOLUTELY NO WARRANTY and is licenced under the
 * GNU General Public License (version 2 or later). This license
 * can be retrieved from http://www.gnu.org/copyleft/gnu.html.
 *
 */

#include <iostream>
#include <sstream>
#include <string>
#include <map>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "DdtManager.h"
#include "Exception.h"
#include "Db.h"
#include "Dns.h"

DdtManager::DdtManager (Logger *log, Db *db, Dns *dns)
{
    this->log = log;
    this->db = db;
    this->dns = dns;
}

DdtManager::~DdtManager ()
{
    // intentionally empty
}


int DdtManager::addUserAccount(UserAccount *account)
{
    int id;
    
    id = db->addUserAccount(account);
    dns->addDnsRecord(id, account->fqdn, A, "169.254.0.0");

    return id;
}

void DdtManager::delUserAccount (int id)
{
    char fqdn[256];
    fetchAccountInfo(id, "fqdn", fqdn, sizeof(fqdn));

//    onlineAccounts.erase(id);   // for 0.6
    delDnsRecords(id);
    delDnsRecord(id, fqdn, A, "");
    db->delUserAccount(id);
}

void DdtManager::logonUserAccount (int id)
{
    // TBD
}

void DdtManager::logoffUserAccount(int id)
{
    // remove accountid from runtime list
//    onlineAccounts.erase(userAccountId); // for 0.6
    char fqdn[256];

    fetchAccountInfo(id, "fqdn", fqdn, sizeof(fqdn));

    // mark offline in the database
    db->modUserAccount(id, "ipaddress", "169.254.0.0");

    // mark offline in the dns
    dns->updateDnsRecord(id, fqdn, A, "169.254.0.0");
}

void DdtManager::updateIPAddress(int id, unsigned long ip)
{
//    runtimeInfo a;
//    a.addr = ip;

    // insert the account in the runtime list
//    onlineAccounts[userAccountId] = a;  // for 0.6

    struct in_addr in;
    in.s_addr = ip;

    char fqdn[256];

    // update database
    db->modUserAccount(id, "ipaddress", inet_ntoa(in));

    updateTimeStamp(id);

    fetchAccountInfo(id, "fqdn", fqdn, sizeof(fqdn));

    // update dns
    dns->updateDnsRecord(id, fqdn, A, inet_ntoa(in));
}

void DdtManager::updateTimeStamp(int id)
{
    db->modUserAccount(id, "lastaccess", "now");
}

void DdtManager::addDnsRecord(int id, const char *dname, DnsRecordType 
                              type, const char *data)
{
    /* add record to database and dns */
    db->addDnsRecord(id, dname, type, data);

    /* NOTE: The database is the authority with respect to name clashes.
     *       if a name is available in the database, the dns must be synced
     *       to it no matter what. Therefore, we register in the database
     *       first, if it succeeds, we sync the dns.
     */

    // FIXME: add the necessary checks to handle errors
    dns->addDnsRecord(id, dname, type, data);
}

void DdtManager::delDnsRecord(int id, const char *dname, DnsRecordType type,
                              const char *data)
{
    /* FIXME: we must check the dns record was really removed before
     * removing db record */
    dns->delDnsRecord(id, dname, type, data);
    db->delDnsRecord(id, dname, type, data);
}
    
void DdtManager::delDnsRecords(int id)
{
    std::vector<DnsRecord> dnsRecordList;

    // get the user's dns records
    listDnsRecords(id, dnsRecordList);

    int nbRecords = dnsRecordList.size();

    int no;
    DnsRecord record;
    for (no = 0; no < nbRecords; no++)
    {
        record = dnsRecordList[no];
        dns->delDnsRecord(no, record.dname, (DnsRecordType)(record.type), 
                          record.data);
    }    
    db->delDnsRecords(id);
}

void DdtManager::listDnsRecords(int id, std::vector<DnsRecord> &dnsRecordList)
{
    db->listDnsRecords(id, dnsRecordList);
}
