package com.tildemh.debbuggtk;

import com.tildemh.debbug.*;
import org.gnu.gtk.*;
import java.util.LinkedList;
import org.gnu.glade.*;
import java.io.IOException;
import java.io.FileNotFoundException;

/**
 * Widget for editing control information of bugs (everything which can be done
 * with emails to control@b.d.o)
 * This is used in the BugEdit form.
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: ControlEdit.java,v 1.13 2004/01/24 10:13:42 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class ControlEdit extends VBox{

	private Bug bug;
	
	private CheckButton helpTag;
	private CheckButton patchTag;
	private CheckButton securityTag;
	private CheckButton upstreamTag;
	private CheckButton potatoTag;
	private CheckButton woodyTag;
	private CheckButton sargeTag;
	private CheckButton sidTag;
	private CheckButton experimentalTag;
	private CheckButton fixedTag;
	private CheckButton pendingTag;
	private CheckButton unreproducibleTag;
	
	private CheckButton d_iTag;
	private CheckButton confirmedTag;
	private CheckButton ipv6Tag;
	private CheckButton lfsTag;
	private CheckButton wontfixTag;
	private CheckButton moreinfoTag;
	private CheckButton fixedInExperimentalTag;
	private CheckButton fixedUpstreamTag;

	 
	private Entry title;
	private Entry merge;
	private Entry reassign;
	private Entry submitter;
	private Entry forwarded;
	private OptionMenu severity;

	private LibGlade glade;
	
	/**
	 * @param bug The bug to be edited
	 */
	public ControlEdit(Bug bug){
		super(false, 0);
		this.bug = bug;
		
		String filename = DebbugGtk.GLADE_XML_DIR+"controledit.glade";
		try {
			glade = new LibGlade(filename, this, "controlEditWidget");
		} catch (GladeXMLException e) {
			System.err.println("Error parsing glade XML file." + e);
		} catch (FileNotFoundException e) {
			System.err.println("Glade XML file not found.");
		} catch (IOException e) {
			System.err.println("Error reading glade XML file.");
		}
		packStart( glade.getWidget("controlEditWidget"), true, true, 0 );
		
		showAll();

		helpTag = (CheckButton) glade.getWidget("helpTag");
		patchTag = (CheckButton) glade.getWidget("patchTag");
		securityTag = (CheckButton) glade.getWidget("securityTag");
		upstreamTag = (CheckButton) glade.getWidget("upstreamTag");
		potatoTag = (CheckButton) glade.getWidget("potatoTag");
		woodyTag = (CheckButton) glade.getWidget("woodyTag");
		sargeTag = (CheckButton) glade.getWidget("sargeTag");
		sidTag = (CheckButton) glade.getWidget("sidTag");
		experimentalTag = (CheckButton) glade.getWidget("experimentalTag");
		fixedTag = (CheckButton) glade.getWidget("fixedTag");
		pendingTag = (CheckButton) glade.getWidget("pendingTag");
		unreproducibleTag = (CheckButton) glade.getWidget("unreproducibleTag");
	
		d_iTag = (CheckButton) glade.getWidget("d_iTag");
		confirmedTag = (CheckButton) glade.getWidget("confirmedTag");
		ipv6Tag = (CheckButton) glade.getWidget("ipv6Tag");
		lfsTag = (CheckButton) glade.getWidget("lfsTag");
		wontfixTag = (CheckButton) glade.getWidget("wontfixTag");
		moreinfoTag = (CheckButton) glade.getWidget("moreinfoTag");
		fixedInExperimentalTag = (CheckButton) glade.getWidget("fixedInExperimentalTag");
		fixedUpstreamTag = (CheckButton) glade.getWidget("fixedUpstreamTag");

	 
		title = (Entry) glade.getWidget("title");
		merge = (Entry) glade.getWidget("merge");
		reassign = (Entry) glade.getWidget("reassign");
		submitter = (Entry) glade.getWidget("submitter");
		forwarded = (Entry) glade.getWidget("forwarded");
		severity = (OptionMenu) glade.getWidget("severity");


		setBug();
	}

	/**
	 * Adds bug details to form
	 */
	private void setBug(){

		//TODO: If the bug has not been checked with the server for a long time,
		// make this very clear.
		helpTag.setState( bug.getTagHelp() );
		patchTag.setState( bug.getTagPatch() );
		securityTag.setState( bug.getTagSecurity() );
		upstreamTag.setState( bug.getTagUpstream() );
		potatoTag.setState( bug.getTagPotato() );
		woodyTag.setState( bug.getTagWoody() );
		sargeTag.setState( bug.getTagSarge() );
		sidTag.setState( bug.getTagSid() );
		experimentalTag.setState( bug.getTagExperimental() );
		fixedTag.setState( bug.getTagFixed() );
		pendingTag.setState( bug.getTagPending() );
		unreproducibleTag.setState( bug.getTagUnreproducible() );
	
		d_iTag.setState( bug.getTagD_I() );
		confirmedTag.setState( bug.getTagConfirmed() );
		ipv6Tag.setState( bug.getTagIPV6() );
		lfsTag.setState( bug.getTagLFS() );
		wontfixTag.setState( bug.getTagWontfix() );
		moreinfoTag.setState( bug.getTagMoreinfo() );
		fixedInExperimentalTag.setState( bug.getTagFixedInExperimental() );
		fixedUpstreamTag.setState( bug.getTagFixedUpstream() );
	
		title.setText(bug.getTitle());
		forwarded.setText( bug.getForwarded() );

		int hist = 0;
		Severity sev = bug.getSeverity();
		if (sev.equals( Severity.CRITICAL )){
			hist = 0;
		}else if (sev.equals( Severity.GRAVE )){
			hist = 1;
		}else if (sev.equals( Severity.SERIOUS )){
			hist = 2;
		}else if (sev.equals( Severity.IMPORTANT )){
			hist = 4;
		}else if (sev.equals( Severity.NORMAL )){
			hist = 5;
		}else if (sev.equals( Severity.MINOR )){
			hist = 6;
		}else if (sev.equals( Severity.WISHLIST )){
			hist = 8;
		}
		severity.setHistory( hist );
		
		String merged = "";
		int[] m = bug.getMerged();
		for(int i = 0; i < m.length; i++){
			if (i > 0){
				merged += " " + Integer.toString(m[i]);
			}else{
				merged += Integer.toString(m[i]);
			}
		}
		merge.setText( merged );
		
		submitter.setText( bug.getSubmitter() );
	}

	/**
	 * Produces a string representing the changes by the form on this page.
	 */
	public String getControlMsg(){
		System.out.println("getControlMsg");
		String mail = "";
		
		boolean needControl = false;  // should we send to  control@bugs?
		
		LinkedList addTags = new LinkedList();
		LinkedList removeTags = new LinkedList();
		if (helpTag.getState() != bug.getTagHelp()){
			if (helpTag.getState())  addTags.add("help"); else removeTags.add("help");
		}
		if (patchTag.getState() != bug.getTagPatch()){
			if (patchTag.getState()) addTags.add("patch"); else  removeTags.add("patch");
		}
		if (securityTag.getState() != bug.getTagSecurity()){
			if (securityTag.getState()) addTags.add("security"); else  removeTags.add("security");
		}
		if (upstreamTag.getState() != bug.getTagUpstream()){
			if (upstreamTag.getState()) addTags.add("upstream"); else  removeTags.add("upstream");
		}
		if (potatoTag.getState() != bug.getTagPotato()){
			if (potatoTag.getState()) addTags.add("potato"); else  removeTags.add("potato");
		}
		if (woodyTag.getState() != bug.getTagWoody()){
			if (woodyTag.getState()) addTags.add("woody"); else  removeTags.add("woody");
		}
		if (sargeTag.getState() != bug.getTagSarge()){
			if (sargeTag.getState()) addTags.add("sarge"); else  removeTags.add("sarge");
		}
		if (sidTag.getState() != bug.getTagSid()){
			if (sidTag.getState()) addTags.add("sid"); else  removeTags.add("sid");
		}
		if (experimentalTag.getState() != bug.getTagExperimental()){
			if (experimentalTag.getState()) addTags.add("experimental"); else  removeTags.add("experimental");
		}
		if (fixedTag.getState() != bug.getTagFixed()){
			if (fixedTag.getState()) addTags.add("fixed"); else  removeTags.add("fixed");
		}
		if (fixedInExperimentalTag.getState() != bug.getTagFixedInExperimental()){
			if (fixedInExperimentalTag.getState()) addTags.add("fixed-in-experimental"); else  removeTags.add("fixed-in-experimental");
		}
		if (pendingTag.getState() != bug.getTagPending()){
			if (pendingTag.getState()) addTags.add("pending"); else  removeTags.add("pending");
		}
		if (unreproducibleTag.getState() != bug.getTagUnreproducible()){
			if (unreproducibleTag.getState()) addTags.add("unreproducible"); else  removeTags.add("unreproducible");
		}
		if (d_iTag.getState() != bug.getTagD_I()){
			if (d_iTag.getState()) addTags.add("d-i"); else  removeTags.add("d-i");
		}
		if (confirmedTag.getState() != bug.getTagConfirmed()){
			if (confirmedTag.getState()) addTags.add("confirmed"); else  removeTags.add("confirmed");
		}
		if (ipv6Tag.getState() != bug.getTagIPV6()){
			if (ipv6Tag.getState()) addTags.add("ipv6"); else  removeTags.add("ipv6");
		}
		if (lfsTag.getState() != bug.getTagLFS()){
			if (lfsTag.getState()) addTags.add("lfs"); else  removeTags.add("lfs");
		}
		if (wontfixTag.getState() != bug.getTagWontfix()){
			if (wontfixTag.getState()) addTags.add("wontfix"); else  removeTags.add("wontfix");
		}
		if (moreinfoTag.getState() != bug.getTagMoreinfo()){
			if (moreinfoTag.getState()) addTags.add("moreinfo"); else  removeTags.add("moreinfo");
		}
		if (fixedUpstreamTag.getState() != bug.getTagFixedUpstream()){
			if (fixedUpstreamTag.getState()) addTags.add("fixed-upstream"); else  removeTags.add("fixed-upstream");
		}
	
		if (addTags.size() > 0){
			needControl = true;
			mail += "tags "+bug.getNumber()+" + ";
			for(int i = 0; i < addTags.size(); i++){
				mail += addTags.get(i) + " ";
			}
			mail += "\n";
		}
		if (removeTags.size() > 0){
			needControl = true;
			mail += "tags "+bug.getNumber()+" - ";
			for(int i = 0; i < removeTags.size(); i++)
				mail += removeTags.get(i) + " ";
			mail += "\n";
		}

		if (! title.getText().equals( bug.getTitle() ) ){
			needControl = true;
			mail += "retitle "+bug.getNumber() + " " + title.getText() + "\n";
		}

		if (! forwarded.getText().equals( bug.getForwarded() ) ){
			needControl = true;
			mail += "forwarded "+bug.getNumber() + " " + forwarded.getText() + "\n";
		}

		if (! merge.getText().equals("") ){
			needControl = true;
			mail += "merge " + bug.getNumber() + " " + merge.getText() + "\n";
		}

		int hist = severity.getHistory();
		int sevValue = -1;
		switch (hist) {
			case 0: sevValue = 0; break;
			case 1: sevValue = 1; break;
			case 2: sevValue = 2; break;
			case 4: sevValue = 3; break;
			case 5: sevValue = 5; break;
			case 6: sevValue = 6; break;
			case 8: sevValue = 7; break;
		};
		if (sevValue!= bug.getSeverity().getValue()){
			needControl = true;			
			mail += "severity " + bug.getNumber() + " " + Severity.fromValue(sevValue).toString() +"\n";
		}

		if (! submitter.getText().equals( bug.getSubmitter() ) ){
			needControl = true;
			mail += "submitter "+bug.getNumber() + " " + submitter.getText() + "\n";
		}

		if (! reassign.getText().equals( "" ) ){
			needControl = true;
			mail += "reassign "+bug.getNumber() + " " + reassign.getText() + "\n";
		}

		if (!needControl){
			return null;
		}else{
			return mail;
		}
	}

	private void cancel(){
		hide();
	}
}
