package com.tildemh.debbug;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OptionalDataException;
import java.lang.ref.SoftReference;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.WeakHashMap;

/**
 * Methods for managing a local cache of bug reports and package listings. 
 *
 * <p>This is currently being rewritten as a three level cache - disk cache,
 * memory cache for full reports and memory cache for bug headers. 
 *
 * <pre>
 * todo:
 * - remove package -- remove all package bugs.
 * - remove methods - memory cache?? * 
 * </pre>
 * 
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: Cache.java,v 1.32 2004/03/15 15:53:03 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class Cache{

	private static final boolean DEBUG=false;
	private String dir;					// Cache directory
	private String virtualDir;
	
	private boolean noRead = false;		// determines whether cache reads are permitted
	private boolean noWrite = false;	// determines whether cache writes are permitted

	// Memory cache objects
	private WeakHashMap cachedBugs;		// contains the memory cached bugs. Will be GC when no other references remain
	private LinkedList bugReportCacheQueue;	
	private static int bugReportCacheSize = 20;


	private CacheWriter writer;

	/**
	 * Creates a new Cache Object
	 */
	private Cache(){
		init();
	}
	
	private static Cache instance = null;
	
	public static  Cache getInstance(){
		if (instance == null) instance = new Cache();
		return instance;
	}

	/**
	 * Sets up the cache
	 */
	private void init(){
		dir = System.getProperty("user.home") + "/.debbug-java/Cache/";
		virtualDir = System.getProperty("user.home") + "/.debbug-java/VirtualListings/";
		setDir(virtualDir);
		setDir(dir);
		cachedBugs = new WeakHashMap();
		bugReportCacheQueue = new LinkedList();	
		writer = new CacheWriter(dir );
		Thread t = new Thread( writer );
		t.setDaemon(true);
		t.setName("Lazy cache writer");
		t.start();
	}

	
	/**
	 * Overrides the default location for the cache directory. If there are
	 * problems with this (e.g. it doesn't exist, it isn't writable, etc. Then
	 * the cache will continue, but with limited functionality.
	 * <p>The default cache location is <code>System.getProperty("user.home") +
	 * "/.debbug-java";</code>
	 * @param dir Directory to use for the cache content.
	 */
	public void setDir(String dir){
		File f = new File(dir);
		if (! f.exists()){
			f.mkdirs();
		}
		if (! f.canWrite()){
			System.err.println("debbug-java: Do not have write permissions for cache directory "+dir);
			noWrite = true;
		}else{
			noWrite = false;
		}
		if (!f.canRead()){ 
			System.err.println("debbug-java: Do not have read permission for cache directory "+dir);
			noRead = true;
		}else{
			noRead = false;
		}
	}

	/**
	 * Removes a bug from the cache
	 */
	public void remove(Bug bug){
		if (noWrite) return;
		File f = new File( dir + Integer.toString(bug.getNumber()));
		if (!f.exists()) return;
		if (!f.delete()){
			System.out.println("ERROR: unable to delete cache file: "+dir + Integer.toString(bug.getNumber()));
		}
	}

	/**
	 * Removes a listing from the cache. 
	 * @param list The listing to be removed
	 * @param rmBugs If true, all bugs for the listing will also be removed.
	 */
	public void remove(Listing list){
		if (noWrite) return;
		File f = null;
		if (list.getType().equals(ListingType.VIRTUAL))
			f = new File( virtualDir + list.getName() );
		else
			f = new File(dir + list.getType().getInfo() + list.getName() + ( list.getType().getIncludeClosed() ? "-non-archived" : "-open" ) );
		try{
		if (DEBUG) System.out.println("REMOVING LISTING "+f.toURL()+ "   |"+list.getType()+"|   |"+list.getName()+"|   ");
		}catch (Exception e){}
		if (!f.exists()) return;
		if (!f.delete()){
			System.out.println("ERROR: unable to delete cache file: "+  dir + list.getType().getInfo() + list.getName()  + ( list.getType().getIncludeClosed() ? "-non-archived" : "-open" ) );
		}
		// FIXME: remove bugs related to listing.
		//		LinkedList bugs = (LinkedList) list.getBugNumbers().clone();
		//		while (bugs.size() >0){
		//			Integer number =  bugs.removeFirst();
		//			remove(bug);
		//		}
	}

	/**
	 * Returns a bug report from the cache
	 *
	 * @param number The number referencing the bug to be returned
	 * @param needComments If false, the returned bug may be returned without
	 * its comments attached. Bugs are kept for longer without attached
	 * comments
	 * @throws CacheMiss if it was unable to get the bug report.
	 * @throws CacheError when other problems prevent the bug report from being
	 * created.
	 */
	public Bug getBug(int number, boolean needComments) throws CacheMiss, CacheError{
		if (DEBUG) System.out.println( System.currentTimeMillis() + "  starting cache.getBug("+number+", "+needComments+")");
		// first see if the bug is in the menory cache.
		Bug bug = null;
		bug = (Bug) cachedBugs.get( new Integer(number) );
		
		if (bug == null)	if (DEBUG) System.out.println(System.currentTimeMillis() + "  Bug not in mem cache");

		if (bug != null){
			addBugMemCache(bug);	// add it to the top of the queue
			if (DEBUG) System.out.println(System.currentTimeMillis() + "returning bug report from memory cache");
			return bug;
		}

		// Now try the disk cache
		if(noRead) throw new CacheMiss("NoRead set - unable to get bug");

		File f = new File(dir + Integer.toString(number));
		if (DEBUG) System.out.println("f.exists("+f.exists());
		if (!f.exists()) throw new CacheMiss("Bug report doesn't exist in cache");
		if (DEBUG) System.out.println(System.currentTimeMillis() + "  reading file from "+dir + Integer.toString( number ) );

		try{
			if (DEBUG) System.out.println("Starting to load bug from file");
			bug = BugFile.loadBug(f);
			if (DEBUG) System.out.println("done loading bug from file");
		}catch(OptionalDataException e){
			if (!noWrite) f.delete(); //fixme: check return
			throw new CacheError("File found - OptionalDataException: "+e);
		}catch(IOException e){
			if (!noWrite) f.delete(); //fixme: check return
			e.printStackTrace();
			throw new CacheError("File Found - IOException: "+e);
		}
		if (DEBUG) System.out.println(System.currentTimeMillis() + "   successfully read  bug #"+bug.getNumber());
		addBugMemCache(bug);

		return bug;
	}

	/**
	 * Stores a bug report in cache
	 * @param bug The bug report to be stored.
	 */
	public void store(Bug bug) {
		// FIXME: What if bug already exist in memory cache???
		// FIXME: Which bug report will getMemCache return??
		addBugMemCache(bug);
		if (noWrite) return;
		writer.addItem(bug);
	}
	/**
	 * Returns a package listing from cache
	 * @throws CacheMiss if this was not possible
	 */
	public Listing getListing(ListingStub listing) throws CacheMiss, CacheError{
		if(noRead) throw new CacheMiss("NoRead set - unable to get bug");
		ListingType type = listing.getType();
		String name = listing.getName();
		if (listingCache.containsKey(type.toString()+name)){
			return (Listing) listingCache.get( type.toString()+name );
		}

		File f = null;
		if (type.equals(ListingType.VIRTUAL))
			f = new File( virtualDir + name );
		else
			f = new File(dir + type.getInfo() + name + ( type.getIncludeClosed() ? "-non-archived" : "-open" ) );
	
		if (DEBUG) System.out.println("DEBUG: reading file from " + f.toString() );
		if (!f.exists()) throw new CacheMiss("Package listing doesn't exist in cache");

		Listing list = null;
		try{
			ObjectInputStream in = new ObjectInputStream( new FileInputStream(f) );
			Object o = in.readObject();
			in.close();
			if (! (o instanceof Listing)){
				if (!noWrite){
					System.err.println("Corrupt cache entry - deleting "+f.toString());
					f.delete(); //fixme: check return
				}
				throw new CacheError("File found but Object not Listing");
			}
			list = (Listing) o;
		}catch(OptionalDataException e){
			if (!noWrite) f.delete(); //fixme: check return
			throw new CacheError("File found - OptionalDataException: "+e);
		}catch(ClassNotFoundException e){
			if (!noWrite) f.delete(); //fixme: check return
			throw new CacheError("File Found - ClassNotFoundException: "+e);
		}catch(IOException e){
			if (!noWrite) f.delete(); //fixme: check return
			e.printStackTrace();
			throw new CacheError("File Found - IOException: "+e);
		}
		if (DEBUG) System.out.println("DEBUG: Cache: getListing: successfully read  Listing #"+list.getName());

		if (!listingCache.containsKey(list.getType().toString()+list.getName())){
			if (DEBUG) System.out.println("Adding "+list.getType().toString()+list.getName()+" to listingCache");
			listingCache.put( list.getType().toString()+list.getName(), list);
		}
		return list;
	}

	/**
	 * Returns true if the virtual listing already exisits
	 */
	public boolean virtualListingExists(String name){
		File f = new File( virtualDir + name );
		return f.exists();
	}

	private Hashtable listingCache = new Hashtable();

	/**
	 * Stores a package listing in cache
	 */
	public void store(Listing list){
		
		if (!listingCache.containsKey(list.getType().toString()+list.getName())){
			if (DEBUG) System.out.println("Adding "+list.getType().toString()+list.getName()+" to listingCache");
			listingCache.put( list.getType().toString()+list.getName(), list);
		}
		/*
		 * TODO (big)
		 * If package already exists in cache, look for bugs which are no longer
		 * part of the package and delete them from the cache.
		 */

		if (noWrite) return;
		if (!list.getComplete()) return;
		File f = null;
		if (list.getType().equals(ListingType.VIRTUAL))
			f = new File( virtualDir + list.getName() );
		else
			f = new File(dir + list.getType().getInfo() + list.getName()  + ( list.getType().getIncludeClosed() ? "-non-archived" : "-open" ) );
	
		if (DEBUG) System.out.println("DEBUG: writing file to "+ f.toString() );
		
		try{
			ObjectOutputStream out = new ObjectOutputStream( new FileOutputStream(f) );

			out.writeObject(list);
			out.flush();
			out.close();
		}catch(IOException e){
			e.printStackTrace();

		}
	}

	/**
	 * Adds a bug to the memory cache (or updates its location in the memory
	 * cache if already present
	 */
	private void addBugMemCache(Bug bug){
		if (DEBUG) System.out.println(System.currentTimeMillis() + "  adding bug to mem cache "+bug.getNumber()+"***");
		bugReportCacheQueue.addFirst( new SoftReference( bug ) );
			// remove elements if the cache is getting full.
			while ( bugReportCacheQueue.size() > bugReportCacheSize ){
				bugReportCacheQueue.removeLast();
			}
		cachedBugs.put(new Integer( bug.getNumber() ), bug);
	}
		
}
