package com.tildemh.debbuggtk;

import com.tildemh.debbug.*;
import org.gnu.gtk.*;
import org.gnu.glade.*;
import org.gnu.gnome.*;
import java.io.*;

/**
 * Simple program to display a bug report
 *
 * Now using libglade
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: BugViewer.java,v 1.29 2004/04/21 20:20:05 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class BugViewer {
	
	/**
	 * Create a new window
	 */
	public BugViewer(){
		makeWindow();
	}
	public BugViewer(Bug bug){
		makeWindow();
		setBug(bug);
	}

	private org.gnu.gtk.Entry number;
	private BugReport report;
	private BTS bts;
	private App app;
	private AppBar appBar;

	private void makeWindow(){
		DebbugGtk.newWindow();
		WindowIcon.setDefaultIcon(DebbugGtk.ICON);
		
		String filename = DebbugGtk.GLADE_XML_DIR+"bugviewer.glade";
		LibGlade glade = null;
		try {
			glade = new LibGlade(filename, this, null);
			number = (org.gnu.gtk.Entry) glade.getWidget("bugEntry");
			app = (App) glade.getWidget("bugviewer");
			app.addListener( DebbugGtk.lifeCycleListener );
			app.setTitle("BugViewer "+DebbugGtk.VERSION);
			VBox vb = (VBox) glade.getWidget("mainVB");
			appBar = (AppBar) glade.getWidget("bugViewerAppBar");
			report = new BugReport(app, appBar );
			vb.packStart( report, true, true, 0);
			app.showAll();
			number.grabFocus();
			glade.getWidget("setBug").grabDefault();
		} catch (GladeXMLException e) {
			System.err.println("Error parsing glade XML file." + e);
		} catch (FileNotFoundException e) {
			System.err.println("Glade XML file not found.");
		} catch (IOException e) {
			System.err.println("Error reading glade XML file.");
		}
	}

	public void setBug(String bugNo){
		try{
			Bug bug = loadBug(bugNo);
			app.setTitle("Bug #" + bug.getNumber() + ": "+bug.getTitle()+" - BugViewer "+DebbugGtk.VERSION);
			report.setBug(bug);
		}catch(Exception e){
			MessageDialog md = new MessageDialog(app, DialogFlags.MODAL.or(DialogFlags.DESTROY_WITH_PARENT), MessageType.ERROR, ButtonsType.CLOSE, 
				"<span size=\"larger\" weight=\"bold\">Unable to set bug report</span>\n"
				+e, true);
			md.run();
			md.destroy();

		}
	}
	public void setBug( Bug bug){
		app.setTitle("Bug #" + bug.getNumber() + ": "+bug.getTitle()+" - BugViewer "+DebbugGtk.VERSION);
			report.setBug(bug);

	}
	
	private Bug loadBug(String toShow) throws Exception{
		Bug bug;
		File f = new File(toShow);
		if (f.exists()){
			StringBuffer htmlPage = new StringBuffer();
			FileReader reader = new FileReader(toShow);
			int ch =0;
			while ((ch = reader.read()) != -1){
				htmlPage.append((char)ch);
			}
		
			bug = null; // TODO WWWBugParser.makeBug(htmlPage.toString(), null);
		}else{
			BTS bts = BTS.getInstance();
			bug = bts.getBug(new Integer(toShow));
		}
		return bug;
	}
////////////////////////////////////////////////////////////////////////////////
// Events defined within glade file
//
	public void newWindow(){
		new BugViewer();
	}
	public void loadBug(){
		String bug = number.getText();
		number.setText("");
		setBug(bug);
	}
	public void refresh(){
		report.refresh();
	}
	public void addComment(){
		report.comment();
	}
	public void editControl(){
		report.control();
	}
	public void sendPending(){
		ControlQueue.getInstance().generateMail();
	}
	public void quit(){
		app.hide();
		DebbugGtk.closeWindow();
	}
	public void about(){
		DebbugGtk.helpAbout();
	}
	public void bugWatcher(){
		new BugWatcher();
	}
	public void bugLister(){
		new BugLister();
	}
	public void bugReporter(){
		new BugReporter();
	}

////////////////////////////////////////////////////////////////////////////////

	/**
	 * shows a bug report 
	 * @param toshow bug number to download, or filename.
	 */
	public BugViewer(String toShow){
		makeWindow();
		setBug( toShow );
	}

	public static void main(String[] args){
		Thread.currentThread().setPriority( Thread.MAX_PRIORITY );

		Program.initGnomeUI("debbuggtk", DebbugGtk.VERSION, args);

		if (args.length != 1){
			new BugViewer();
		}else{
			new BugViewer(args[0]);
		}
		Gtk.main();
	}
	public void help(){
		try{
			Help.display( "debbuggtk", "contents" );
		}catch( HelpException e ){
			e.printStackTrace();
			throw new RuntimeException( e.toString() );
		}
	}
}
