package com.tildemh.debbuggtk;

import com.tildemh.debbug.*;
import org.gnu.gtk.*;
import org.gnu.gtk.event.*;
import org.gnu.gnome.*;
import org.gnu.glade.*;
import java.util.*;
import java.io.*;

/**
 * Application for monitoring bugs in the Debian Bug Tracking System (bts).
 *
 * <p>This will eventually work in a similar way to an email on nntp client -
 * you will select a number of packages to monitor (set up email acount /
 * subscrie to newsgroup); then a list of bugs will be downloaded. Whenever you
 * synchronise with the server, updated items will be highlighted. Checking what
 * has changed is quite difficult though, so may take a while to implement. 
 *
 * <p>The interaction with the bts is done through the {@link
 * com.tildemh.jdebbug} package. Much of the functionality of viewing package
 * bug listings and bug reports is done elsewhere in the {@link
 * com.tildemh.debbuggtk} package. These are also used in a number of other
 * applications, such as {@link com.tildemh.debbuggtk.BugViewer} 
 *
 * <p>The layout of the application is quite static, following this convention:
 * <pre>
 * +--------------------------------------------------------------------+
 * | List of        |                                                   |
 * | Watched        |                   Bug Listing                     |
 * | packages and   |                 for selected package              |
 * | bugs           |                                                   |
 * +----------------+---------------------------------------------------+
 * |                                                                    |
 * |                     Selected bug report                            |
 * |                                                                    |
 * +--------------------------------------------------------------------+
 * </pre>
 *
 * <h3>FIXME</h3>
 * <p>All help is appreciated - feel free to email patches to me!
 * <ul>
 * <li>make this a gnome application - unfortunately, problems with java-gnome
 * prevent this at the moment. 
 * <li>(Requires gnome) Add toolbar and menubar.
 * <li>Highlighting of changed items.
 * </ul>
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: BugWatcher.java,v 1.74 2004/04/21 20:20:05 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class BugWatcher implements ListingReportListener, ControlQueue.Listener{

	private App app;
	private AppBar appBar;

	private BugReport bugReport;					// Bug report widget (bottom of screen)
	private ListingReport listingReport;				// Bug listing widget (top right)
	private WatchedReport watchedReport;

	private WatchedList watchList;				// List of bugs/listings being followed
	
	private TreeView pkgView;					// View for watched items
	private ListStore pkgStore;					// Associated store.

	private LinkedList store = new LinkedList();

	private LibGlade glade;
	private ToolButton btnSendControl;
	private MenuItem menuSendControl;

	private ToolButton btnRefresh;
	private MenuItem menuRefresh;
	private Menu popupmenu;
	
	/**
	 * Constructs a new BugWatcher
	 */
	public BugWatcher(){
		DebbugGtk.newWindow();
		WindowIcon.setDefaultIcon(DebbugGtk.ICON);
		
		String filename = DebbugGtk.GLADE_XML_DIR+"bugwatcher.glade";
		try {
			glade = new LibGlade(filename, this, null);
			app = (App) glade.getWidget("bugwatcher");
			app.setTitle("BugWatcher "+DebbugGtk.VERSION);
		} catch (GladeXMLException e) {
			System.err.println("Error parsing glade XML file." + e);
		} catch (FileNotFoundException e) {
			System.err.println("Glade XML file not found.");
		} catch (IOException e) {
			System.err.println("Error reading glade XML file.");
		}

		appBar = (AppBar) glade.getWidget("appBar");
		
		app.addListener( DebbugGtk.lifeCycleListener );
		app.maximize();
		
		boolean nonEmpty = ! ControlQueue.getInstance().isEmpty();		
		btnSendControl = (ToolButton) glade.getWidget("btnSendControl");
		menuSendControl = (MenuItem) glade.getWidget("sendControlMenu");
		btnSendControl.setSensitive(nonEmpty);
		menuSendControl.setSensitive(nonEmpty);
		ControlQueue.getInstance().addListener( (ControlQueue.Listener) this );
		
		glade.getWidget("commentbtn").setSensitive(false);
		glade.getWidget("controlbtn").setSensitive(false);
		btnRefresh = (ToolButton) glade.getWidget("btnRefresh");
		menuRefresh = (MenuItem) glade.getWidget("menuRefresh");

		app.showAll();
		while( Gtk.eventsPending() ){ Gtk.mainIteration(); }
        
		HPaned hp = (HPaned) glade.getWidget("hPaned");
		hp.setPosition(300);
		listingReport = new ListingReport( this, app);		// Widget containing list of bugs for a particular package/maintainer, etc.
		hp.pack2(listingReport, true, true);
		app.showAll();
		while( Gtk.eventsPending() ){ Gtk.mainIteration(); }

		bugReport = new BugReport( app, appBar);						// Widget displaying a single bug report.
		VPaned vp = (VPaned) glade.getWidget("vPaned");
		vp.pack2(bugReport, true, true);
		vp.setPosition(300);
		
		app.showAll();
		while( Gtk.eventsPending() ){ Gtk.mainIteration(); }

		
		watchedReport = new WatchedReport( listingReport, bugReport, app, this);
		hp.pack1(watchedReport, false, true);

		app.showAll();
		while( Gtk.eventsPending() ){ Gtk.mainIteration(); }

		watchedReport.loadItems();

		listingReport.setStatusListener( watchedReport );
		app.showAll();
	}

		
	/**
	 * Event handler for changes to the selected bug in the listing report
	 */
	public void listingSelectionChanged(com.tildemh.debbug.Bug bug){
		bugReport.setBug( bug );
		glade.getWidget("commentbtn").setSensitive(true);
		glade.getWidget("controlbtn").setSensitive(true);
		glade.getWidget("open_in_bugviewer1").setSensitive(true);
		glade.getWidget("update1").setSensitive(true);
		glade.getWidget("add_comment1").setSensitive(true);
		glade.getWidget("edit_headers1").setSensitive(true);
	}

	public void listingChanged( ListingStub listing ){
		ListingType type = listing.getType();
		String open = type.getIncludeClosed() ? "(non-archived) " : "";
		app.setTitle( listing.getName() +" " + type.getTitle() + " Bugs " + open +"- BugWatcher "+DebbugGtk.VERSION);
	}
			
	public void listingReportStateChanged( ListingState state ){
		appBar.clearStack();
		appBar.pushStack( state.toString() );
		if (state.equals(ListingState.DLLISTING)){
			appBar.getProgressBar().pulse();
		}else{
			appBar.setProgressPercentage( (double) state.getCurrent() / (double) state.getMax() );
		}
		appBar.showAll();

		btnRefresh.addListener( new MouseListener(){
			public boolean mouseEvent(MouseEvent event){
				if (event.isOfType( MouseEvent.Type.BUTTON_PRESS ) ){
					if( event.getButtonPressed() == 3){
						popupRefreshMenu();
						return true;
					}
				}
				return false;
			}
		});
	}

	public void controlQueueChanged( boolean nonEmpty ){
		btnSendControl.setSensitive(nonEmpty);
		menuSendControl.setSensitive(nonEmpty);
	}

	/**
	 * Initialises GUI libraries and creates the applications.
	 */
	public static void main(String[] args) {
		Program.initGnomeUI("debbuggtk", DebbugGtk.VERSION, args);
		new BugWatcher();
		Thread.currentThread().setPriority( Thread.MAX_PRIORITY );
		Gtk.main();
	}

	public void popupRefreshMenu(){
		popupmenu = (Menu) glade.getWidget("refreshPopup");
		popupmenu.popup();
	}

	////////////////////////////////////////////////////////////////////////////////
	// Events from the UI
	//
	
	public void refreshBug(){
		bugReport.refresh();
	}

	public void refreshListing(){
		listingReport.update();
	}

	public void listingOpen(){
		new BugLister( listingReport.getListing() );
	}

	public void editListingCols(){
		listingReport.editColumns();
	}

	public void bugAddComment(){
		bugReport.comment();
	}

	public void about(){
		DebbugGtk.helpAbout();
	}

	public void openBugInViewer(){
		new BugViewer( bugReport.getBug() );
	}

	public void sendPendingControl(){
		ControlQueue.getInstance().generateMail();
	}

	public void newVirtual(){
		watchedReport.newVirt();
	}

	public void bugLister(){
		new BugLister();
	}

	public void bugReporter(){
		new BugReporter();
	}

	public void refreshClicked(){
		Downloader.getInstance().updateAll( bugReport.getBug(), listingReport.getListing(), watchedReport.getList() );
	}

	public void removeWatched(){
		watchedReport.removeSelected();
	}

	public void addListing(){
		watchedReport.watchNew();
	}

	public void quit(){
		DebbugGtk.closeWindow();
		app.hide();
	}

	public void bugEditControl(){
		bugReport.control();
	}

	public void bugViewer(){
		new BugViewer();
	}

	public void help(){
		try{
			Help.display( "debbuggtk", "contents" );
		}catch( HelpException e ){
			e.printStackTrace();
			throw new RuntimeException( e.toString() );
		}
	}

}
