#!/bin/sh
#
#  creating Debian Daily CD.
#
#  Usage: daily-cd.sh [<debian_archive>]
#
#  Debian Daily CD is CD which consists of new Debian packages
#  which aren't available in old CD.
#
#  For example, say you have Debian CD created at 2000/1/1. 
#  Debian Daily CD (2000/1/2) has only packages which are uploaded
#  at 2000/1/1.
#
#  First, you must specify your debian mirror manually
# (maybe /home/ftp/debian) to create a daily CD environment.
#  You don't need full debian mirror, but Packages.gz/Sources.gz.
#
#   % daily-sh.cd /home/ftp/debian
#
#  This should create /home/ftp/pub/debcd/<today>/ for daily-cd.
#  Once you do this, you can create debian diff CD to do as follows:
#
#   % daily-cd.sh
#
#  This searches the lastest daily CD environment and creates a new daily
#  CD automatically.
#
#   % daily-cd.sh /home/ftp/pub/debcd/20000101/
#
#  This creates Diff-CD from 2000/1/1's Debian.
#
#  Have fun :)
#
# Copyright (C) 2002  Masato Taruishi <taru@debian.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License with
#  the Debian GNU/Linux distribution in file /usr/share/common-licenses/GPL;
#  if not, write to the Free Software Foundation, Inc., 59 Temple Place,
#  Suite 330, Boston, MA  02111-1307  USA

MIRRORHOST=ftp.debian.org
DIST=unstable
SECTS=main,contrib,non-free
ARCHES=i386
DIFFDEB=/usr/share/doc/debpartial/examples/diffdeb
CDDIR=/home/ftp/pub/debcd
CDSIZE=CD74
ROOTDIR=/debian

if test -f $HOME/.daily-cd.conf; then
 . $HOME/.daily-cd.conf
fi

##################

set -e

install -d $CDDIR
DATE=`date +%Y%m%d`
DIR=$CDDIR/$DATE

if [ -z "$1" ]; then
  MIRRORDIR=`/bin/ls $CDDIR/ | grep -v $DATE | sort -n | tail -1`
  if [ -z "$MIRRORDIR" ] || [ ! -d $CDDIR/$MIRRORDIR ]; then
    echo "You must specify debian archive directory manually."
    exit 1
  fi
  MIRRORDIR=$CDDIR/$MIRRORDIR
else
  MIRRORDIR=$1
fi

tmp=`tempfile`
trap "rm -f $tmp" 0

echo -n "Downloading New Packages/Sources.gz... "
debmirror $DIR -r $ROOTDIR -h $MIRRORHOST -d $DIST -a $ARCHES -s $SECTS --exclude=. --nocleanup
echo "done."

uptodate="true"
IFS=,
for sect in $SECTS
do
  for arch in $ARCHES
  do
    echo -n "Comparing Packages.gz for $arch/$sect... "

    install -d $DIR/DIFF/dists/$DIST/$sect/binary-$arch/
    if [ -f $MIRRORDIR/dists/$DIST/$sect/binary-$arch/Packages.gz ]; then
      $DIFFDEB $MIRRORDIR/dists/$DIST/$sect/binary-$arch/Packages.gz \
        $DIR/dists/$DIST/$sect/binary-$arch/Packages.gz > $tmp
      gzip -c $tmp > $DIR/DIFF/dists/$DIST/$sect/binary-$arch/Packages.gz
      if [ -s $tmp ]; then
        uptodate="false"
        echo "done."
      else
        echo "up-to-date."
      fi
    else
      cp $DIR/dists/$DIST/$sect/binary-$arch/Packages.gz \
        $DIR/DIFF/dists/$DIST/$sect/binary-$arch/Packages.gz
      uptodate="false"
      echo "new."
    fi
  done

  echo -n "Comparing Sources.gz for $sect... "
  $DIFFDEB $MIRRORDIR/dists/$DIST/$sect/source/Sources.gz \
    $DIR/dists/$DIST/$sect/source/Sources.gz > $tmp

  install -d $DIR/DIFF/dists/$DIST/$sect/source/
  gzip -c $tmp > $DIR/DIFF/dists/$DIST/$sect/source/Sources.gz

  if [ -s $tmp ]; then
    uptodate="false"
    echo "done."
  else
    echo "up-to-date."
  fi
done
unset IFS

if [ $uptodate = true ]; then
  echo "No Changed"
  exit 0 
fi

debpartial -a $ARCHES -d $DIST -s $SECTS -S $CDSIZE -m $DIR/DIFF $DIR/CD

for part in $DIR/CD/*; do
  debmirror $part -r $ROOTDIR -h $MIRRORHOST -d $DIST -a $ARCHES -s $SECTS --skippackages
  mkisofs -J -r -o daily-`basename $MIRRORDIR`-$DATE-`basename $part`.iso $part
done
