# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: btformats.py 199 2007-08-06 00:50:01Z camrdale-guest $

"""Functions for verifying debtorrent metainfo.

These functions all work on bdecoded debtorrent metainfo, and are used
to verify their conformance with the protocol.

@type reg: C{regex}
@var reg: a compiled regex for verifying the security of path names
@type ints: C{tuple} of C{types}
@var ints: the types that are acceptable for integer values

"""

from types import StringType, LongType, IntType, ListType, DictType
from re import compile
from bisect import bisect

reg = compile(r'^[^/\\.~][^/\\]*$')

ints = (LongType, IntType)

def check_info(info):
    """Checks the info dictionary for conformance.
    
    Verifies that the info dictionary of the metainfo conforms to the
    debtorrent protocol.
    
    @type info: C{dictionary}
    @param info: the info field from the metainfo dictionary
    @raise ValueError: if the info doesn't conform
    
    """
    
    if type(info) != DictType:
        raise ValueError, 'bad metainfo - not a dictionary'
    pieces = info.get('pieces')
    if type(pieces) != StringType or len(pieces) % 20 != 0:
        raise ValueError, 'bad metainfo - bad pieces key'
    piecelengths = info.get('piece lengths')
    if type(piecelengths) != ListType:
        raise ValueError, 'bad metainfo - bad piece lengths list'
    total_length = 0L
    piece_bounds = [0L]
    for length in piecelengths:
        if type(length) not in ints or length <= 0:
            raise ValueError, 'bad metainfo - bad piece length'
        total_length += length
        piece_bounds.append(total_length)
    if info.has_key('files') == info.has_key('length'):
        raise ValueError, 'single/multiple file mix'
    files = info.get('files')
    if type(files) != ListType:
        raise ValueError, 'bad metainfo - bad files list'
    total_length = 0L
    for f in files:
        if type(f) != DictType:
            raise ValueError, 'bad metainfo - bad file value'
        length = f.get('length')
        if type(length) not in ints or length < 0:
            raise ValueError, 'bad metainfo - bad length'
        total_length += length
        if piece_bounds[bisect(piece_bounds,total_length)-1] != total_length:
            raise ValueError, 'bad metainfo - file does not end on piece boundary'
        path = f.get('path')
        if type(path) != ListType or path == []:
            raise ValueError, 'bad metainfo - bad path'
        for p in path:
            if type(p) != StringType:
                raise ValueError, 'bad metainfo - bad path dir'
            if not reg.match(p):
                raise ValueError, 'path %s disallowed for security reasons' % p
# Removed to speed up checking of large files
#        for i in xrange(len(files)):
#            for j in xrange(i):
#                if files[i]['path'] == files[j]['path']:
#                    raise ValueError, 'bad metainfo - duplicate path'

def check_message(message):
    """Checks the metainfo dictionary for conformance.
    
    Verifies that the metainfo dictionary conforms to the
    debtorrent protocol.
    
    @type message: C{dictionary}
    @param message: the bdecoded metainfo dictionary
    @raise ValueError: if the metainfo doesn't conform
    
    """
    
    if type(message) != DictType:
        raise ValueError, 'bad dictionary'
    check_info(message.get('info'))
    if type(message.get('announce')) != StringType:
        raise ValueError, 'bad announce'
    name = message.get('name')
    if type(name) != StringType:
        raise ValueError, 'bad name'
    if not reg.match(name):
        raise ValueError, 'name %s disallowed for security reasons' % name

def check_peers(message):
    """Checks the peers dictionary returned by a tracker for conformance.
    
    Verifies that the peers dictionary returned by a tracker conforms to the
    debtorrent protocol.
    
    @type message: C{dictionary}
    @param message: the bdecoded peers dictionary returned by a tracker
    @raise ValueError: if the info doesn't conform
    
    """
    
    if type(message) != DictType:
        raise ValueError, 'bad dictionary'
    if message.has_key('failure reason'):
        if type(message['failure reason']) != StringType:
            raise ValueError, 'bad failure reason'
        return
    peers = message.get('peers')
    if type(peers) == ListType:
        for p in peers:
            if type(p) != DictType:
                raise ValueError, 'bad peers - bad dictionary'
            if type(p.get('ip')) != StringType:
                raise ValueError, 'bad peers - bad ip'
            port = p.get('port')
            if type(port) not in ints or p <= 0:
                raise ValueError, 'bad peers - bad port'
            if p.has_key('peer id'):
                id = p['peer id']
                if type(id) != StringType or len(id) != 20:
                    raise ValueError, 'bad peers - bad peer ID'
    elif type(peers) != StringType or len(peers) % 6 != 0:
        raise ValueError, 'bad compact peers'
    interval = message.get('interval', 1)
    if type(interval) not in ints or interval <= 0:
        raise ValueError, 'bad interval'
    minint = message.get('min interval', 1)
    if type(minint) not in ints or minint <= 0:
        raise ValueError, 'bad min interval'
    if type(message.get('tracker id', '')) != StringType:
        raise ValueError, 'bad tracker id'
    npeers = message.get('num peers', 0)
    if type(npeers) not in ints or npeers < 0:
        raise ValueError, 'bad num peers'
    dpeers = message.get('done peers', 0)
    if type(dpeers) not in ints or dpeers < 0:
        raise ValueError, 'bad done peers'
    last = message.get('last', 0)
    if type(last) not in ints or last < 0:
        raise ValueError, 'bad last'
