# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: track.py 365 2008-01-28 04:27:31Z camrdale-guest $

"""Tools to track a download swarm.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module
@type defaults: C{list} of (C{string}, unknown, C{string})
@var defaults: the parameter names, default values, and descriptions
@type alas: C{string}
@var alas: the message to send when the data is not found
@type local_IPs: L{DebTorrent.subnetparse.IP_List}
@var local_IPs: the list of IP subnets that are considered local
@type http_via_filter: C{Regular Expression}
@var http_via_filter: the regular expression object to search 'via' 
    header information for the NAT IP address

"""

from DebTorrent.parseargs import parseargs, formatDefinitions
from DebTorrent.RawServer import RawServer, autodetect_ipv6, autodetect_socket_style
from DebTorrent.HTTPHandler import HTTPHandler, months, weekdays
from DebTorrent.parsedir import parsedir
from NatCheck import NatCheck, CHECK_PEER_ID_ENCRYPTED
from DebTorrent.BTcrypto import CRYPTO_OK
from T2T import T2TList
from DebTorrent.subnetparse import IP_List, ipv6_to_ipv4, to_ipv4, is_valid_ip, is_ipv4
from DebTorrent.iprangeparse import IP_List as IP_Range_List
from DebTorrent.torrentlistparse import parsetorrentlist
from threading import Event, Thread
from DebTorrent.bencode import bencode, bdecode, Bencached
from DebTorrent.zurllib import urlopen, quote, unquote
from Filter import Filter
from urlparse import urlparse
from os import rename, getpid
from os.path import exists, isfile
from cStringIO import StringIO
from time import time, gmtime, strftime, localtime
from DebTorrent.clock import clock
from random import shuffle, seed, randrange
from sha import sha
from types import StringType, IntType, LongType, ListType, DictType
from binascii import b2a_hex, a2b_hex
from string import lower
import sys, os, logging
import signal
import re
import DebTorrent.__init__
from DebTorrent.__init__ import version, createPeerID
from DebTorrent.ConfigDir import ConfigDir

logger = logging.getLogger('DebTorrent.BT1.track')

defaults = [
    # Not in the config file
    ('configfile', '', 'the configuration file to use, if not specified then ' +
        'a file in /etc/debtorrent will be used, followed by ' +
        'a file in the .DebTorrent directory in the user\'s home directory'),
    # Locations
    ('cache_dir', '', 'the directory to use to get/store cache files, if not ' + 
        'specified then a .DebTorrent directory in the user\'s home directory ' +
        'will be used'),
    ('save_state_interval', 5 * 60, 'seconds between saving state to a file'),
    ('log_dir', '',
        'directory to write the logfiles to (default is to use the cache directory)'),
    ('log_level', 10,
        'level to write the logfiles at, varies from 10 (debug) to 50 (critical)'),
    # Connections
    ('port', 6969, "port to listen on"),
    ('bind', '', 'comma-separated list of ips/hostnames to bind to locally'),
#    ('ipv6_enabled', autodetect_ipv6(),
    ('ipv6_enabled', 0,
         'allow the client to connect to peers via IPv6'),
    ('ipv6_binds_v4', autodetect_socket_style(),
        'set if an IPv6 server socket will also field IPv4 connections'),
    ('socket_timeout', 15, 'timeout for closing connections'),
    ('timeout_check_interval', 5,
        'time to wait between checking if any connections have timed out'),
    # Allowed Torrents and Peers
    ('allowed_dir', '', 'only allow downloads for .dtorrents in this dir'),
    ('allowed_list', '', 'only allow downloads for hashes in this list (hex format, one per line)'),
    ('allowed_controls', 0, 'allow special keys in torrents in the allowed_dir to affect tracker access'),
    ('allowed_ips', '', 'only allow connections from IPs specified in the given file; '+
             'file contains subnet data in the format: aa.bb.cc.dd/len'),
    ('banned_ips', '', "don't allow connections from IPs specified in the given file; "+
             'file contains IP range data in the format: xxx:xxx:ip1-ip2'),
    ('parse_dir_interval', 60, 'seconds between reloading of allowed_dir or allowed_file ' +
             'and allowed_ips and banned_ips lists'),
    # Peer Requests
    ('compact_reqd', 1, "only allow peers that accept a compact response"),
    ('reannounce_interval', 30 * 60, 'seconds downloaders should wait between reannouncements'),
    ('response_size', 50, 'number of peers to send in an info message'),
    ('nat_check', 3,
        "how many times to check if a downloader is behind a NAT (0 = don't check)"),
    ('timeout_downloaders_interval', 45 * 60, 'seconds between expiring downloaders'),
    ('min_time_between_cache_refreshes', 600.0,
        'minimum time in seconds before a cache is considered stale and is flushed'),
    ('only_local_override_ip', 2, "ignore the ip GET parameter from machines which aren't on local network IPs " +
             "(0 = never, 1 = always, 2 = ignore if NAT checking is not enabled)"),
    ('dedicated_seed_id', '', 'allows tracker to monitor dedicated seed(s) and flag torrents as seeded'),
    # Non-Peer Requests
    ('show_infopage', 1, "whether to display an info page when the tracker's root dir is loaded"),
    ('infopage_redirect', '', 'a URL to redirect the info page to'),
    ('favicon', '', 'file containing x-icon data to return when browser requests favicon.ico'),
    ('show_names', 1, 'whether to display names from allowed dir'),
    ('allow_get', 0, 'use with allowed_dir; adds a /file?hash={hash} url that allows users to download the torrent file'),
    ('keep_dead', 0, 'keep dead torrents after they expire (so they still show up on your scrape and web page)'),
    ('scrape_allowed', 'full', 'scrape access allowed (can be none, specific or full)'),
    # Request Logging
    ('min_time_between_log_flushes', 3.0,
        'minimum time it must have been since the last flush to do another one'),
    ('hupmonitor', 0, 'whether to reopen the log file upon receipt of HUP signal'),
    ('log_nat_checks', 1,
        "whether to add entries to the log for nat-check results"),
    # Multi-Tracker
    ('multitracker_enabled', 0, 'whether to enable multitracker operation'),
    ('multitracker_allowed', 'autodetect', 'whether to allow incoming tracker announces (can be none, autodetect or all)'),
    ('multitracker_reannounce_interval', 2 * 60, 'seconds between outgoing tracker announces'),
    ('multitracker_maxpeers', 20, 'number of peers to get in a tracker announce'),
    ('aggregate_forward', '', 'format: <url>[,<password>] - if set, forwards all non-multitracker to this url with this optional password'),
    ('aggregator', '0', 'whether to act as a data aggregator rather than a tracker.  If enabled, may be 1, or <password>; ' +
             'if password is set, then an incoming password is required for access'),
    ('http_timeout', 60, 
        'number of seconds to wait before assuming that an http connection to another tracker has timed out'),
  ]

def statefiletemplate(x):
    """Check the saved state file for corruption.
    
    @type x: C{dictionary}
    @param x: the dictionary of information retrieved from the state file
    @raise ValueError: if the state file info is corrupt
    
    """
    
    if type(x) != DictType:
        raise ValueError
    for cname, cinfo in x.items():
        if cname == 'peers':
            for y in cinfo.values():      # The 'peers' key is a dictionary of SHA hashes (torrent ids)
                if type(y) != DictType:   # ... for the active torrents, and each is a dictionary
                    raise ValueError
                for id, info in y.items(): # ... of client ids interested in that torrent
                    if (len(id) != 20):
                        raise ValueError
                    if type(info) != DictType:  # ... each of which is also a dictionary
                        raise ValueError # ... which has an IP, a Port, and a Bytes Left count for that client for that torrent
                    if type(info.get('ip', '')) != StringType:
                        raise ValueError
                    port = info.get('port')
                    if type(port) not in (IntType,LongType) or port < 0:
                        raise ValueError
                    left = info.get('left')
                    if type(left) not in (IntType,LongType) or left < 0:
                        raise ValueError
                    if type(info.get('supportcrypto')) not in (IntType,LongType):
                        raise ValueError
                    if type(info.get('requirecrypto')) not in (IntType,LongType):
                        raise ValueError
        elif cname == 'stats':
            if (type(cinfo) != DictType): # The 'stats' key is a dictionary of SHA hashes (torrent ids)
                raise ValueError          # ... for keeping track of the total completions per torrent
            for y in cinfo.values():      # ... each torrent has an integer value
                if type(y) != DictType:
                    raise ValueError      # ... for the number of reported completions for that torrent
                completed = y.get('completed')
                if type(completed) not in (IntType,LongType) or completed < 0:
                    raise ValueError
                uploaded = y.get('uploaded', 0)
                if type(uploaded) not in (IntType,LongType) or uploaded < 0:
                    raise ValueError
                downloaded = y.get('downloaded', 0)
                if type(downloaded) not in (IntType,LongType) or downloaded < 0:
                    raise ValueError
        elif cname == 'allowed':
            if (type(cinfo) != DictType): # a list of info_hashes and included data
                raise ValueError
            if x.has_key('allowed_dir_files'):
                adlist = [z[1] for z in x['allowed_dir_files'].values()]
                for y in cinfo.keys():        # and each should have a corresponding key here
                    if not y in adlist:
                        raise ValueError
        elif cname == 'allowed_dir_files':
            if (type(cinfo) != DictType): # a list of files, their attributes and info hashes
                raise ValueError
            dirkeys = {}
            for y in cinfo.values():      # each entry should have a corresponding info_hash
                if not y[1]:
                    continue
                if not x['allowed'].has_key(y[1]):
                    raise ValueError
                if dirkeys.has_key(y[1]): # and each should have a unique info_hash
                    raise ValueError
                dirkeys[y[1]] = 1
        elif cname == 'version':
            if (type(cinfo) != IntType): # the version of the state file
                raise ValueError

def loadState(configdir):
    """Load the saved state from a previous run.
    
    @type configdir: L{DebTorrent.ConfigDir.ConfigDir}
    @param configdir: the configuration and cache directory manager
    @rtype: C{dictionary}
    @return: the saved state from the previous run
    
    """
    
    tempstate = configdir.getState()
    if tempstate:
        version = tempstate.get('version', 0)
        # Upgrade from the previous version
        if version < 1:
            try:
                tempstate['stats'] = {}
                if 'completed' in tempstate:
                    for hash, num in tempstate['completed'].items():
                        tempstate['stats'][hash] = {'completed': num,
                                                    'downloaded': 0,
                                                    'uploaded': 0}
                    del tempstate['completed']
                tempstate['version'] = 1
                logger.info('successfully upgraded the previous state file')
            except:
                logger.exception('old statefile corrupt; resetting')
                tempstate = {'version': 1}

        # Test the state to make sure it's not been corrupted
        try:
            statefiletemplate(tempstate)
            logger.info('successfully loaded the previous state file')
        except:
            logger.exception('statefile corrupt; resetting')
            tempstate = {'version': 1}
    else:
        tempstate = {'version': 1}
    
    return tempstate

alas = 'your file may exist elsewhere in the universe\nbut alas, not here\n'

local_IPs = IP_List()
local_IPs.set_intranet_addresses()


def isotime(secs = None):
    """Create an ISO formatted string of the time.
    
    @type secs: C{float}
    @param secs: number of seconds since the epoch 
        (optional, default is to use the current time)
    @rtype: C{string}
    @return: the ISO formatted string representation of the time
    
    """
    
    if secs == None:
        secs = time()
    return strftime('%Y-%m-%d %H:%M UTC', gmtime(secs))

http_via_filter = re.compile(' for ([0-9.]+)\Z')

def _get_forwarded_ip(headers):
    """Extract the unNATed IP address from the headers.
    
    @type headers: C{dictionary}
    @param headers: the headers received from the client
    @rtype: C{string}
    @return: the extracted IP address
    
    """
    
    header = headers.get('x-forwarded-for')
    if header:
        try:
            x,y = header.split(',')
        except:
            return header
        if is_valid_ip(x) and not local_IPs.includes(x):
            return x
        return y
    header = headers.get('client-ip')
    if header:
        return header
    header = headers.get('via')
    if header:
        x = http_via_filter.search(header)
        try:
            return x.group(1)
        except:
            pass
    header = headers.get('from')
    #if header:
    #    return header
    #return None
    return header

def get_forwarded_ip(headers):
    """Extract the unNATed IP address from the headers.
    
    @type headers: C{dictionary}
    @param headers: the headers received from the client
    @rtype: C{string}
    @return: the extracted IP address (or None if one could not be extracted)
    
    """
    
    x = _get_forwarded_ip(headers)
    if not is_valid_ip(x) or local_IPs.includes(x):
        return None
    return x

def compact_peer_info(ip, port):
    """Create a compact representation of peer contact info.
    
    @type ip: C{string}
    @param ip: the IP address of the peer
    @type port: C{int}
    @param port: the port number to contact the peer on
    @rtype: C{string}
    @return: the compact representation (or the empty string if there is no
        compact representation)
    
    """
    
    try:
        s = ( ''.join([chr(int(i)) for i in ip.split('.')])
              + chr((port & 0xFF00) >> 8) + chr(port & 0xFF) )
        if len(s) != 6:
            raise ValueError
    except:
        s = ''  # not a valid IP, must be a domain name
    return s

def clean_name(name):
    """Clean up the name of the torrent to remove mirror info.
    
    @type name: C{string}
    @param name: the torrent name
    @rtype: C{string}
    @return: the cleaned name
    
    """
    
    if not name:
        return ''
    
    safe_name = name
    try:
        # remove the mirror name
        safe_name = safe_name.split('_', 1)[1]
    except:
        pass
    try:
        # remove everything to the last 'dists'
        safe_name = safe_name[safe_name.rindex('dists_'):]
    except:
        try:
            # remove everything to the last 'debian'
            safe_name = safe_name[safe_name.rindex('debian_')+7:]
        except:
            pass
    return safe_name

class Tracker:
    """Track a download swarm.
    
    @type config: C{dictionary}
    @ivar config: the configuration parameters
    @type response_size: C{int}
    @ivar response_size: default number of peers to send in an info message
    @type configdir: L{DebTorrent.ConfigDir.ConfigDir}
    @ivar configdir: the configuration and cache directory manager
    @type natcheck: C{int}
    @ivar natcheck: how many times to check if a downloader is behind a NAT
    @type parse_dir_interval: C{int}
    @ivar parse_dir_interval: seconds between reloading of the allowed 
        directory or file, and the lists of allowed and banned IPs
    @type favicon: C{string}
    @ivar favicon: file containing x-icon data
    @type rawserver: L{DebTorrent.RawServer.RawServer}
    @ivar rawserver: the server to use for scheduling
    @type cached: unknown
    @ivar cached: unknown
    @type cached_t: unknown
    @ivar cached_t: unknown
    @type times: C{dictionary}
    @ivar times: keys are info hashes, values are dictionaries with keys the
        peer IDs of peers connected to that torrent and values the last time
        the peer was seen
    @type state: C{dictionary}
    @ivar state: the current state information for the tracking
    @type seedcount: C{dictionary}
    @ivar seedcount: keys are info hashes, values are the number of seeds
        connected to that torrent
    @type allowed_IPs: L{DebTorrent.subnetparse.IP_List}
    @ivar allowed_IPs: the IPs that are allowed to connect, or None if all are
    @type banned_IPs: L{DebTorrent.iprangeparse.IP_List}
    @ivar banned_IPs: the IPs that are not allowed to connect
    @type allowed_ip_mtime: C{int}
    @ivar allowed_ip_mtime: the last modification time of the allowed IPs file
    @type banned_ip_mtime: C{int}
    @ivar banned_ip_mtime: the last modification time of the banned IPs file
    @type only_local_override_ip: C{boolean}
    @ivar only_local_override_ip: whether to ignore the "ip" parameter from 
        machines which aren't on local network IPs
    @type downloads: C{dictionary}
    @ivar downloads: keys are info hashes, values are dictionaries with keys the
        peer IDs of peers connected to that torrent and values the dictionaries
        of information about the peer
    @type stats: C{dictionary}
    @ivar stats: keys are info hashes, values are dictionaries containing
        statistics related to the torrent
    @type becache: C{list} of C{list} of C{dictionary}
    @ivar becache: keys are the infohashes, values are the cached peer data.
        peer set format::
            [[l0, s0], [l1, s1], ...]
                l,s   = dictionaries of leechers and seeders (by peer ID)
                l0,s0 = compact representation, don't require crypto
                l1,s1 = compact representation, support crypto
                l2,s2 = [compact representation, crypto_flag], for all peers
            additionally, if --compact_reqd = 0:
                l3,s3 = [ip,port,peerid] for all peers
                l4,l4 = [ip,port] for all peers
    @type cache_default: C{list} of (C{dictionary}, C{dictionary})
    @ivar cache_default: the default cache entry for new caches
    @type trackerid: C{string}
    @ivar trackerid: the randomly generated tracker ID of this tracker
    @type reannounce_interval: C{int}
    @ivar reannounce_interval: seconds downloaders should wait between reannouncements
    @type save_state_interval: C{int}
    @ivar save_state_interval: seconds between saving the state file
    @type show_names: C{boolean}
    @ivar show_names: whether to display names from allowed dir
    @type prevtime: C{float}
    @ivar prevtime: the last time downloaders were expired from the cache
    @type timeout_downloaders_interval: C{int}
    @ivar timeout_downloaders_interval: seconds between expiring downloaders
    @type allow_get: C{boolean}
    @ivar allow_get: whether torrets can be downloaded from the info page
    @type t2tlist: L{T2T.T2TList}
    @ivar t2tlist: the list of tracker to tracker connections
    @type allowed: C{dictionary}
    @ivar allowed: the torrents that are allowed to be tracked, keys are the
        info hashes, values are the torrent data
    @type allowed_list_mtime: C{int}
    @ivar allowed_list_mtime: the last modification time of the list of allowed
        torrents file
    @type allowed_dir_files: C{dictionary}
    @ivar allowed_dir_files: torrent files found in the allowed directory
    @type allowed_dir_blocked: C{dictionary}
    @ivar allowed_dir_blocked: unparseable torrent files found in the allowed directory
    @type uq_broken: C{boolean}
    @ivar uq_broken: whether URL quoting of '+' is broken
    @type keep_dead: C{boolean}
    @ivar keep_dead: whether to keep dead torrents after they expire
    @type Filter: L{Filter.Filter}
    @ivar Filter: the filter to use for disallowing connecting peers
    @type is_aggregator: C{boolean}
    @ivar is_aggregator: whether to act as a data aggregator rather than a tracker
    @type aggregator_key: C{string}
    @ivar aggregator_key: the password to use to verify connecting trackers
    @type aggregate_forward: C{string}
    @ivar aggregate_forward: URL to forward all non-multitracker connections to
    @type aggregate_password: C{string}
    @ivar aggregate_password: password to use when forwarding non-multitracker connections
    @type dedicated_seed_id: C{string}
    @ivar dedicated_seed_id: dedicated seed ID to expect from clients
    @type is_seeded: C{dictionary}
    @ivar is_seeded: keys are info hashes, values are true if the torrent has
        a dedicated seed connected
    @type cachetime: C{int}
    @ivar cachetime: elasped time since starting for caclulating cache refreshes
    
    """
    
    def __init__(self, config, rawserver, configdir):
        """Initialize the instance.
        
        @type config: C{dictionary}
        @param config: the configuration parameters
        @type rawserver: L{DebTorrent.RawServer.RawServer}
        @param rawserver: the server to use for scheduling
        @type configdir: L{DebTorrent.ConfigDir.ConfigDir}
        @param configdir: the configuration and cache directory manager
        
        """
        
        self.config = config
        self.response_size = config['response_size']
        self.configdir = configdir
        self.natcheck = config['nat_check']
        favicon = config['favicon']
        self.parse_dir_interval = config['parse_dir_interval']
        self.favicon = None
        if favicon:
            try:
                h = open(favicon,'r')
                self.favicon = h.read()
                h.close()
            except:
                logger.warning("specified favicon file does not exist.")
        self.rawserver = rawserver
        self.cached = {}    # format: infohash: [[time1, l1, s1], [time2, l2, s2], ...]
        self.cached_t = {}  # format: infohash: [time, cache]
        self.times = {}
        self.seedcount = {}
        self.torrent_names = {}

        self.allowed_IPs = None
        self.banned_IPs = None
        if config['allowed_ips'] or config['banned_ips']:
            self.allowed_ip_mtime = 0
            self.banned_ip_mtime = 0
            self.read_ip_lists()
                
        self.only_local_override_ip = config['only_local_override_ip']
        if self.only_local_override_ip == 2:
            self.only_local_override_ip = not config['nat_check']

        if CHECK_PEER_ID_ENCRYPTED and not CRYPTO_OK:
            logger.warning('crypto library not installed,' +
                   ' cannot completely verify encrypted peers')

        self.state = loadState(configdir)
        self.downloads = self.state.setdefault('peers', {})
        self.stats = self.state.setdefault('stats', {})

        self.becache = {}

        if config['compact_reqd']:
            self.cache_default_len = 3
        else:
            self.cache_default_len = 5
        for infohash, ds in self.downloads.items():
            torrent_name = []
            self.torrent_names[infohash] = torrent_name
            self.seedcount[infohash] = 0
            for x,y in ds.items():
                ip = y['ip']
                if ( (self.allowed_IPs and not self.allowed_IPs.includes(ip))
                     or (self.banned_IPs and self.banned_IPs.includes(ip)) ):
                    del ds[x]
                    continue
                if not y['left']:
                    self.seedcount[infohash] += 1
                name = clean_name(y.get('name', ''))
                if name and name not in torrent_name:
                    torrent_name.append(name)
                if y.get('nat',-1):
                    continue
                gip = y.get('given_ip')
                if is_valid_ip(gip) and (
                    not self.only_local_override_ip or local_IPs.includes(ip) ):
                    ip = gip
                self.natcheckOK(infohash,x,ip,y['port'],y)
            
        for x in self.downloads.keys():
            self.times[x] = {}
            for y in self.downloads[x].keys():
                self.times[x][y] = 0

        self.trackerid = createPeerID('-T-')
        seed(self.trackerid)
                
        self.reannounce_interval = config['reannounce_interval']
        self.save_state_interval = config['save_state_interval']
        self.show_names = config['show_names']
        rawserver.add_task(self.save_state, self.save_state_interval)
        self.prevtime = clock()
        self.timeout_downloaders_interval = config['timeout_downloaders_interval']
        rawserver.add_task(self.expire_downloaders, self.timeout_downloaders_interval)
                
        self.allow_get = config['allow_get']
        
        self.t2tlist = T2TList(config['multitracker_enabled'], self.trackerid,
                               config['multitracker_reannounce_interval'],
                               config['multitracker_maxpeers'], config['http_timeout'],
                               self.rawserver)

        if config['allowed_list']:
            if config['allowed_dir']:
                logger.warning('allowed_dir and allowed_list options cannot be used together,'+
                                ' disregarding allowed_dir')
                config['allowed_dir'] = ''
            self.allowed = self.state.setdefault('allowed_list',{})
            self.allowed_list_mtime = 0
            self.parse_allowed()
            self.remove_from_state('allowed','allowed_dir_files')
            if config['multitracker_allowed'] == 'autodetect':
                config['multitracker_allowed'] = 'none'
            config['allowed_controls'] = 0

        elif config['allowed_dir']:
            self.allowed = self.state.setdefault('allowed',{})
            self.allowed_dir_files = self.state.setdefault('allowed_dir_files',{})
            self.allowed_dir_blocked = {}
            self.parse_allowed()
            self.remove_from_state('allowed_list')

        else:
            self.allowed = None
            self.remove_from_state('allowed','allowed_dir_files', 'allowed_list')
            if config['multitracker_allowed'] == 'autodetect':
                config['multitracker_allowed'] = 'none'
            config['allowed_controls'] = 0
                
        self.uq_broken = unquote('+') != ' '
        self.keep_dead = config['keep_dead']
        self.Filter = Filter(rawserver.add_task)
        
        aggregator = config['aggregator']
        if aggregator == '0':
            self.is_aggregator = False
            self.aggregator_key = None
        else:
            self.is_aggregator = True
            if aggregator == '1':
                self.aggregator_key = None
            else:
                self.aggregator_key = aggregator
            self.natcheck = False
                
        send = config['aggregate_forward']
        if not send:
            self.aggregate_forward = None
        else:
            try:
                self.aggregate_forward, self.aggregate_password = send.split(',')
            except:
                self.aggregate_forward = send
                self.aggregate_password = None

        self.dedicated_seed_id = config['dedicated_seed_id']
        self.is_seeded = {}

        self.cachetime = 0
        self.cachetimeupdate()

    def cachetimeupdate(self):
        """Update the L{cachetime} every second."""
        self.cachetime += 1     # raw clock, but more efficient for cache
        self.rawserver.add_task(self.cachetimeupdate,1)

    def aggregate_senddata(self, query):
        """Fork sending data to a tracker aggregator.
        
        @type query: C{string}
        @param query: the query to send
        
        """
        
        url = self.aggregate_forward+'?'+query
        if self.aggregate_password is not None:
            url += '&password='+self.aggregate_password
        rq = Thread(target = self._aggregate_senddata, args = [url], name = 'Tracker._aggregate_senddata')
        rq.setDaemon(False)
        rq.start()

    def _aggregate_senddata(self, url):
        """Send a URL request to a tracker data aggregator.
                     
        Just send, don't attempt to error check, discard any returned data.
        
        @type url: C{string}
        @param url: the URL to request

        """
        
        try:
            h = urlopen(url)
            h.read()
            h.close()
        except:
            return


    def get_infopage(self):
        """Format the info page to display for normal browsers.
        
        Formats the currently tracked torrents into a table in human-readable
        format to display in a browser window.
        
        @rtype: (C{int}, C{string}, C{dictionary}, C{string})
        @return: the HTTP status code, status message, headers, and message body
        
        """
        
        try:
            if not self.config['show_infopage']:
                return (404, 'Not Found', {'Content-Type': 'text/plain', 'Pragma': 'no-cache'}, alas)
            red = self.config['infopage_redirect']
            if red:
                return (302, 'Found', {'Content-Type': 'text/html', 'Location': red},
                        '<A HREF="'+red+'">Click Here</A>')
            
            # Write the document header
            s = StringIO()
            s.write('<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1//EN" ' \
                    '"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd">\n' \
                    '<html><head><title>DebTorrent tracker info</title>\n')
            if self.favicon is not None:
                s.write('<link rel="shortcut icon" href="/favicon.ico">\n')
            s.write('</head>\n<body>\n' \
                    '<h3>DebTorrent tracker info</h3>\n'\
                    '<ul>\n'
                    '<li><strong>tracker version:</strong> %s</li>\n' \
                    '<li><strong>server time:</strong> %s</li>\n' \
                    '</ul>\n' % (version, isotime()))

            # Collect the torrent hashes and names
            if self.config['allowed_dir']:
                if self.show_names:
                    names = [ (self.allowed[hash]['name'],hash)
                              for hash in self.allowed.keys() ]
                else:
                    names = [ (None,hash)
                              for hash in self.allowed.keys() ]
            else:
                if self.show_names:
                    names = [ ('\n<br>'.join(self.torrent_names[hash]),hash) for hash in self.downloads.keys() ]
                else:
                    names = [ (None,hash) for hash in self.downloads.keys() ]

            if not names:
                s.write('<p>not tracking any files yet...</p>\n')
            else:
                names.sort()
                number_files = 0
                total_peers = 0
                total_uploaded = 0L
                total_downloaded = 0L
                total_size = 0L
                
                # Write the table headers
                s.write('<table summary="files" border="1">\n\n' \
                        '<tr><th>torrent name/<br><code>    info hash</code></th>\n')
                if self.config['allowed_dir']:
                    s.write('<th align="right">size</th>\n')
                s.write('<th align="right">peers</th>\n' \
                        '<th align="right">downloaded</th>\n' \
                        '<th align="right">uploaded</th>\n' \
                        '<th align="right">saved</th></tr>\n\n')

                # Display a table row for each torrent
                for name,hash in names:
                    # Get the stats for this torrent
                    uploaded = self.stats.get(hash, {}).get('uploaded', 0)
                    total_uploaded += uploaded
                    downloaded = self.stats.get(hash, {}).get('downloaded', 0)
                    total_downloaded += downloaded
                    saved = 0.0
                    if downloaded > 0:
                        saved = 100.0 * float(uploaded) / float(downloaded)
                    peers = len(self.downloads[hash])
                    total_peers += peers

                    # Display is different if we are using allowed_dir
                    if self.config['allowed_dir'] and self.allowed.has_key(hash):
                        number_files += 1
                        size = self.allowed[hash]['length']  # size
                        total_size += long(size)
                        if self.allow_get == 1:
                            linkname = '<a href="/file?info_hash=' + quote(hash) + '">' + name + '</a>'
                        else:
                            linkname = name
                        s.write('<tr><td>%s<br><code>    %s</code></td>\n' \
                                '<td align="right">%s</td>\n' \
                                '<td align="right">%i</td>\n' \
                                '<td align="right">%s</td>\n' \
                                '<td align="right">%s</td>\n' \
                                '<td align="right">%0.1f %%</td></tr>\n\n' \
                                % (linkname, b2a_hex(hash), size_format(size),
                                   peers, size_format(downloaded),
                                   size_format(uploaded), saved))
                    else:
                        number_files += 1
                        s.write('<tr><td>')
                        if name:
                            s.write('%s<br>' % name)
                        s.write('<code>    %s</code></td>\n' \
                                '<td align="right">%i</td>\n' \
                                '<td align="right">%s</td>\n' \
                                '<td align="right">%s</td>\n' \
                                '<td align="right">%0.1f %%</td></tr>\n\n' \
                                % (b2a_hex(hash), peers, size_format(downloaded),
                                   size_format(uploaded), saved))

                # The table footer with summary statistics
                saved = 0.0
                if total_downloaded > 0:
                    saved = 100.0 * float(total_uploaded) / float(total_downloaded)
                s.write('<tr><td align="right">%i torrents</td>\n' % number_files)
                if self.config['allowed_dir']:
                    s.write('<td align="right">%s</td>\n' % size_format(total_size))
                s.write('<td align="right">%i</td>\n' \
                        '<td align="right">%s</td>\n' \
                        '<td align="right">%s</td>\n' \
                        '<td align="right">%0.1f %%</td></tr>\n\n'
                        % (total_peers, size_format(total_downloaded),
                           size_format(total_uploaded), saved))

                # Write a rudimentary legend
                s.write('</table>\n' \
                    '<ul>\n' \
                    '<li><em>torrent name:</em> possible names for the torrent</li>\n' \
                    '<li><em>info hash:</em> torrent identifier</li>\n' \
                    '<li><em>peers:</em> number of connected clients</li>\n' \
                    '<li><em>downloaded:</em> amount of data downloaded by all peers, this data may have been downloaded from a mirror or another peer</li>\n' \
                    '<li><em>uploaded:</em> amount of data uploaded by all peers, this is the amount of bandwidth that was saved from the mirrors</li>\n' \
                    '<li><em>saved:</em> percentage of data that was NOT downloaded from a mirror</li>\n' \
                    '</ul>\n')

            s.write('</body>\n' \
                    '</html>\n')
            return (200, 'OK', {'Content-Type': 'text/html; charset=iso-8859-1'}, s.getvalue())
        except:
            logger.exception('Error generating info page')
            return (500, 'Internal Server Error', {'Content-Type': 'text/html; charset=iso-8859-1'}, 'Server Error')


    def scrapedata(self, hash, return_name = True):
        """Retrieve the scrape data for a single torrent.
        
        @type hash: C{string}
        @param hash: the infohash of the torrent to get scrape data for
        @type return_name: C{boolean}
        @param return_name: whether to return the name of the torrent
            (optional, defaults to True)
        @rtype: C{dictionary}
        @return: the scrape data for the torrent
        
        """
        
        l = self.downloads[hash]
        n = self.stats.get(hash, {}).get('completed', 0)
        c = self.seedcount[hash]
        d = len(l) - c
        f = {'complete': c, 'incomplete': d, 'downloaded': n}
        if return_name and self.show_names and self.config['allowed_dir']:
            f['name'] = self.allowed[hash]['name']
        return (f)

    def get_scrape(self, paramslist):
        """Get the scrape data for all the active torrents.
        
        @type paramslist: C{dictionary}
        @param paramslist: the query parameters from the GET request
        @rtype: (C{int}, C{string}, C{dictionary}, C{string})
        @return: the HTTP status code, status message, headers, and bencoded 
            message body
       
        """
        
        fs = {}
        if paramslist.has_key('info_hash'):
            if self.config['scrape_allowed'] not in ['specific', 'full']:
                return (400, 'Not Authorized', {'Content-Type': 'text/plain', 'Pragma': 'no-cache'},
                    bencode({'failure reason':
                    'specific scrape function is not available with this tracker.'}))
            for hash in paramslist['info_hash']:
                if self.allowed is not None:
                    if self.allowed.has_key(hash):
                        fs[hash] = self.scrapedata(hash)
                else:
                    if self.downloads.has_key(hash):
                        fs[hash] = self.scrapedata(hash)
        else:
            if self.config['scrape_allowed'] != 'full':
                return (400, 'Not Authorized', {'Content-Type': 'text/plain', 'Pragma': 'no-cache'},
                    bencode({'failure reason':
                    'full scrape function is not available with this tracker.'}))
            if self.allowed is not None:
                keys = self.allowed.keys()
            else:
                keys = self.downloads.keys()
            for hash in keys:
                fs[hash] = self.scrapedata(hash)

        return (200, 'OK', {'Content-Type': 'text/plain'}, bencode({'files': fs}))


    def get_file(self, hash):
        """Get the metainfo file for a torrent.
        
        @type hash: C{string}
        @param hash: the infohash of the torrent to get the metainfo of
        @rtype: (C{int}, C{string}, C{dictionary}, C{string})
        @return: the HTTP status code, status message, headers, and bencoded 
            metainfo file
        
        """
        
        if not self.allow_get:
            logger.warning('Unauthorized request for torrent file: '+b2a_hex(hash))
            return (400, 'Not Authorized', {'Content-Type': 'text/plain', 'Pragma': 'no-cache'},
                'get function is not available with this tracker.')
        if not self.allowed.has_key(hash):
            logger.warning('Request for unknown torrent file: '+b2a_hex(hash))
            return (404, 'Not Found', {'Content-Type': 'text/plain', 'Pragma': 'no-cache'}, alas)
    def cache_default(self):
        return [({},{}) for i in xrange(self.cache_default_len)]
        fname = self.allowed[hash]['file']
        fpath = self.allowed[hash]['path']
        return (200, 'OK', {'Content-Type': 'application/x-debtorrent',
            'Content-Disposition': 'attachment; filename=' + fname},
            open(fpath, 'rb').read())


    def check_allowed(self, infohash, paramslist):
        """Determine whether the tracker is tracking this torrent.
        
        @type infohash: C{string}
        @param infohash: the infohash of the torrent to check
        @type paramslist: C{dictionary}
        @param paramslist: the query parameters from the GET request
        @rtype: (C{int}, C{string}, C{dictionary}, C{string})
        @return: the HTTP status code, status message, headers, and bencoded 
            message body if the request is not allowed, or None if it is

        """
        
        if ( self.aggregator_key is not None
                and not ( paramslist.has_key('password')
                        and paramslist['password'][0] == self.aggregator_key ) ):
            return (200, 'Not Authorized', {'Content-Type': 'text/plain', 'Pragma': 'no-cache'},
                bencode({'failure reason':
                'Requested download is not authorized for use with this tracker.'}))

        if self.allowed is not None:
            if not self.allowed.has_key(infohash):
                return (200, 'Not Authorized', {'Content-Type': 'text/plain', 'Pragma': 'no-cache'},
                    bencode({'failure reason':
                    'Requested download is not authorized for use with this tracker.'}))
            if self.config['allowed_controls']:
                if self.allowed[infohash].has_key('failure reason'):
                    return (200, 'Not Authorized', {'Content-Type': 'text/plain', 'Pragma': 'no-cache'},
                        bencode({'failure reason': self.allowed[infohash]['failure reason']}))

        if paramslist.has_key('tracker'):
            if ( self.config['multitracker_allowed'] == 'none' or       # turned off
                          paramslist['peer_id'][0] == self.trackerid ): # oops! contacted myself
                return (200, 'Not Authorized', {'Content-Type': 'text/plain', 'Pragma': 'no-cache'},
                    bencode({'failure reason': 'disallowed'}))
            
            if ( self.config['multitracker_allowed'] == 'autodetect'
                        and not self.allowed[infohash].has_key('announce-list') ):
                return (200, 'Not Authorized', {'Content-Type': 'text/plain', 'Pragma': 'no-cache'},
                    bencode({'failure reason':
                    'Requested download is not authorized for multitracker use.'}))

        return None


    def add_data(self, infohash, event, ip, paramslist):
        """Add the received data from the peer to the cache.
        
        @type infohash: C{string}
        @param infohash: the infohash of the torrent to add to
        @type event: C{string}
        @param event: the type of event being reported by the peer
        @type ip: C{string}
        @param ip: the IP address of the peer
        @type paramslist: C{dictionary}
        @param paramslist: the query parameters from the GET request
        @rtype: C{int}
        @return: the number of peers to return in a peer list
        @raise ValueError: if the request from the peer is corrupt
        
        """
        
        peers = self.downloads.setdefault(infohash, {})
        ts = self.times.setdefault(infohash, {})
        stats = self.stats.setdefault(infohash, {'completed': 0,
                                                 'uploaded': 0L,
                                                 'downloaded': 0L})
        self.seedcount.setdefault(infohash, 0)
        torrent_name = self.torrent_names.setdefault(infohash, [])

        def params(key, default = None, l = paramslist):
            """Get the user parameter, or the default.
            
            @type key: C{string}
            @param key: the parameter to get
            @type default: C{string}
            @param default: the default value to use if no parameter is set
                (optional, defaults to None)
            @type l: C{dictionary}
            @param l: the user parameters (optional, defaults to L{paramslist})
            @rtype: C{string}
            @return: the parameter's value
            
            """
            
            if l.has_key(key):
                return l[key][0]
            return default
        
        myid = params('peer_id','')
        if len(myid) != 20:
            raise ValueError, 'id not of length 20'
        if event not in ['started', 'completed', 'stopped', 'snooped', None]:
            raise ValueError, 'invalid event'
        port = params('cryptoport')
        if port is None:
            port = params('port','')
        port = long(port)
        if port < 0 or port > 65535:
            raise ValueError, 'invalid port'
        name = params('name')
        left = long(params('left',''))
        if left < 0:
            raise ValueError, 'invalid amount left'
        uploaded = long(params('uploaded', 0L))
        downloaded = long(params('downloaded', 0L))
        if params('supportcrypto'):
            supportcrypto = 1
            try:
                s = int(params['requirecrypto'])
                chr(s)
            except:
                s = 0
            requirecrypto = s
        else:
            supportcrypto = 0
            requirecrypto = 0

        peer = peers.get(myid)
        islocal = local_IPs.includes(ip)
        mykey = params('key')
        if peer:
            auth = peer.get('key',-1) == mykey or peer.get('ip') == ip

        gip = params('ip')
        if is_valid_ip(gip) and (islocal or not self.only_local_override_ip):
            ip1 = gip
        else:
            ip1 = ip

        if params('numwant') is not None:
            rsize = min(int(params('numwant')),self.response_size)
        else:
            rsize = self.response_size

        if event == 'stopped':
            if peer:
                if uploaded > peer.setdefault('uploaded', uploaded):
                    stats['uploaded'] += uploaded - peer['uploaded']
                    peer['uploaded'] = uploaded
                if downloaded > peer.setdefault('downloaded', downloaded):
                    stats['downloaded'] += downloaded - peer['downloaded']
                    peer['downloaded'] = downloaded

                if auth:
                    self.delete_peer(infohash,myid)
        
        elif not peer:
            ts[myid] = clock()
            peer = { 'ip': ip, 'port': port, 'left': left,
                     'uploaded': uploaded, 'downloaded': downloaded,
                     'supportcrypto': supportcrypto,
                     'requirecrypto': requirecrypto }
            if mykey:
                peer['key'] = mykey
            if gip:
                peer['given ip'] = gip
            if name:
                peer['name'] = name
                safe_name = clean_name(name)
                if safe_name and safe_name not in torrent_name:
                    torrent_name.append(safe_name)
            if port:
                if not self.natcheck or islocal:
                    peer['nat'] = 0
                    self.natcheckOK(infohash,myid,ip1,port,peer)
                else:
                    NatCheck(self.connectback_result,infohash,myid,ip1,port,
                             self.rawserver,encrypted=requirecrypto)
            else:
                peer['nat'] = 2**30
            if event == 'completed':
                stats['completed'] += 1
            if not left:
                self.seedcount[infohash] += 1
                
            peers[myid] = peer

        else:
            if not auth:
                return rsize    # return w/o changing stats

            ts[myid] = clock()
            if not left and peer['left']:
                stats['completed'] += 1
                self.seedcount[infohash] += 1
                if not peer.get('nat', -1):
                    for bc in self.becache[infohash]:
                        if bc[0].has_key(myid):
                            bc[1][myid] = bc[0][myid]
                            del bc[0][myid]
            elif left and not peer['left']:
                stats['completed'] -= 1
                self.seedcount[infohash] -= 1
                if not peer.get('nat', -1):
                    for bc in self.becache[infohash]:
                        if bc[1].has_key(myid):
                            bc[0][myid] = bc[1][myid]
                            del bc[1][myid]
            peer['left'] = left
            
            if uploaded > peer.setdefault('uploaded', uploaded):
                stats['uploaded'] += uploaded - peer['uploaded']
                peer['uploaded'] = uploaded
            if downloaded > peer.setdefault('downloaded', downloaded):
                stats['downloaded'] += downloaded - peer['downloaded']
                peer['downloaded'] = downloaded

            if port:
                recheck = False
                if ip != peer['ip']:
                    peer['ip'] = ip
                    recheck = True
                if gip != peer.get('given ip'):
                    if gip:
                        peer['given ip'] = gip
                    elif peer.has_key('given ip'):
                        del peer['given ip']
                    recheck = True

                natted = peer.get('nat', -1)
                if recheck:
                    if natted == 0:
                        l = self.becache[infohash]
                        y = not peer['left']
                        for x in l:
                            if myid in x[y]:
                                del x[y][myid]
                    if natted >= 0:
                        del peer['nat'] # restart NAT testing
                if natted and natted < self.natcheck:
                    recheck = True

                if recheck:
                    if not self.natcheck or islocal:
                        peer['nat'] = 0
                        self.natcheckOK(infohash,myid,ip1,port,peer)
                    else:
                        NatCheck(self.connectback_result,infohash,myid,ip1,port,
                                 self.rawserver,encrypted=requirecrypto)

        return rsize


    def peerlist(self, infohash, stopped, tracker, is_seed,
                 return_type, rsize, supportcrypto):
        """Create a list of peers to return to the client.
        
        @type infohash: C{string}
        @param infohash: the infohash of the torrent to get peers from
        @type stopped: C{boolean}
        @param stopped: whether the peer has stopped
        @type tracker: C{boolean}
        @param tracker: whether the peer is a tracker
        @type is_seed: C{boolean}
        @param is_seed: whether the peer is currently seeding (currently ignored)
        @type return_type: C{int}
        @param return_type: the format of the list to return (compact, ...)
        @type rsize: C{int}
        @param rsize: the number of peers to return
        @type supportcrypto: C{boolean}
        @param supportcrypto: whether the peer supports encrypted communication
            (not used)
        @rtype: C{dictionary}
        @return: the info to return to the client
        
        """
        
        is_seed = False;
        data = {}    # return data
        seeds = self.seedcount[infohash]
        data['complete'] = seeds
        data['incomplete'] = len(self.downloads[infohash]) - seeds
        
        if ( self.config['allowed_controls']
                and self.allowed[infohash].has_key('warning message') ):
            data['warning message'] = self.allowed[infohash]['warning message']

        if tracker:
            data['interval'] = self.config['multitracker_reannounce_interval']
            if not rsize:
                return data
            cache = self.cached_t.setdefault(infohash, None)
            if ( not cache or len(cache[1]) < rsize
                 or cache[0] + self.config['min_time_between_cache_refreshes'] < clock() ):
                bc = self.becache.setdefault(infohash,self.cache_default())
                cache = [ clock(), bc[0][0].values() + bc[0][1].values() ]
                self.cached_t[infohash] = cache
                shuffle(cache[1])
                cache = cache[1]

            data['peers'] = cache[-rsize:]
            del cache[-rsize:]
            return data

        data['interval'] = self.reannounce_interval
        if stopped or not rsize:     # save some bandwidth
            data['peers'] = []
            return data

        bc = self.becache.setdefault(infohash,self.cache_default())
        len_l = len(bc[2][0])
        len_s = len(bc[2][1])
        if not (len_l+len_s):   # caches are empty!
            data['peers'] = []
            return data
        l_get_size = int(float(rsize)*(len_l)/(len_l+len_s))
        cache = self.cached.setdefault(infohash,[None,None,None])[return_type]
        if cache and ( not cache[1]
                       or (is_seed and len(cache[1]) < rsize)
                       or len(cache[1]) < l_get_size
                       or cache[0]+self.config['min_time_between_cache_refreshes'] < self.cachetime ):
            cache = None
        if not cache:
            peers = self.downloads[infohash]
            if self.config['compact_reqd']:
                vv = ([],[],[])
            else:
                vv = ([],[],[],[],[])
            for key, ip, port in self.t2tlist.harvest(infohash):   # empty if disabled
                if not peers.has_key(key):
                    cp = compact_peer_info(ip, port)
                    vv[0].append(cp)
                    vv[2].append((cp,'\x00'))
                    if not self.config['compact_reqd']:
                        vv[3].append({'ip': ip, 'port': port, 'peer id': key})
                        vv[4].append({'ip': ip, 'port': port})
            cache = [ self.cachetime,
                      bc[return_type][0].values()+vv[return_type],
                      bc[return_type][1].values() ]
            shuffle(cache[1])
            shuffle(cache[2])
            self.cached[infohash][return_type] = cache
            for rr in xrange(len(self.cached[infohash])):
                if rr != return_type:
                    try:
                        self.cached[infohash][rr][1].extend(vv[rr])
                    except:
                        pass
        if len(cache[1]) < l_get_size:
            peerdata = cache[1]
            if not is_seed:
                peerdata.extend(cache[2])
            cache[1] = []
            cache[2] = []
        else:
            if not is_seed:
                peerdata = cache[2][l_get_size-rsize:]
                del cache[2][l_get_size-rsize:]
                rsize -= len(peerdata)
            else:
                peerdata = []
            if rsize:
                peerdata.extend(cache[1][-rsize:])
                del cache[1][-rsize:]
        if return_type == 0:
            data['peers'] = ''.join(peerdata)
        elif return_type == 1:
            data['crypto_flags'] = "0x01"*len(peerdata)
            data['peers'] = ''.join(peerdata)
        elif return_type == 2:
            data['crypto_flags'] = ''.join([p[1] for p in peerdata])
            data['peers'] = ''.join([p[0] for p in peerdata])
        else:
            data['peers'] = peerdata
        return data


    def get(self, connection, path, headers, httpreq):
        """Respond to a GET request to the tracker.
        
        Process a GET request from a peer/tracker/browser. Process the request,
        calling the helper functions above if needed. Return the response to
        be returned to the requester.
        
        @type connection: L{DebTorrent.HTTPHandler.HTTPConnection}
        @param connection: the conection the request came in on
        @type path: C{string}
        @param path: the URL being requested
        @type headers: C{dictionary}
        @param headers: the headers from the request
        @type httpreq: L{DebTorrent.HTTPHandler.HTTPRequest}
        @param httpreq: not used since HTTP 1.1 is not used by the tracker
        @rtype: (C{int}, C{string}, C{dictionary}, C{string})
        @return: the HTTP status code, status message, headers, and message body
        
        """
        
        real_ip = connection.get_ip()
        ip = real_ip
        if is_ipv4(ip):
            ipv4 = True
        else:
            try:
                ip = ipv6_to_ipv4(ip)
                ipv4 = True
            except ValueError:
                ipv4 = False

        if ( (self.allowed_IPs and not self.allowed_IPs.includes(ip))
             or (self.banned_IPs and self.banned_IPs.includes(ip)) ):
            logger.warning('Unauthorized request from '+ip+' for: '+path)
            return (400, 'Not Authorized', {'Content-Type': 'text/plain', 'Pragma': 'no-cache'},
                bencode({'failure reason':
                'your IP is not allowed on this tracker'}))

        nip = get_forwarded_ip(headers)
        if nip and not self.only_local_override_ip:
            ip = nip
            try:
                ip = to_ipv4(ip)
                ipv4 = True
            except ValueError:
                ipv4 = False

        paramslist = {}
        def params(key, default = None, l = paramslist):
            """Get the user parameter, or the default.
            
            @type key: C{string}
            @param key: the parameter to get
            @type default: C{string}
            @param default: the default value to use if no parameter is set
                (optional, defaults to None)
            @type l: C{dictionary}
            @param l: the user parameters (optional, defaults to L{paramslist})
            @rtype: C{string}
            @return: the parameter's value
            
            """
            
            if l.has_key(key):
                return l[key][0]
            return default

        try:
            (scheme, netloc, path, pars, query, fragment) = urlparse(path)
            if self.uq_broken == 1:
                path = path.replace('+',' ')
                query = query.replace('+',' ')
            path = unquote(path)[1:]
            for s in query.split('&'):
                if s:
                    i = s.index('=')
                    kw = unquote(s[:i])
                    paramslist.setdefault(kw, [])
                    paramslist[kw] += [unquote(s[i+1:])]
                    
            if path == '' or path == 'index.html':
                return self.get_infopage()
            if (path == 'file'):
                return self.get_file(params('info_hash'))
            if path == 'favicon.ico' and self.favicon is not None:
                return (200, 'OK', {'Content-Type' : 'image/x-icon'}, self.favicon)

            # automated access from here on

            if path in ('scrape', 'scrape.php', 'tracker.php/scrape'):
                return self.get_scrape(paramslist)
            
            if not path in ('announce', 'announce.php', 'tracker.php/announce'):
                return (404, 'Not Found', {'Content-Type': 'text/plain', 'Pragma': 'no-cache'}, alas)

            # main tracker function

            filtered = self.Filter.check(real_ip, paramslist, headers)
            if filtered:
                return (400, 'Not Authorized', {'Content-Type': 'text/plain', 'Pragma': 'no-cache'},
                    bencode({'failure reason': filtered}))
            
            infohash = params('info_hash')
            if not infohash:
                raise ValueError, 'no info hash'

            notallowed = self.check_allowed(infohash, paramslist)
            if notallowed:
                return notallowed

            event = params('event')

            rsize = self.add_data(infohash, event, ip, paramslist)

        except ValueError, e:
            logger.exception('Bad request from: '+ip)
            return (400, 'Bad Request', {'Content-Type': 'text/plain'}, 
                'you sent me garbage - ' + str(e))

        if self.aggregate_forward and not paramslist.has_key('tracker'):
            self.aggregate_senddata(query)

        if self.is_aggregator:      # don't return peer data here
            return (200, 'OK', {'Content-Type': 'text/plain', 'Pragma': 'no-cache'},
                    bencode({'response': 'OK'}))

        if params('compact') and ipv4:
            if params('requirecrypto'):
                return_type = 1
            elif params('supportcrypto'):
                return_type = 2
            else:
                return_type = 0
        elif self.config['compact_reqd'] and ipv4:
            return (400, 'Bad Request', {'Content-Type': 'text/plain'}, 
                'your client is outdated, please upgrade')
        elif params('no_peer_id'):
            return_type = 4
        else:
            return_type = 3
            
        data = self.peerlist(infohash, event=='stopped',
                             params('tracker'), not params('left'),
                             return_type, rsize, params('supportcrypto'))

        if paramslist.has_key('scrape'):    # deprecated
            data['scrape'] = self.scrapedata(infohash, False)

        if self.dedicated_seed_id:
            if params('seed_id') == self.dedicated_seed_id and params('left') == 0:
                self.is_seeded[infohash] = True
            if params('check_seeded') and self.is_seeded.get(infohash):
                data['seeded'] = 1
            
        return (200, 'OK', {'Content-Type': 'text/plain', 'Pragma': 'no-cache'}, bencode(data))


    def natcheckOK(self, infohash, peerid, ip, port, peer):
        """Add the unNATed peer to the cache.
        
        @type infohash: C{string}
        @param infohash: the infohash of the torrent the peer is in
        @type peerid: C{string}
        @param peerid: the peer ID of the peer
        @type ip: C{string}
        @param ip: the IP address of the peer
        @type port: C{int}
        @param port: the port to contact the peer on
        @type peer: C{dictionary}
        @param peer: various information about the peer
        
        """
        
        seed = not peer['left']
        bc = self.becache.setdefault(infohash,self.cache_default())
        cp = compact_peer_info(ip, port)
        reqc = peer['requirecrypto']
        bc[2][seed][peerid] = (cp,chr(reqc))
        if peer['supportcrypto']:
            bc[1][seed][peerid] = cp
        if not reqc:
            bc[0][seed][peerid] = cp
            if not self.config['compact_reqd']:
                bc[3][seed][peerid] = Bencached(bencode({'ip': ip, 'port': port,
                                                         'peer id': peerid}))
                bc[4][seed][peerid] = Bencached(bencode({'ip': ip, 'port': port}))


    def connectback_result(self, result, downloadid, peerid, ip, port):
        """Process a NAT check attempt and result.
        
        @type result: C{boolean}
        @param result: whether the NAT check was successful
        @type downloadid: C{string}
        @param downloadid: the infohash of the torrent the peer is in
        @type peerid: C{string}
        @param peerid: the peer ID of the peer
        @type ip: C{string}
        @param ip: the IP address of the peer
        @type port: C{int}
        @param port: the port to contact the peer on
        
        """
        
        record = self.downloads.get(downloadid,{}).get(peerid)
        if ( record is None 
                 or (record['ip'] != ip and record.get('given ip') != ip)
                 or record['port'] != port ):
            logger.info('natcheck on '+ip+' ('+quote(peerid)+'), '+str(port)+': '+str(404))
            return
        if self.config['log_nat_checks']:
            if result:
                x = 200
            else:
                x = 503
            logger.info('natcheck on '+ip+' ('+quote(peerid)+'), '+str(port)+': '+str(x))
        if not record.has_key('nat'):
            record['nat'] = int(not result)
            if result:
                self.natcheckOK(downloadid,peerid,ip,port,record)
        elif result and record['nat']:
            record['nat'] = 0
            self.natcheckOK(downloadid,peerid,ip,port,record)
        elif not result:
            record['nat'] += 1


    def remove_from_state(self, *l):
        """Remove all the input parameter names from the current state."""
        for s in l:
            try:
                del self.state[s]
            except:
                pass

    def save_state(self):
        """Save the state file to disk."""
        self.rawserver.add_task(self.save_state, self.save_state_interval)
        self.configdir.saveState(self.state)


    def parse_allowed(self):
        """Periodically parse the directory and list for allowed torrents."""
        self.rawserver.add_task(self.parse_allowed, self.parse_dir_interval)

        if self.config['allowed_dir']:
            r = parsedir( self.config['allowed_dir'], self.allowed,
                          self.allowed_dir_files, self.allowed_dir_blocked,
                          [".dtorrent"] )
            ( self.allowed, self.allowed_dir_files, self.allowed_dir_blocked,
                added, garbage2 ) = r
            
            self.state['allowed'] = self.allowed
            self.state['allowed_dir_files'] = self.allowed_dir_files

            self.t2tlist.parse(self.allowed)
            
        else:
            f = self.config['allowed_list']
            if self.allowed_list_mtime == os.path.getmtime(f):
                return
            try:
                r = parsetorrentlist(f, self.allowed)
                (self.allowed, added, garbage2) = r
                self.state['allowed_list'] = self.allowed
            except (IOError, OSError):
                logger.exception('unable to read allowed torrent list')
                return
            self.allowed_list_mtime = os.path.getmtime(f)

        for infohash in added.keys():
            self.downloads.setdefault(infohash, {})
            self.completed.setdefault(infohash, 0)
            self.seedcount.setdefault(infohash, 0)


    def read_ip_lists(self):
        """Periodically parse the allowed and banned IPs lists."""
        self.rawserver.add_task(self.read_ip_lists,self.parse_dir_interval)
            
        f = self.config['allowed_ips']
        if f and self.allowed_ip_mtime != os.path.getmtime(f):
            self.allowed_IPs = IP_List()
            try:
                self.allowed_IPs.read_fieldlist(f)
                self.allowed_ip_mtime = os.path.getmtime(f)
            except (IOError, OSError):
                logger.exception('unable to read allowed_IP list')
                
        f = self.config['banned_ips']
        if f and self.banned_ip_mtime != os.path.getmtime(f):
            self.banned_IPs = IP_Range_List()
            try:
                self.banned_IPs.read_rangelist(f)
                self.banned_ip_mtime = os.path.getmtime(f)
            except (IOError, OSError):
                logger.exception('unable to read banned_IP list')
                

    def delete_peer(self, infohash, peerid):
        """Delete all cached data for the peer.
        
        @type infohash: C{string}
        @param infohash: the infohash of the torrent to delete the peer from
        @type peerid: C{string}
        @param peerid: the peer ID of the peer to delete
        
        """
        
        dls = self.downloads[infohash]
        peer = dls[peerid]
        if not peer['left']:
            self.seedcount[infohash] -= 1
        if not peer.get('nat',-1):
            l = self.becache[infohash]
            y = not peer['left']
            for x in l:
                if x[y].has_key(peerid):
                    del x[y][peerid]
        del self.times[infohash][peerid]
        del dls[peerid]

    def expire_downloaders(self):
        """Periodically remove all old downloaders from the cached data."""
        for x in self.times.keys():
            for myid, t in self.times[x].items():
                if t < self.prevtime:
                    self.delete_peer(x,myid)
        self.prevtime = clock()
        if (self.keep_dead != 1):
            for key, value in self.downloads.items():
                if len(value) == 0 and (
                        self.allowed is None or not self.allowed.has_key(key) ):
                    del self.times[key]
                    del self.downloads[key]
                    del self.seedcount[key]
        self.rawserver.add_task(self.expire_downloaders, self.timeout_downloaders_interval)


def track(params):
    """Start the server and tracker.
    
    @type params: C{list}
    @param params: the command line arguments to the tracker
    @rtype: C{boolean}
    @return: whether the server should be restarted
    
    """
    
    restart = False
    configdefaults = {}
    try:
        # Load the configuration data
        configdir = ConfigDir('debtorrent-tracker')
        defaultsToIgnore = ['configfile']
        configdir.setDefaults(defaults,defaultsToIgnore)
        configdefaults = configdir.loadConfig(params)
        defaults.append(('save_options',0,
            "whether to save the current options as the new default configuration " +
            "(only for debtorrent-tracker.py)"))
        config, files = parseargs(params, defaults, 0, 0, configdefaults)
        configdir.setCacheDir(config['cache_dir'], False)
        if config['log_dir']:
            outfilename = os.path.join(config['log_dir'], 'debtorrent-tracker.log')
            logfile = os.path.join(config['log_dir'], 'tracker-access.log')
        else:
            outfilename = os.path.join(configdir.cache_dir, 'debtorrent-tracker.log')
            logfile = os.path.join(configdir.cache_dir, 'tracker-access.log')

        # Create the root handler (removing any others)
        hdlr = logging.FileHandler(outfilename, 'a')
        hdlr.setFormatter(logging.Formatter('%(asctime)s %(threadName)s %(name)s %(levelname)s %(message)s'))
        for h in logging.root.handlers:
            logging.root.removeHandler(h)
        logging.root.addHandler(hdlr)
        logging.root.setLevel(config['log_level'])
        logger.info('Logging begins')
        
        # Continue
        if config['save_options']:
            configdir.saveConfig(config)
    except ValueError, e:
        logger.error('error: ' + str(e))
        logger.error("Usage: debtorrent-tracker.py <global options>")
        logger.error(formatDefinitions(defaults, 80))
        logging.shutdown()
        sys.exit(1)
    except:
        logger.exception('unhandled exception')

    try:
        r = RawServer(Event(), config['timeout_check_interval'],
                      config['socket_timeout'], ipv6_enable = config['ipv6_enabled'])
        
        t = Tracker(config, r, configdir)
        
        r.bind(config['port'], config['bind'],
               reuse = True, ipv6_socket_style = config['ipv6_binds_v4'])
    
        restart = r.listen_forever(HTTPHandler(t.get,
                                config['min_time_between_log_flushes'],
                                logfile, config['hupmonitor']))
        
        t.save_state()

        r.shutdown()
    except:
        logger.exception('unhandled exception')

    logger.info('Shutting down')
    logging.shutdown()
    return restart

def size_format(s):
    """Format a byte size for reading by the user.
    
    @type s: C{long}
    @param s: the number of bytes
    @rtype: C{string}
    @return: the formatted size with appropriate units
    
    """
    
    if (s < 1024):
        r = str(s) + 'B'
    elif (s < 10485):
        r = str(int((s/1024.0)*100.0)/100.0) + 'KiB'
    elif (s < 104857):
        r = str(int((s/1024.0)*10.0)/10.0) + 'KiB'
    elif (s < 1048576):
        r = str(int(s/1024)) + 'KiB'
    elif (s < 10737418L):
        r = str(int((s/1048576.0)*100.0)/100.0) + 'MiB'
    elif (s < 107374182L):
        r = str(int((s/1048576.0)*10.0)/10.0) + 'MiB'
    elif (s < 1073741824L):
        r = str(int(s/1048576)) + 'MiB'
    elif (s < 1099511627776L):
        r = str(int((s/1073741824.0)*100.0)/100.0) + 'GiB'
    else:
        r = str(int((s/1099511627776.0)*100.0)/100.0) + 'TiB'
    return(r)

