# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import gtk, random

from gtk     import gdk
from gobject import signal_new, TYPE_INT, TYPE_LONG, TYPE_PYOBJECT, TYPE_NONE, SIGNAL_RUN_LAST


# Identifiers of accepted DND targets
(
    DND_INTERNAL,
    DND_EXTERNAL_URI
) = range(2)


# DND targets that this tree may accept
DND_TARGETS = {
                DND_INTERNAL:     ('internal',      gtk.TARGET_SAME_WIDGET, DND_INTERNAL),
                DND_EXTERNAL_URI: ('text/uri-list',                      0, DND_EXTERNAL_URI)
              }


# Custom signals
signal_new('listview-dnd', gtk.TreeView, SIGNAL_RUN_LAST, TYPE_NONE, (gdk.DragContext, TYPE_INT, TYPE_INT, gtk.SelectionData, TYPE_LONG))
signal_new('listview-modified', gtk.TreeView, SIGNAL_RUN_LAST, TYPE_NONE, ())
signal_new('listview-button-pressed', gtk.TreeView, SIGNAL_RUN_LAST, TYPE_NONE, (gdk.Event, TYPE_PYOBJECT))


class ListView(gtk.TreeView):


    def __init__(self, columns):
        """ Constructor """
        gtk.TreeView.__init__(self)

        self.selection = self.get_selection()

        # Default configuration for this list
        self.set_rules_hint(True)
        self.set_headers_visible(True)
        self.selection.set_mode(gtk.SELECTION_MULTIPLE)

        self.set_headers_clickable(True)

        # Create the columns
        nbEntries = 0
        dataTypes = []
        for (title, renderers, sortIndex, expandable) in columns:
            if title is None:
                for (renderer, type) in renderers:
                    nbEntries += 1
                    dataTypes.append(type)
            else:
                column = gtk.TreeViewColumn(title)
                column.set_sizing(gtk.TREE_VIEW_COLUMN_AUTOSIZE)
                column.set_expand(expandable)
                # FIXME
                # From the PyGTK doc: "This means that once the model has been sorted, it can't go back to the default state"
                # Is this a joke???
                # column.set_sort_column_id(sortIndex)
                self.append_column(column)

                for (renderer, type) in renderers:
                    nbEntries += 1
                    dataTypes.append(type)
                    column.pack_start(renderer, True)
                    if isinstance(renderer, gtk.CellRendererText): column.add_attribute(renderer, 'text',   nbEntries-1)
                    else:                                          column.add_attribute(renderer, 'pixbuf', nbEntries-1)

        # Create the ListStore associated with this tree
        self.store = gtk.ListStore(*dataTypes)
        self.set_model(self.store)

        # Drag'n'drop management
        self.dndContext  = None
        self.dndEnabled  = False
        self.dndStartPos = None
        self.motionEvtId = None

        self.connect('drag-begin',           self.onDragBegin)
        self.connect('drag-motion',          self.onDragMotion)
        self.connect('button-press-event',   self.onButtonPressed)
        self.connect('drag-data-received',   self.onDragDataReceived)
        self.connect('button-release-event', self.onButtonReleased)

        # Mark management
        self.markedRow = None

        # Show the list
        self.show()

    # --== Miscellaneous ==--


    def shuffle(self):
        """ Shuffle the content of the list """
        order = range(len(self.store))
        random.shuffle(order)

        # Move the mark if needed
        if self.hasMark():
            for i in xrange(len(order)):
                if order[i] == self.markedRow:
                    self.markedRow = i
                    break

        self.store.reorder(order)
        self.emit('listview-modified')


    def __getIterOnSelectedRows(self):
        """ Return a list of iterators pointing to the selected rows """
        return [self.store.get_iter(path) for path in self.selection.get_selected_rows()[1]]


    # --== Mark management ==--


    def setMark(self, rowIndex):
        """ Put the mark on the given row, it will move with the row itself (e.g., D'n'D) """
        self.markedRow = rowIndex


    def clearMark(self):
        """ Remove the mark """
        self.markedRow = None


    def hasMark(self):
        """ True if a mark has been set """
        return self.markedRow is not None


    def getMark(self):
        """ Return the index of the marked row """
        return self.markedRow


    # --== Retrieving content ==--


    def getCount(self):
        """ Return how many rows are stored in the list """
        return len(self.store)


    def getRow(self, rowIndex):
        """ Return the given row """
        return tuple(self.store[rowIndex])


    def getAllRows(self):
        """ Return all rows """
        return [tuple(row) for row in self.store]


    def getItem(self, rowIndex, colIndex):
        """ Return the value of the given item """
        return self.store.get_value(self.store.get_iter(rowIndex), colIndex)


    # --== Adding/removing content ==--


    def clear(self):
        """ Remove all rows from the list """
        self.clearMark()
        self.store.clear()


    def getSelectedRowsCount(self):
        """ Return how many rows are currently selected """
        return self.selection.count_selected_rows()


    def getSelectedRows(self):
        """ Return all selected row(s) """
        return [tuple(self.store[path]) for path in self.selection.get_selected_rows()[1]]


    def getFirstSelectedRowIndex(self):
        """ Return the index of the first selected row """
        return self.selection.get_selected_rows()[1][0][0]


    def setItem(self, rowIndex, colIndex, value):
        """ Change the value of the given item """
        self.store.set_value(self.store.get_iter(rowIndex), colIndex, value)


    def removeSelectedRows(self):
        """ Remove the selected row(s) """
        self.freeze_child_notify()
        for iter in self.__getIterOnSelectedRows():
            # Move the mark if needed
            if self.hasMark():
                currentPath = self.store.get_path(iter)[0]
                if   currentPath < self.markedRow:  self.markedRow -= 1
                elif currentPath == self.markedRow: self.markedRow  = None
            # Remove the current row
            if   self.store.remove(iter): self.set_cursor(self.store.get_path(iter))
            elif len(self.store) != 0:    self.set_cursor(len(self.store)-1)
        self.thaw_child_notify()
        self.emit('listview-modified')


    def cropSelectedRows(self):
        """ Remove all rows but the selected ones """
        pathsList = self.selection.get_selected_rows()[1]
        self.freeze_child_notify()
        self.selection.select_all()
        for path in pathsList:
            self.selection.unselect_path(path)
        self.removeSelectedRows()
        self.selection.select_all()
        self.thaw_child_notify()


    def insertRows(self, rows, position=None):
        """ Insert or append (if position is None) some rows to the list """
        # Move the mark if needed
        if self.hasMark() and position is not None and position <= self.markedRow:
            self.markedRow += len(rows)

        # Insert the new rows
        self.freeze_child_notify()
        if position is None:
            for row in rows:
                self.store.append(row)
        else:
            for row in rows:
                self.store.insert(position, row)
                position += 1
        self.thaw_child_notify()
        self.emit('listview-modified')


    # --== D'n'D management ==--


    def enableDND(self):
        """ Enable Drag'n'Drop for this list """
        self.dndEnabled = True
        self.enable_model_drag_dest(DND_TARGETS.values(), gdk.ACTION_DEFAULT)


    def __isDropBefore(self, pos):
        """ Helper function, True if pos is gtk.TREE_VIEW_DROP_BEFORE or gtk.TREE_VIEW_DROP_INTO_OR_BEFORE """
        return pos == gtk.TREE_VIEW_DROP_BEFORE or pos == gtk.TREE_VIEW_DROP_INTO_OR_BEFORE


    def __moveSelectedRows(self, x, y):
        """ Internal function used for drag'n'drop """
        iterList = self.__getIterOnSelectedRows()
        dropInfo = self.get_dest_row_at_pos(int(x), int(y))

        if dropInfo is None: pos, path = gtk.TREE_VIEW_DROP_INTO_OR_AFTER, len(self.store) - 1
        else:                pos, path = dropInfo[1], dropInfo[0][0]

        self.freeze_child_notify()
        for srcIter in iterList:
            srcPath = self.store.get_path(srcIter)[0]

            if self.__isDropBefore(pos): dstIter = self.store.insert_before(self.store.get_iter(path), self.store[srcIter])
            else:                        dstIter = self.store.insert_after(self.store.get_iter(path),  self.store[srcIter])

            self.store.remove(srcIter)
            dstPath = self.store.get_path(dstIter)[0]

            if srcPath > dstPath:
                path += 1

            if self.markedRow is not None:
                if   srcPath == self.markedRow:                              self.markedRow  = dstPath
                elif srcPath < self.markedRow and dstPath >= self.markedRow: self.markedRow -= 1
                elif srcPath > self.markedRow and dstPath <= self.markedRow: self.markedRow += 1
        self.thaw_child_notify()
        self.emit('listview-modified')


    # --== GTK Handlers ==--


    def onButtonPressed(self, tree, event):
        """ A mouse button has been pressed """
        retVal   = False
        pathInfo = self.get_path_at_pos(int(event.x), int(event.y))

        if pathInfo is None: path = None
        else:                path = pathInfo[0]

        if event.button == 1 or event.button == 3:
            if path is None:
                self.selection.unselect_all()
            else:
                if self.dndEnabled and self.motionEvtId is None and event.button == 1:
                    self.dndStartPos = (int(event.x), int(event.y))
                    self.motionEvtId = gtk.TreeView.connect(self, 'motion-notify-event', self.onMouseMotion)

                if event.state == 0 and not self.selection.path_is_selected(path):
                    self.selection.unselect_all()
                    self.selection.select_path(path)
                else:
                    retVal = (event.state == 0 and self.getSelectedRowsCount() > 1 and self.selection.path_is_selected(path))

        self.emit('listview-button-pressed', event, path)

        return retVal


    def onButtonReleased(self, tree, event):
        """ A mouse button has been released """
        if self.motionEvtId is not None:
            self.disconnect(self.motionEvtId)
            self.dndContext  = None
            self.motionEvtId = None

            if self.dndEnabled:
                self.enable_model_drag_dest(DND_TARGETS.values(), gdk.ACTION_DEFAULT)

        if event.state == gtk.gdk.BUTTON1_MASK and self.getSelectedRowsCount() > 1:
            pathInfo = self.get_path_at_pos(int(event.x), int(event.y))
            if pathInfo is not None:
                self.selection.unselect_all()
                self.selection.select_path(pathInfo[0][0])


    def onMouseMotion(self, tree, event):
        """ The mouse has been moved """
        if self.dndContext is None and self.drag_check_threshold(self.dndStartPos[0], self.dndStartPos[1], int(event.x), int(event.y)):
            self.dndContext = self.drag_begin([DND_TARGETS[DND_INTERNAL]], gtk.gdk.ACTION_COPY, 1, event)


    def onDragBegin(self, tree, context):
        """ A drag'n'drop operation has begun """
        if self.getSelectedRowsCount() == 1: context.set_icon_stock(gtk.STOCK_DND,          0, 0)
        else:                                context.set_icon_stock(gtk.STOCK_DND_MULTIPLE, 0, 0)


    def onDragDataReceived(self, tree, context, x, y, selection, dndId, time):
        """ Some data has been dropped into the list """
        if   dndId == DND_INTERNAL:     self.__moveSelectedRows(x, y)
        elif dndId == DND_EXTERNAL_URI: self.emit('listview-dnd', context, int(x), int(y), selection, time)


    def onDragMotion(self, tree, context, x, y, time):
        """ Prevent rows from being dragged *into* other rows (this is a list, not a tree) """
        drop = self.get_dest_row_at_pos(int(x), int(y))

        if drop is not None and (drop[1] == gtk.TREE_VIEW_DROP_INTO_OR_AFTER or drop[1] == gtk.TREE_VIEW_DROP_INTO_OR_BEFORE):
            self.enable_model_drag_dest([('invalid-position', 0, -1)], gdk.ACTION_DEFAULT)
        else:
            self.enable_model_drag_dest(DND_TARGETS.values(), gdk.ACTION_DEFAULT)
