# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import cgi, gtk, modules, os.path

from gui        import infoMsgBox
from tools      import consts, prefs
from media      import track
from modules    import ThreadedModule
from gettext    import gettext as _
from gui.window import Window


# Module configuration
MOD_NAME            = _('Desktop Notification')
MOD_DESC            = _('Display a desktop notification on track change')
MOD_DEPS            = ['pynotify']
MOD_IS_MANDATORY    = False
MOD_IS_CONFIGURABLE = True


# Default preferences
PREFS_DEFAULT_BODY    = 'by {artist}\nfrom {album}\n\nTrack {playlist_pos} out of {playlist_len}'
PREFS_DEFAULT_TITLE   = '<b>{title}</b>  [{duration_str}]'
PREFS_DEFAULT_TIMEOUT = 10


class DesktopNotification(ThreadedModule):


    def __init__(self, wTree):
        """ Constructor """
        import pynotify

        ThreadedModule.__init__(self)
        # Widgets
        self.notif  = None
        self.window = None
        # It seems that this call must be done in the GTK main loop
        pynotify.init(consts.appNameShort)
        modules.register(self, [modules.MSG_EVT_NEW_TRACK, modules.MSG_EVT_APP_QUIT, modules.MSG_EVT_MOD_UNLOADED, modules.MSG_EVT_STOPPED])


    def configure(self, parent):
        """ Show the configuration window """
        if self.window is None:
            self.window      = Window('DesktopNotification.glade', 'vbox1', __name__, MOD_NAME, 355, 345)
            self.btnOk       = self.window.getWidget('btn-ok')
            self.txtBody     = self.window.getWidget('txt-body')
            self.txtTitle    = self.window.getWidget('txt-title')
            self.spnDuration = self.window.getWidget('spn-duration')
            # GTK handlers
            self.window.getWidget('btn-help').connect('clicked', self.onHelp)
            self.window.getWidget('btn-cancel').connect('clicked', lambda btn: self.window.hide())
            self.btnOk.connect('clicked', self.onOk)

        # Fill current preferences
        if not self.window.isVisible():
            self.txtTitle.set_text(prefs.get(__name__, 'title', PREFS_DEFAULT_TITLE))
            self.txtBody.get_buffer().set_text(prefs.get(__name__, 'body', PREFS_DEFAULT_BODY))
            self.spnDuration.set_value(prefs.get(__name__, 'timeout', PREFS_DEFAULT_TIMEOUT))
            self.btnOk.grab_focus()

        self.window.show()


    def showNotification(self, track):
        """ Show the notification based on the given track """
        import pynotify

        icon  = os.path.join(consts.dirPix, 'decibel-audio-player-64.png')
        body  = track.formatHTMLSafe(prefs.get(__name__, 'body', PREFS_DEFAULT_BODY))
        title = '</b>%s<b>' % track.formatHTMLSafe(prefs.get(__name__, 'title', PREFS_DEFAULT_TITLE))

        if not os.path.exists(icon):
            icon = gtk.STOCK_DIALOG_INFO

        if self.notif is not None:
            self.notif.update(title, body, icon)
        else:
            self.notif = pynotify.Notification(title, body, icon)
            self.notif.set_timeout(prefs.get(__name__, 'timeout', PREFS_DEFAULT_TIMEOUT) * 1000)

        self.notif.show()


    # --== GTK handlers ==--


    def onHelp(self, btn):
        """ Display a small help message box """
        helpMsg  = _('Any field of the form <b>{<tt>field</tt>}</b> will be replaced by the corresponding value. Available fields are:\n\n')
        helpMsg += track.getFormatSpecialFields()
        infoMsgBox(self.window, helpMsg)


    def onOk(self, btn):
        """ Save the new preferences """
        prefs.set(__name__, 'title', self.txtTitle.get_text())
        prefs.set(__name__, 'timeout', int(self.spnDuration.get_value()))
        (start, end) = self.txtBody.get_buffer().get_bounds()
        prefs.set(__name__, 'body', self.txtBody.get_buffer().get_text(start, end))
        self.window.hide()


    # --== Message handler ==--


    def handleMsg(self, msg, params):
        """ Handle messages sent to this module """
        if msg == modules.MSG_EVT_NEW_TRACK:
            self.showNotification(params['track'])
        elif msg == modules.MSG_EVT_APP_QUIT or msg == modules.MSG_EVT_MOD_UNLOADED or msg == modules.MSG_EVT_STOPPED:
            if self.notif is not None:
                self.notif.close()
