# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import gtk, modules

from gtk     import gdk
from tools   import consts, prefs
from modules import Module
from gettext import gettext as _
from gobject import TYPE_PYOBJECT, TYPE_INT, TYPE_BOOLEAN, TYPE_STRING


# The format of a combo box entry
(
    EXP_NAME,         # Displayed name
    EXP_ICON,         # Displayed icon
    EXP_MOD_NAME,     # The module that manages the explorer
    EXP_NUM_PAGE,     # The page associated with this explorer
    EXP_IS_SEPARATOR, # True if the entry is a separator
    EXP_USR_PARAM     # Arbitrary user parameter
) = range(6)


# Default preferences
DEFAULT_LAST_EXPLORER = ''


class Explorer(Module):
    """
        This module manages the left part that allows the user to browse media files (e.g., file explorer, library explorer, radio streams)
        It adds/remove entries to the combo box and pages to the notebook as requested by other modules
        The tabs of the notebook are not shown, the switch is automatically done by this module based on the combo box' selection
    """


    def __init__(self, wTree):
        """ Constructor """
        Module.__init__(self)
        self.modules   = {}   # For each module, the names of all explorers it has registered
        self.widgets   = {}   # For each widget, the page number used to display it in the notebook
        self.explorers = {}   # For each (modName, expName), the data associated with it (e.g., icon, usrParam)
        # Widgets
        self.combo    = wTree.get_widget('combo-explorer')
        self.notebook = wTree.get_widget('notebook-explorer')
        # Rendering of the combo box
        txtRdr     = gtk.CellRendererText()
        imgRdr     = gtk.CellRendererPixbuf()
        self.store = gtk.ListStore(TYPE_STRING, gdk.Pixbuf, TYPE_PYOBJECT, TYPE_INT, TYPE_BOOLEAN, TYPE_STRING)
        self.combo.pack_start(imgRdr, False)
        self.combo.add_attribute(imgRdr, 'pixbuf', EXP_ICON)
        self.combo.pack_start(txtRdr, True)
        self.combo.add_attribute(txtRdr, 'text', EXP_NAME)
        self.combo.set_sensitive(False)
        self.combo.set_model(self.store)
        # Add a default page, displayed when nothing is selected in the combo box
        label = gtk.Label(_('Please select an explorer\nin the combo box below.'))
        label.show()
        self.notebook.append_page(label)
        # Messages handled by this module
        modules.register(self, [modules.MSG_CMD_EXPLORER_ADD, modules.MSG_CMD_EXPLORER_REMOVE])
        # GTK handlers
        self.changedHandler = self.combo.connect('changed', self.onExplorerChanged)


    def addExplorer(self, modName, expName, icon, widget, usrParam):
        """ Add a new explorer, create a page in the notebook for the widget if needed (one page per widget) """
        if icon is None:
            icon = consts.icoDir

        if modName in self.modules:
            self.modules[modName].append(expName)
            self.modules[modName].sort()
        else:
            self.modules[modName] = [expName]

        if widget not in self.widgets:
            numPage = self.notebook.get_n_pages()
            self.widgets[widget] = numPage
            self.notebook.append_page(widget)

        self.explorers[(modName, expName)] = (widget, icon, usrParam)
        self.populateComboBox()


    def removeExplorer(self, modName, expName):
        """ Remove an existing explorer """
        (widget, icon, usrParam) = self.explorers[(modName, expName)]
        del self.explorers[(modName, expName)]

        delWidget = False
        self.modules[modName].remove(expName)
        if len(self.modules[modName]) == 0:
            del self.modules[modName]
            delWidget = True
        else:
            delWidget = True
            for otherExpName in self.modules[modName]:
                if self.explorers[(modName, otherExpName)][0] == widget:
                    delWidget = False
                    break

        if delWidget:
            numPage = self.widgets[widget]
            del self.widgets[widget]
            self.notebook.remove_page(numPage)
            for otherWidget in self.widgets:
                if self.widgets[otherWidget] > numPage:
                    self.widgets[otherWidget] = numPage - 1

        self.populateComboBox()


    def populateComboBox(self):
        """ Populate the combo box based on the currently known explorers """
        oldIndex = self.combo.get_active()
        if oldIndex == -1: currName = prefs.get(__name__, 'last-explorer', None)
        else:              currName = self.store.get_value(self.store.get_iter(oldIndex), EXP_NAME)[1:]

        newIndex = None
        self.combo.freeze_child_notify()
        self.store.clear()
        for modName in self.modules:
            for expName in self.modules[modName]:
                (widget, icon, usrParam) = self.explorers[(modName, expName)]
                self.store.append((' %s' % expName, icon, modName, self.widgets[widget], False, usrParam))
                if expName == currName:
                    newIndex = len(self.store) - 1
        self.combo.thaw_child_notify()

        self.combo.set_sensitive(len(self.store) != 0)

        if newIndex is None:
            self.notebook.set_current_page(0)
        elif oldIndex == -1:
            self.combo.set_active(newIndex)
        else:
            self.combo.disconnect(self.changedHandler)
            self.combo.set_active(newIndex)
            self.changedHandler = self.combo.connect('changed', self.onExplorerChanged)


   # --== Message handler ==--


    def handleMsg(self, msg, params):
        """ Handle messages sent to this modules """
        if msg == modules.MSG_CMD_EXPLORER_ADD:
            self.addExplorer(params['modName'], params['expName'], params['icon'], params['widget'], params['usrParam'])
        elif msg == modules.MSG_CMD_EXPLORER_REMOVE:
            self.removeExplorer(params['modName'], params['expName'])


    # --== GTK handlers ==--


    def onExplorerChanged(self, combo):
        """ A new explorer has been selected with the combo box """
        iter     = self.store.get_iter(self.combo.get_active())
        expName  = self.store.get_value(iter, EXP_NAME)[1:]
        modName  = self.store.get_value(iter, EXP_MOD_NAME)
        numPage  = self.store.get_value(iter, EXP_NUM_PAGE)
        usrParam = self.store.get_value(iter, EXP_USR_PARAM)

        prefs.set(__name__, 'last-explorer', expName)
        modules.postMsg(modules.MSG_EVT_EXPLORER_CHANGED, {'modName': modName, 'expName': expName, 'usrParam': usrParam})
        self.notebook.set_current_page(numPage)
