# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import gobject, gtk, media, modules, os.path, pango, tools, urllib

from gtk          import gdk
from gui          import fileChooser
from tools        import consts
from media        import playlist
from modules      import Module
from gettext      import gettext as _
from gobject      import TYPE_STRING, TYPE_INT
from media.track  import Track
from gui.listview import ListView


(
    ROW_ICO,   # Icon drawn in front of the row
    ROW_NUM,   # Track number
    ROW_TIT,   # Track title
    ROW_ART,   # Track artist
    ROW_ALB,   # Track album
    ROW_LEN,   # Track length in seconds
    ROW_PTH    # Full path to the file
) = range(7)


class Tracklist(Module):
    """
        This module manages the tracklist
    """


    def __init__(self, wTree):
        """ Constructor """
        Module.__init__(self)
        self.popup    = None
        self.playtime = 0
        # Retrieve widgets
        self.window     = wTree.get_widget('win-main')
        self.btnClear   = wTree.get_widget('btn-tracklistClear')
        self.btnShuffle = wTree.get_widget('btn-tracklistShuffle')
        self.btnClear.set_sensitive(False)
        self.btnShuffle.set_sensitive(False)
        # Create the list and its columns
        pixbufRdr   = gtk.CellRendererPixbuf()
        txtLeftRdr  = gtk.CellRendererText()
        txtRightRdr = gtk.CellRendererText()
        txtRightRdr.set_property('xalign', 1.0)

        columns = (('#',         [(pixbufRdr, gdk.Pixbuf), (txtRightRdr, TYPE_INT)], ROW_NUM, False),
                   (_('Title'),  [(txtLeftRdr, TYPE_STRING)],                        ROW_TIT, True),
                   (_('Artist'), [(txtLeftRdr, TYPE_STRING)],                        ROW_ART, True),
                   (_('Album'),  [(txtLeftRdr, TYPE_STRING)],                        ROW_ALB, True),
                   (_('Length'), [(txtRightRdr, TYPE_INT)],                          ROW_LEN, False),
                   (None,        [(None, TYPE_STRING)],                              None,    False))

        self.list = ListView(columns)
        self.list.enableDND()
        wTree.get_widget('scrolledwindow1').add(self.list)
        # FIXME Could be cleaner
        self.list.get_column(4).set_cell_data_func(txtRightRdr, lambda c, cll, mdl, it: cll.set_property('text', tools.sec2str(mdl.get_value(it, ROW_LEN))))
        # Messages handled by this module
        modules.register(self, [modules.MSG_CMD_TRACKLIST_SET, modules.MSG_CMD_TRACKLIST_CLR, modules.MSG_CMD_TRACKLIST_ADD,
                                modules.MSG_CMD_TOGGLE_PAUSE, modules.MSG_CMD_NEXT, modules.MSG_CMD_PREVIOUS, modules.MSG_EVT_NEED_BUFFER,
                                modules.MSG_EVT_STOPPED, modules.MSG_EVT_PAUSED, modules.MSG_EVT_UNPAUSED, modules.MSG_EVT_TRACK_ENDED])
        # GTK handlers
        self.list.connect('listview-dnd',            self.onDND)
        self.list.connect('key-press-event',         self.onKeyboard)
        self.list.connect('listview-modified',       self.onListModified)
        self.list.connect('listview-button-pressed', self.onButtonPressed)
        self.btnClear.connect('clicked', lambda widget: modules.postMsg(modules.MSG_CMD_TRACKLIST_CLR))
        self.btnShuffle.connect('clicked', lambda widget: self.list.shuffle())


    # Helper functions
    def hasNext(self):     return self.list.hasMark() and self.list.getMark() < self.list.getCount()-1
    def hasPrev(self):     return self.list.hasMark() and self.list.getMark() > 0
    def getAllFiles(self): return [row[ROW_PTH] for row in self.list.getAllRows()]


    def setVisible(self, widget, show):
        """ Show/hide a widget """
        widget.set_sensitive(show)
        if show: widget.show()
        else:    widget.hide()


    def jumpTo(self, trackIndex):
        """ Jump to the track located at the given index """
        if self.list.hasMark():
            self.list.setItem(self.list.getMark(), ROW_ICO, consts.icoNull)
        self.list.setMark(trackIndex)
        self.list.scroll_to_cell(trackIndex)
        self.list.setItem(trackIndex, ROW_ICO, consts.icoPlay)
        modules.postMsg(modules.MSG_CMD_PLAY, {'filename': self.list.getItem(trackIndex, ROW_PTH)})
        modules.postMsg(modules.MSG_EVT_NEW_TRACK, {'track': Track(self.list.getRow(trackIndex)[1:], trackIndex+1, self.list.getCount())})
        modules.postMsg(modules.MSG_EVT_TRACK_MOVED, {'hasPrevious': self.hasPrev(), 'hasNext': self.hasNext()})


    def onTrackEnded(self):
        """ The current track has ended, jump to the next one if any """
        if self.hasNext(): self.jumpTo(self.list.getMark()+1)
        else:              modules.postMsg(modules.MSG_CMD_STOP)


    def onBufferingNeeded(self):
        """ The current track is close to its end, se we try to buffer the next one to avoid gaps """
        if self.hasNext():
            modules.postMsg(modules.MSG_CMD_BUFFER, {'filename': self.list.getItem(self.list.getMark()+1, ROW_PTH)})


    def insert(self, files, position=None):
        """ Insert some files in the tracklist, append them if position is None """
        tracks = [[consts.icoNull] + track for track in media.getTracks(files)]

        if len(tracks) != 0:
            for track in tracks: self.playtime += track[ROW_LEN]

            self.btnClear.set_sensitive(True)
            self.btnShuffle.set_sensitive(True)
            self.list.insertRows(tracks, position)


    def set(self, files, playNow):
        """ Replace the tracklist, clear it if files is None """
        self.playtime = 0
        self.list.clear()

        if files is not None:
            self.insert(files)

        if self.list.getCount() == 0:
            modules.postMsg(modules.MSG_CMD_STOP)
            self.btnClear.set_sensitive(False)
            self.btnShuffle.set_sensitive(False)
            modules.postMsg(modules.MSG_EVT_NEW_TRACKLIST, {'files': [], 'playtime': 0})
        elif playNow:
            self.jumpTo(0)


    def onStopped(self):
        """ Playback has been stopped """
        if self.list.hasMark():
            self.list.setItem(self.list.getMark(), ROW_ICO, consts.icoNull)
            self.list.clearMark()


    def onPausedToggled(self, icon):
        """ Switch between paused and unpaused """
        if self.list.hasMark():
            self.list.setItem(self.list.getMark(), ROW_ICO, icon)


    def savePlaylist(self):
        """ Save the current tracklist to a playlist """
        directory = os.path.dirname(self.list.getItem(0, ROW_PTH))
        filename  = fileChooser.save(self.window, _('Save playlist'), 'playlist.m3u', directory)
        if filename is not None:
            playlist.save(self.getAllFiles(), filename)


    def cropSelection(self):
        """ Remove all tracks but the selection """
        self.playtime = 0
        for row in self.list.getSelectedRows():
            self.playtime += row[ROW_LEN]

        self.list.cropSelectedRows()


    def removeSelection(self):
        """ Remove all the currently selected tracks """
        for row in self.list.getSelectedRows():
            self.playtime -= row[ROW_LEN]
            if row[ROW_ICO] != consts.icoNull:
                modules.postMsg(modules.MSG_CMD_STOP)

        self.list.removeSelectedRows()


    def showPopupMenu(self, path, button, time):
        """ The index parameter may be None """
        if self.popup is None:
            self.popup = tools.loadGladeFile('TracklistMenu.glade')
            # Connect signals
            self.popup.get_widget('item-save').connect('activate', lambda wdg: self.savePlaylist())
            self.popup.get_widget('item-play').connect('activate', lambda wdg: self.jumpTo(self.list.getFirstSelectedRowIndex()))
            self.popup.get_widget('item-crop').connect('activate', lambda wdg: self.cropSelection())
            self.popup.get_widget('item-stop').connect('activate', lambda wdg: modules.postMsg(modules.MSG_CMD_STOP))
            self.popup.get_widget('item-pause').connect('activate', lambda wdg: modules.postMsg(modules.MSG_CMD_TOGGLE_PAUSE))
            self.popup.get_widget('item-clear').connect('activate', lambda wdg: modules.postMsg(modules.MSG_CMD_TRACKLIST_CLR))
            self.popup.get_widget('item-remove').connect('activate', lambda wdg: self.removeSelection())
            self.popup.get_widget('item-unpause').connect('activate', lambda wdg: modules.postMsg(modules.MSG_CMD_TOGGLE_PAUSE))

        # Enable only relevant menu entries
        if path is None: icon = None
        else:            icon = self.list.getItem(path, ROW_ICO)

        if self.list.getSelectedRowsCount() == 1:
            self.setVisible(self.popup.get_widget('item-play'),    icon == consts.icoNull)
            self.setVisible(self.popup.get_widget('item-unpause'), icon == consts.icoPause)
            self.setVisible(self.popup.get_widget('item-pause'),   icon == consts.icoPlay)
            self.setVisible(self.popup.get_widget('item-stop'),    icon == consts.icoPlay or icon == consts.icoPause)
        else:
            self.popup.get_widget('item-play').show()
            self.popup.get_widget('item-play').set_sensitive(False)
            self.popup.get_widget('item-unpause').hide()
            self.popup.get_widget('item-pause').hide()
            self.popup.get_widget('item-stop').hide()

        self.popup.get_widget('item-sep1').set_sensitive(icon is not None)
        self.popup.get_widget('item-crop').set_sensitive(icon is not None)
        self.popup.get_widget('item-remove').set_sensitive(icon is not None)
        self.popup.get_widget('item-sep2').set_sensitive(icon is not None)

        self.popup.get_widget('menu-popup').popup(None, None, None, button, time)


    # --== Message handler ==--


    def handleMsg(self, msg, params):
        """ A message has been received """
        if   msg == modules.MSG_EVT_PAUSED:                                   self.onPausedToggled(consts.icoPause)
        elif msg == modules.MSG_EVT_STOPPED:                                  self.onStopped()
        elif msg == modules.MSG_EVT_UNPAUSED:                                 self.onPausedToggled(consts.icoPlay)
        elif msg == modules.MSG_EVT_TRACK_ENDED:                              self.onTrackEnded()
        elif msg == modules.MSG_EVT_NEED_BUFFER:                              self.onBufferingNeeded()
        elif msg == modules.MSG_CMD_TRACKLIST_SET:                            self.set(params['files'], params['playNow'])
        elif msg == modules.MSG_CMD_TRACKLIST_ADD:                            self.insert(params['files'])
        elif msg == modules.MSG_CMD_TRACKLIST_CLR:                            self.set(None, False)
        elif msg == modules.MSG_CMD_NEXT and self.hasNext():                  self.jumpTo(self.list.getMark()+1)
        elif msg == modules.MSG_CMD_PREVIOUS and self.hasPrev():              self.jumpTo(self.list.getMark()-1)
        elif msg == modules.MSG_CMD_TOGGLE_PAUSE and not self.list.hasMark(): self.jumpTo(0)


    # --== GTK handlers ==--


    def onButtonPressed(self, list, event, path):
        """ Play the selected track on double click, or show a popup menu on right click """
        if event.button == 1 and event.type == gdk._2BUTTON_PRESS and path is not None:
            self.jumpTo(path[0])
        elif event.button == 3 and event.state == 0 and self.list.getCount() != 0:
            self.showPopupMenu(path, event.button, event.time)


    def onKeyboard(self, list, event):
        """ Remove the selection if possible """
        if gdk.keyval_name(event.keyval) == 'Delete':
            self.removeSelection()


    def onListModified(self, list):
        """ Some rows have been added/removed/moved """
        modules.postMsg(modules.MSG_EVT_NEW_TRACKLIST, {'files': self.getAllFiles(),   'playtime': self.playtime})
        modules.postMsg(modules.MSG_EVT_TRACK_MOVED,   {'hasPrevious': self.hasPrev(), 'hasNext':  self.hasNext()})


    def onDND(self, list, context, x, y, dragData, time):
        """ External Drag'n'Drop """
        files    = [urllib.url2pathname(uri.strip()[7:]) for uri in dragData.data.split() if uri.startswith('file://')]
        dropInfo = self.list.get_dest_row_at_pos(x, y)

        if len(files) == 0:
            context.finish(False, False, time)

        if dropInfo is None: self.insert(files)
        else:                self.insert(files, dropInfo[0][0])

        context.finish(True, False, time)
