# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import gobject, gtk, gtk.glade, os, sys, traceback

from gtk       import gdk
from tools     import consts, prefs
from gettext   import gettext as _
from threading import Lock, Semaphore, Thread


# List of messages that can be sent/received by modules
# A message is always associated with a (potentially empty) dictionnary that contains its parameters
(
    # --== COMMANDS ==--

    # GStreamer player
    MSG_CMD_PLAY,             # Play a file         Parameters: 'filename'
    MSG_CMD_STOP,             # Stop playing        Parameters:
    MSG_CMD_SEEK,             # Jump to a position  Parameters: 'seconds'
    MSG_CMD_BUFFER,           # Buffer a file       Parameters: 'filename'
    MSG_CMD_TOGGLE_PAUSE,     # Toggle play/pause   Parameters:

    # Tracklist
    MSG_CMD_NEXT,             # Play the next track       Parameters:
    MSG_CMD_PREVIOUS,         # Play the previous track   Parameters:
    MSG_CMD_TRACKLIST_SET,    # Replace the tracklist     Parameters: 'files', 'playNow'
    MSG_CMD_TRACKLIST_ADD,    # Extend the tracklist      Parameters: 'files'
    MSG_CMD_TRACKLIST_CLR,    # Clear the tracklist       Parameters:

    # Explorer manager
    MSG_CMD_EXPLORER_ADD,     # Add a new explorer    Parameters: 'modName', 'expName', 'icon', 'widget', 'usrParam'
    MSG_CMD_EXPLORER_REMOVE,  # Remove an explorer    Parameters: 'modName', 'expName'

    # --== EVENTS ==--

    # Current track
    MSG_EVT_PAUSED,           # Paused                              Parameters:
    MSG_EVT_STOPPED,          # Stopped                             Parameters:
    MSG_EVT_UNPAUSED,         # Unpaused                            Parameters:
    MSG_EVT_NEW_TRACK,        # The current track has changed       Parameters: 'track'
    MSG_EVT_NEED_BUFFER,      # The next track should be buffered   Parameters:
    MSG_EVT_TRACK_ENDED,      # The current track has ended         Parameters:
    MSG_EVT_TRACK_POSITION,   # The track position has changed      Parameters: 'seconds'

    # Tracklist
    MSG_EVT_TRACK_MOVED,      # The position of the current track has changed    Parameters: 'hasPrevious', 'hasNext'
    MSG_EVT_NEW_TRACKLIST,    # A new tracklist has been set                     Parameters: 'files', 'playtime'

    # Application
    MSG_EVT_APP_QUIT,         # The application is quitting         Parameters:
    MSG_EVT_APP_STARTED,      # The application has just started    Parameters:

    # Modules
    MSG_EVT_MOD_LOADED,       # The module has been loaded by request of the user      Parameters:
    MSG_EVT_MOD_UNLOADED,     # The module has been unloaded by request of the user    Parameters:

    # Explorer manager
    MSG_EVT_EXPLORER_CHANGED, # A new explorer has been selected    Parameters: 'modName', 'expName', 'usrParam'

    # End value
    MSG_END_VALUE
) = range(27)


# Values associated with a module
(
    MOD_PMODULE,      # The actual Python module object
    MOD_CLASSNAME,    # The classname of the module
    MOD_DESC,         # Description
    MOD_INSTANCE,     # Instance, None if not currently enabled
    MOD_MANDATORY,    # If True, the module cannot be unloaded
    MOD_CONFIGURABLE, # If True, the module may be configured
    MOD_DEPS          # Special dependencies of the module (Python modules that may not be installed on the system (e.g., pynotify, pyosd))
) = range(7)


mWTree          = None
mModules        = {}                                                 # All known modules, together with an 'active' boolean
mHandlers       = dict([(msg, []) for msg in xrange(MSG_END_VALUE)]) # For each message, store the set of registered modules
mModulesLock    = Lock()                                             # This lock protects the modules list from concurrent access
mHandlersLock   = Lock()                                             # This lock protects the handlers list from concurrent access
mEnabledModules = None                                               # List of modules currently enabled


# Make sure the path to the modules is known to the Python loader
mModDir = os.path.dirname(__file__)
if not mModDir in sys.path:
    sys.path.append(mModDir)


def __checkDeps(deps):
    """ Given a list of Python modules, return a new list containing the modules that are not currently installed or None """
    unmetDeps = []
    for dep in deps:
        try:    __import__(dep)
        except: unmetDeps.append(dep)

    if len(unmetDeps) == 0: return None
    else:                   return unmetDeps


def initialLoad(wTree):
    """
        Find all modules, instantiate those that are mandatory or that have been previously enabled by the user
        This should be the first called function of this module
    """
    global mWTree, mEnabledModules

    mWTree          = wTree
    mEnabledModules = prefs.get(__name__, 'enabled_modules', [])

    for filename in [os.path.splitext(f)[0] for f in os.listdir(mModDir) if f.endswith('.py') and f != '__init__.py']:
        try:
            pModule = __import__(filename)
            # The module name
            if hasattr(pModule, 'MOD_NAME'): name = getattr(pModule, 'MOD_NAME')
            else:                            name = '%s %d' % (_('Unnamed module'), len(mModules))
            # The module description
            if hasattr(pModule, 'MOD_DESC'): desc = getattr(pModule, 'MOD_DESC')
            else:                            desc = ''
            # Is it a mandatory module?
            if hasattr(pModule, 'MOD_IS_MANDATORY'): isMandatory = getattr(pModule, 'MOD_IS_MANDATORY')
            else:                                    isMandatory = True
            # Can it be configured?
            if hasattr(pModule, 'MOD_IS_CONFIGURABLE'): isConfigurable = getattr(pModule, 'MOD_IS_CONFIGURABLE')
            else:                                       isConfigurable = False
            # Does it require special Python modules?
            if hasattr(pModule, 'MOD_DEPS'): deps = getattr(pModule, 'MOD_DEPS')
            else:                            deps = []
            # Should it be instanciated immediately?
            if isMandatory or name in mEnabledModules:
                unmetDeps = __checkDeps(deps)
                if unmetDeps is None: instance = getattr(pModule, filename)(wTree)
                else:                 instance = None
            else:
                instance = None
            # Ok, add it to the dictionary
            mModules[name] = [pModule, filename, desc, instance, isMandatory, isConfigurable, deps]
        except:
            print _('ERROR: Unable to load module "%s"!') % os.path.join(mModDir, filename)
            traceback.print_exc()
    # Remove modules that are no longer there from the list of enabled modules
    mEnabledModules[:] = [module for module in mEnabledModules if module in mModules]
    prefs.set(__name__, 'enabled_modules', mEnabledModules)
    # Start the threaded modules, this has no effect on non-threaded ones
    for module in [m for m in mModules.itervalues() if m[MOD_INSTANCE] is not None]:
        module[MOD_INSTANCE].start()


def unload(name):
    """ Unload the given module """
    mModulesLock.acquire()
    data               = mModules[name]
    instance           = data[MOD_INSTANCE]
    data[MOD_INSTANCE] = None
    mModulesLock.release()

    if instance is None:
        return

    mHandlersLock.acquire()
    # Warn the module that it is going to be unloaded
    if instance in mHandlers[MSG_EVT_MOD_UNLOADED]:
        instance.postMsg(MSG_EVT_MOD_UNLOADED, {})
    # Remove it from all registered handlers
    for handlers in [h for h in mHandlers.itervalues() if instance in h]:
        handlers.remove(instance)
    mHandlersLock.release()

    mEnabledModules.remove(name)
    prefs.set(__name__, 'enabled_modules', mEnabledModules)


def load(name):
    """ Load the given module, return an error message if not possible or None """
    mModulesLock.acquire()
    modData = mModules[name]

    # Check dependencies
    unmetDeps = __checkDeps(modData[MOD_DEPS])
    if unmetDeps is not None:
        mModulesLock.release()
        errMsg  = _('The following Python modules are not available:')
        errMsg += '\n     * '
        errMsg += '\n     * '.join(unmetDeps)
        errMsg += '\n\n'
        errMsg += _('You must install them if you want to enable this module.')
        return errMsg

    try:
        instance              = getattr(modData[MOD_PMODULE], modData[MOD_CLASSNAME])(mWTree)
        modData[MOD_INSTANCE] = instance
        mEnabledModules.append(name)
        prefs.set(__name__, 'enabled_modules', mEnabledModules)
    except:
        return traceback.format_exc()
    finally:
        mModulesLock.release()

    instance.start()

    mHandlersLock.acquire()
    if instance in mHandlers[MSG_EVT_MOD_LOADED]:
        instance.postMsg(MSG_EVT_MOD_LOADED, {})
    mHandlersLock.release()

    return None


def getModules():
    """ Return a copy of all known modules """
    mModulesLock.acquire()
    copy = mModules.items()
    mModulesLock.release()

    return copy


def register(module, msgList):
    """ Register the given module for all messages in the given list """
    mHandlersLock.acquire()
    for msg in msgList:
        mHandlers[msg].append(module)
    mHandlersLock.release()


def configure(modName, parentDlg):
    """ Ask the given module to display its configuration dialog """
    mModulesLock.acquire()
    mModules[modName][MOD_INSTANCE].configure(parentDlg)
    mModulesLock.release()


def __postMsg(msg, params={}):
    """
        This is the 'real' postMsg() function
        It MUST be executed in the main GTK loop
    """
    mHandlersLock.acquire()
    for module in mHandlers[msg]:
        module.postMsg(msg, params)
    mHandlersLock.release()


def __postQuitMsg():
    """
        This is the 'real' postQuitMsg() function
        It MUST be executed in the main GTK loop
    """
    __postMsg(MSG_EVT_APP_QUIT)

    for module in [m for m in mModules.itervalues() if m[MOD_INSTANCE] is not None]:
        module[MOD_INSTANCE].join()

    prefs.save()
    gtk.main_quit()


def postMsg(msg, params={}):
    """
        Post a message in the queue of all modules
        Only registered modules for this message will receive it
    """
    # We need to ensure that posting messages will be done by the main GTK thread
    # Otherwise, the code of non-threaded modules will be executed in the caller's thread, which may not be the main GTK thread
    # This can cause problems when such modules call GTK functions
    gobject.idle_add(__postMsg, msg, params)


def postQuitMsg():
    """
        Post a MSG_EVT_APP_QUIT in each module's queue and exit the application
        Preferences are saved before exiting
    """
    # As with postMsg(), we need to ensure that the code will be executed by the main GTK thread
    gobject.idle_add(__postQuitMsg)


    # --== Base classes for the modules ==--


class ModuleBase:
    """
        This is the base class for all kinds of modules
    """


    def join(self):                    pass
    def start(self):                   pass
    def __init__(self):                pass
    def configure(self, parent):       pass
    def postMsg(self, msg, params):    pass
    def handleMsg(self, msg, params):  pass


class Module(ModuleBase):
    """
        This is the base class for all non-threaded modules
    """
    def __init__(self):             ModuleBase.__init__(self)
    def postMsg(self, msg, params): self.handleMsg(msg, params)


class ThreadedModule(Thread, ModuleBase):
    """
        This is the base class for all threaded modules
    """


    def __init__(self):
        """
            Constructor
        """
        Thread.__init__(self)
        ModuleBase.__init__(self)

        self.queue        = []              # List of queued messages
        self.mutex        = Lock()          # Protect access to the queue
        self.semaphore    = Semaphore(0)    # Block the thread until some messages are available
        self.gtkSemaphore = Semaphore(0)    # Used to execute some code in the GTK main loop

        # Some messages required by this class
        register(self, [MSG_EVT_APP_QUIT, MSG_EVT_MOD_UNLOADED])


    def __gtkExecute(self, func):
        """ Private function, must be executed in the GTK main loop """
        func()
        self.gtkSemaphore.release()


    def gtkExecute(self, func):
        """ Execute func in the GTK main loop, and block the execution until done """
        gobject.idle_add(self.__gtkExecute, func)
        self.gtkSemaphore.acquire()


    def postMsg(self, msg, params):
        """
            Enqueue a new message in this threads's message queue
            This function is thread-safe
        """
        self.mutex.acquire()
        self.queue.append((msg, params))
        self.mutex.release()
        self.semaphore.release()


    def run(self):
        """
            The thread's entry point
            The execution is suspended until some messages are available
            When this happens, the main message handler is called
        """
        while True:
            self.semaphore.acquire()
            self.mutex.acquire()
            (msg, params) = self.queue.pop(0)
            self.mutex.release()
            self.handleMsg(msg, params)
            # The thread may need to stop
            if msg == MSG_EVT_APP_QUIT or msg == MSG_EVT_MOD_UNLOADED:
                break
