/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "accountmanager.h"
#include "accountmanageradaptor.h"
#include "connectionfacade.h"

#include <QtTapioca/Connection>
#include <QtTapioca/ContactBase>

#include <Decibel/Errors>
#include <Decibel/AccountData>

#include <QtCore/QSettings>
#include <QtCore/QHash>
#include <QtCore/QPointer>
#include <QtCore/QDebug>

namespace
{
    static const QString accountmanager_group("AccountManager");
    static const QString account_array("Accounts");
}

/// @cond INCLUDE_PRIVATE

/**
 * @brief Private class implementing the AccountManager.
 *
 * A private class implementing the AccountManager.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */
class AccountManagerPrivate
{
public:
    /**
     * @brief Constructor.
     */
    AccountManagerPrivate(ConnectionFacade * conn_mgr) :
        m_currentHandle(1),
        m_connectionFacade(conn_mgr),
        blockReconnects(false)
    {
        Q_ASSERT(0 != conn_mgr);

        // read account data:
        QSettings settings;
        settings.beginGroup(accountmanager_group);

        int num_accounts = settings.beginReadArray(account_array);
        for (int i = 0; i < num_accounts; ++i)
        {
            settings.setArrayIndex(i);
            QVariantMap account_data;

            QStringList keys = settings.allKeys();
            QString current_key;
            foreach (current_key, keys)
            {
                QVariant value = settings.value(current_key);
                if (current_key == Decibel::name_presence)
                { value = QVariant(value.toInt()); }
                account_data.insert(current_key, value);
            }
            account_data = sanityCheck(account_data);
            createAccount(account_data);
        }
        settings.endArray();
        settings.endGroup();

        Q_ASSERT(m_accounts.size() == num_accounts);
    }

    /**
     * @brief Destructor.
     */
    ~AccountManagerPrivate()
    {
        // Store account data:
        QSettings settings;
        settings.beginGroup(accountmanager_group);

        settings.beginWriteArray(account_array, m_accounts.size());
        AccountData current_data;
        int i = 0;
        foreach (current_data, m_accounts.values())
        {
            settings.setArrayIndex(i);

            QString current_key;
            foreach (current_key, current_data.data.keys())
            { settings.setValue(current_key, current_data.data[current_key]); }
            ++i;
        }
        settings.endArray();
        settings.endGroup();
    }

// Helper functions:

    /**
     * @brief Create an account in the accounts database.
     * @param nv_pairs The account data.
     * @return A handle to the account.
     *
     * This method creates an account in the account database. It assumes
     * the account data is complete.
     */
    uint createAccount(const QVariantMap & nv_pairs)
    {
        Q_ASSERT(nv_pairs.contains(Decibel::name_display_name));
        Q_ASSERT(nv_pairs.contains(Decibel::name_protocol));
        Q_ASSERT(nv_pairs.contains(Decibel::name_presence));
        Q_ASSERT(nv_pairs.contains(Decibel::name_presence_message));
        Q_ASSERT(nv_pairs.contains(Decibel::name_current_presence));
        Q_ASSERT(nv_pairs.contains(Decibel::name_autoreconnect));

        m_accounts[m_currentHandle] = nv_pairs;
        return m_currentHandle++;
    }

    /**
     * @brief Make sure a account has the required presence information.
     * @param nv_pairs The account data.
     * @returns A new set of account data.
     *
     * Make sure a account has the required presence information and fill in
     * the blanks if there are any.
     */
    QVariantMap sanityCheck(const QVariantMap & nv_pairs) const
    {
        QVariantMap pairs(nv_pairs);

        if (!pairs.contains(Decibel::name_protocol))
        { return QVariantMap(); }

        if (!pairs.contains(Decibel::name_presence))
        { pairs.insert(Decibel::name_presence, QVariant(QtTapioca::ContactBase::Offline)); }
        else
        {
            int presence = pairs[Decibel::name_presence].toInt();
            if (presence < QtTapioca::ContactBase::Offline ||
                presence > QtTapioca::ContactBase::Busy)
            { pairs[Decibel::name_presence] = QVariant(QtTapioca::ContactBase::Offline); }
        }
        if (!pairs.contains(Decibel::name_presence_message))
        { pairs.insert(Decibel::name_presence_message, QVariant(QString())); }
        if (!pairs.contains(Decibel::name_autoreconnect))
        { pairs.insert(Decibel::name_autoreconnect, QVariant(false)); }
        if (!pairs.contains(Decibel::name_display_name))
        {
            // create a sensible display name:
            QString protocol;
            QString account;
            QString server;
            QString port;

            if (pairs.contains(Decibel::name_protocol))
            { protocol = pairs[Decibel::name_protocol].toString(); }
            if (pairs.contains(Decibel::name_tp_account))
            { account = pairs[Decibel::name_tp_account].toString(); }
            if (pairs.contains(Decibel::name_tp_server))
            { server = pairs[Decibel::name_tp_server].toString(); }
            if (pairs.contains(Decibel::name_tp_port))
            { port = pairs[Decibel::name_tp_port].toString(); }

            if (protocol.isEmpty()) { protocol = "unknown"; }
            if (account.isEmpty()) { account = "unknown"; }
            if (server.isEmpty()) { server = "unknown"; }
            if (!port.isEmpty()) { port = ':' + port; }

            QString name(protocol + "://" +
                         account + '@' + server + port + '/');
            pairs.insert(Decibel::name_display_name, QVariant(name));
        }

        pairs.insert(Decibel::name_current_presence, QVariant(QtTapioca::ContactBase::Offline));

        Q_ASSERT(pairs.contains(Decibel::name_protocol));
        Q_ASSERT(pairs.contains(Decibel::name_presence));
        Q_ASSERT(pairs.contains(Decibel::name_current_presence));
        Q_ASSERT(pairs.contains(Decibel::name_autoreconnect));
        Q_ASSERT(pairs.contains(Decibel::name_presence_message));
        Q_ASSERT(pairs.contains(Decibel::name_display_name));
        Q_ASSERT(pairs.size() >= nv_pairs.size() &&
                 pairs.size() <= nv_pairs.size() + 5 );
        return pairs;
    }

    /**
     * @brief Manage presence information on a connection.
     * @param account_handle A handle to an account.
     * @param presence_state The presence state.
     * @param message A presence message.
     * @return A value > 0 on success and a negative error code otherwise.
     *
     * Change the presence information on an account. This brings an account
     * on/off line as required.
     */
    int manageConnection(const uint account_handle,
                         const QtTapioca::ContactBase::Presence presence_state,
                         const QString message)
    {
        Q_ASSERT(gotAccount(account_handle));
        Q_ASSERT(presence_state >= QtTapioca::ContactBase::Offline &&
                 presence_state <= QtTapioca::ContactBase::Busy);
        int result = presence_state;

        QtTapioca::Connection * connection = connectionOf(account_handle);

        if (presence_state == QtTapioca::ContactBase::Offline)
        {
            // We need to disconnect:
            if (0 != connection)
            { connection->disconnect(); }
            // connection pointer is set on signal!
        }
        else
        {
            if (connection == 0)
            {
                // We need to connect if we are not yet connected.
                connection =
                    m_connectionFacade->connectUsingAccount(m_accounts[account_handle].data, presence_state, message);
                if (connection == 0)
                {
                    return QtTapioca::ContactBase::Offline; /* FIXME: Give proper error code! */
                }
                setConnection(account_handle, connection);
            }
            else
            {
                // update presence info
                int new_presence = m_connectionFacade->updatePresence(connection, presence_state, message);
                if (new_presence < 0) { result = new_presence; }
            }
            Q_ASSERT(0 != connectionOf(account_handle));
        }
        return result;
    }

    /**
     * @brief Associate a account with a connection.
     * @param account_handle A handle to an account.
     * @param connection A pointer to an Connection.
     *
     * Associate an account with a connection.
     */
    void setConnection(const uint account_handle,
                       QtTapioca::Connection * connection)
    {
        Q_ASSERT(0 == connectionOf(account_handle) ||
                 0 == connection);
        if (!gotAccount(account_handle)) { return; }
        m_accounts[account_handle].connection = connection;
    }

    /**
     * @brief Get the connection an account is using.
     * @param account_handle A handle to an account.
     * @return A pointer to the connection the account is using. This connection
     * will be 0 if the account is not on line.
     */
    QtTapioca::Connection * connectionOf(const uint account_handle) const
    {
        if (!gotAccount(account_handle)) { return 0; }
        return m_accounts[account_handle].connection;
    }

    /**
     * @brief Check whether the account handle is defined.
     * @param account_handle A handle to an account.
     * @return true if the account handle is defined and false otherwise.
     *
     * Check whether the account handle is defined.
     */
    bool gotAccount(const uint account_handle) const
    { return m_accounts.contains(account_handle); }

    /**
     * @brief Find the account using a connection.
     * @param connection A pointer to an Connection.
     * @return A handle to an account. This handle is 0 if no account is using
     * that connection.
     *
     * Find the account using a connection.
     */
    uint findAccountWithConnection(QtTapioca::Connection * connection)
    {
        int account_handle;
        foreach(account_handle, m_accounts.keys())
        {
            if (m_accounts[account_handle].connection == connection)
            { return account_handle; }
        }
        return 0;
    }

    /**
     * @brief Update presence information on an account.
     * @param account_handle A handle to an account.
     * @param presence_state The new presence state.
     * @param message The new presence message.
     *
     * Update presence information on an account, bringing it on/off line as
     * required.
     */
    void updatePresenceLocally(const uint account_handle,
                        const QtTapioca::ContactBase::Presence presence_state,
                        const QString message)
    {
        Q_ASSERT(gotAccount(account_handle));
        Q_ASSERT(m_accounts[account_handle].data.contains(Decibel::name_current_presence));
        Q_ASSERT(m_accounts[account_handle].data.contains(Decibel::name_presence));
        Q_ASSERT(m_accounts[account_handle].data.contains(Decibel::name_autoreconnect));
        Q_ASSERT(m_accounts[account_handle].data.contains(Decibel::name_presence_message));

        m_accounts[account_handle].data[Decibel::name_current_presence] =
            QVariant(presence_state);
        if (!message.isEmpty())
        {
            m_accounts[account_handle].data[Decibel::name_presence_message] =
                QVariant(message);
        }

        // Get back to intended presence state:
        if (blockReconnects) { return; }

        if (m_accounts[account_handle].data[Decibel::name_autoreconnect].toBool())
        {
            const QtTapioca::ContactBase::Presence intended_state =
                static_cast<QtTapioca::ContactBase::Presence>(
                m_accounts[account_handle].data[Decibel::name_presence].toInt());
            if (intended_state != presence_state)
            {
                manageConnection(account_handle, intended_state,
                    m_accounts[account_handle].data[Decibel::name_presence_message].toString());
            }
        }
    }

    /**
     * @brief Update the presence message on an account.
     * @param account_handle A handle to an account.
     * @param message The new presence message.
     *
     * Update the presence message on an account.
     */
    void updatePresenceMessageLocally(const uint account_handle, const QString & message)
    {
        Q_ASSERT(gotAccount(account_handle));
        m_accounts[account_handle].data[Decibel::name_presence_message] = QVariant(message);
    }

    /**
     * @brief A internal data structure.
     *
     * @author Tobias Hunger <info@basyskom.de>
     */
    struct AccountData
    {
        /** @brief Constructor */
        AccountData(const QVariantMap & nv_pairs) :
            data(nv_pairs), connection(0)
        { }
        /** @brief Default constructor */
        AccountData() :
            data(QVariantMap()), connection(0)
        { }

        /** @brief Copy constructor */
        AccountData(const AccountData& account_data) :
            data(account_data.data),
            connection(account_data.connection)
        { }
        /** @brief operator = */
        AccountData& operator =(const AccountData& account_data)
        {
            data = account_data.data;
            connection = account_data.connection;
            return *this;
        }

        /** @brief The actual account data. */
        QVariantMap data;
        /** @brief The connection of the account. */
        QtTapioca::Connection * connection;
    };
    /** @brief Account data, addressed by account_handle. */
    QHash<uint, AccountData> m_accounts;

    /** @brief The next available handle. */
    uint m_currentHandle;

    /** @brief A pointer to the ConnectionFacade */
    QPointer<ConnectionFacade> m_connectionFacade;
    /** @brief A pointer to the D-Bus Adaptor of the AccountManager. */
    QPointer<AccountManagerAdaptor> m_adaptor;

    bool blockReconnects;

    friend class AccountManager;
};

/// @endcond

// ****************************************************************************

AccountManager::AccountManager(ConnectionFacade * connection_facade, QObject * parent) :
    QObject(parent),
    d(new AccountManagerPrivate(connection_facade))
{
    Q_ASSERT(d != 0);
    d->m_adaptor = new AccountManagerAdaptor(this);
    Q_ASSERT(d->m_adaptor != 0);

    connect(connection_facade,
        SIGNAL(ownPresenceUpdated(QtTapioca::Connection *, const int, const QString)),
        SLOT(onOwnPresenceUpdated(QtTapioca::Connection *, const int, const QString)));
    connect(connection_facade,
        SIGNAL(connectionClosed(QtTapioca::Connection *)),
        SLOT(onConnectionClosed(QtTapioca::Connection *)));
}

AccountManager::~AccountManager()
{
    delete d;
}

void AccountManager::bringUpAccounts()
{
    uint i;
    foreach(i, d->m_accounts.keys())
    {
        Q_ASSERT(d->m_accounts[i].data.contains(Decibel::name_protocol));
        Q_ASSERT(d->m_accounts[i].data.contains(Decibel::name_current_presence));
        Q_ASSERT(d->m_accounts[i].data.contains(Decibel::name_presence));
        Q_ASSERT(d->m_accounts[i].data.contains(Decibel::name_presence_message));
        Q_ASSERT(d->m_accounts[i].data.contains(Decibel::name_autoreconnect));

        qDebug() << "Bringing up" << i << ":"
                 << d->m_accounts[i].data[Decibel::name_protocol]
                 << "ip:" << d->m_accounts[i].data[Decibel::name_presence]
                 << "cp:"
                 << d->m_accounts[i].data[Decibel::name_current_presence]
                 << "pm:"
                 << d->m_accounts[i].data[Decibel::name_presence_message];

        setPresenceAndMessage(i,
                              d->m_accounts[i].data[Decibel::name_presence].toInt(),
                              d->m_accounts[i].data[Decibel::name_presence_message].toString());
    }
}

QString AccountManager::protocol(const uint account_handle) const
{
    if (!d->m_accounts.contains(account_handle)) { return QString(); }
    Q_ASSERT(d->m_accounts[account_handle].data.contains(Decibel::name_protocol));
    return d->m_accounts[account_handle].data.value(Decibel::name_protocol).toString();
}

bool AccountManager::gotAccount(const uint account_handle) const
{ return d->gotAccount(account_handle); }

QtTapioca::Connection *
AccountManager::connectionOf(const uint account_handle) const
{
    return d->connectionOf(account_handle);
}

QList<uint> AccountManager::listAccounts() const
{ return d->m_accounts.keys(); }

QVariantMap AccountManager::queryAccount(const uint account_handle) const
{
    if (!d->gotAccount(account_handle)) { return QVariantMap(); }

    Q_ASSERT(d->m_accounts[account_handle].data.contains(Decibel::name_protocol));
    Q_ASSERT(d->m_accounts[account_handle].data.contains(Decibel::name_presence));

    return d->m_accounts[account_handle].data;
}

uint AccountManager::addAccount(const QVariantMap & nv_pairs)
{
    int num_accounts = d->m_accounts.size();

    QList<uint> matching_accounts = findAccounts(nv_pairs);
    if (!matching_accounts.isEmpty()) { return matching_accounts[0]; }

    QVariantMap data = d->sanityCheck(nv_pairs);
    if (data.isEmpty()) { return 0; }
    uint account_handle = d->createAccount(data);

    if (account_handle != 0) { emit accountCreated(account_handle); }

    Q_ASSERT(d->m_accounts.size() == num_accounts + 1);

    return account_handle;
}

uint AccountManager::updateAccount(const uint account_handle,
                                   const QVariantMap & nv_pairs)
{
    int num_accounts = d->m_accounts.size();

    QVariantMap data = nv_pairs;
    // Remove internal stuff:
    data.remove(Decibel::name_current_presence);
    int presence_state = data[Decibel::name_presence].toInt();
    QString presence_message = data[Decibel::name_presence_message].toString();
    data.remove(Decibel::name_presence);
    data.remove(Decibel::name_presence_message);

    // merge old and new data:
    QVariantMap old_data = queryAccount(account_handle);
    QString key;
    foreach (key, old_data.keys())
    {
        if (!data.contains(key)) { data.insert(key, old_data[key]); }
    }

    // make sure the data is sane:
    data = d->sanityCheck(data);

    if (data.isEmpty()) { return 0; }

    d->m_accounts[account_handle] = data;
    if (presence_state != presence(account_handle) &&
        presence_message != presenceMessage(account_handle))
    { setPresenceAndMessage(account_handle, presence_state, presence_message); }
    else if (presence_state != presence(account_handle))
    { setPresence(account_handle, presence_state); }
    else if (presence_message != presenceMessage(account_handle))
    { setPresenceMessage(account_handle, presence_message); }
    else
    { emit accountUpdated(account_handle); }

    Q_ASSERT(d->m_accounts.size() == num_accounts);

    return account_handle;
}

void AccountManager::deleteAccount(const uint account_handle)
{
    int num_accounts = d->m_accounts.size();
    int num_removals = d->m_accounts.remove(account_handle);

    Q_ASSERT(num_removals == 0 || num_removals == 1);
    Q_ASSERT(num_removals != 0 || d->m_accounts.size() == num_accounts);
    Q_ASSERT(num_removals != 1 || d->m_accounts.size() == num_accounts - 1);

    if (num_removals == 1) { emit accountDeleted(account_handle); }
}

QList<uint> AccountManager::findAccounts(const QVariantMap & nv_pairs) const
{
    QList<uint> results;
    uint account_key;
    foreach (account_key, d->m_accounts.keys())
    {
        bool does_match(true);
        QString nv_key;
        foreach (nv_key, nv_pairs.keys())
        {
            if (!d->m_accounts[account_key].data.contains(nv_key) ||
                nv_pairs[nv_key].typeName() !=
                    d->m_accounts[account_key].data[nv_key].typeName() ||
                nv_pairs[nv_key] != d->m_accounts[account_key].data[nv_key])
            {
                does_match = false;
                break;
            }
        }
        if (does_match) { results.append(account_key); }
    }
    return results;
}

int AccountManager::setPresence(const uint account_handle,
                                const int presence_state)
{
    if (presence_state < QtTapioca::ContactBase::Offline ||
        presence_state > QtTapioca::ContactBase::Busy)
    { return Decibel::ERROR_INVALID_PRESENCE; }

    if (!d->gotAccount(account_handle))
    { return Decibel::ERROR_NO_SUCH_ACCOUNT; }

    if (presence(account_handle) == presence_state) { return presence_state; }

    d->m_accounts[account_handle].data[Decibel::name_presence] =
        QVariant(presence_state);

    QString presence_message = presenceMessage(account_handle);
    int result = d->manageConnection(account_handle,
                                     QtTapioca::ContactBase::Presence(presence_state),
                                     presence_message);
    return result;
}

int AccountManager::presence(const uint account_handle)
{
    if (!d->gotAccount(account_handle))
    { return Decibel::ERROR_NO_SUCH_ACCOUNT; }

    Q_ASSERT(d->m_accounts[account_handle].data.contains(Decibel::name_presence));
    int result = d->m_accounts[account_handle].data[Decibel::name_presence].toInt();
    Q_ASSERT(result >= QtTapioca::ContactBase::Offline &&
             result <= QtTapioca::ContactBase::Busy);
    return result;
}

int AccountManager::currentPresence(const uint account_handle)
{
    if (!d->gotAccount(account_handle))
    { return Decibel::ERROR_NO_SUCH_ACCOUNT; }

    Q_ASSERT(d->m_accounts[account_handle].data.contains(Decibel::name_current_presence));
    int result = d->m_accounts[account_handle].data[Decibel::name_current_presence].toInt();
    Q_ASSERT(result >= QtTapioca::ContactBase::Offline &&
             result <= QtTapioca::ContactBase::Busy);
    return result;
}

int AccountManager::setPresenceMessage(const uint account_handle,
                                       const QString & message)
{
    if (!d->gotAccount(account_handle))
    { return Decibel::ERROR_NO_SUCH_ACCOUNT; }

    int result = 0;
    if (0 != d->connectionOf(account_handle))
    {
        result = d->manageConnection(account_handle,
                                     QtTapioca::ContactBase::Presence(presence(account_handle)),
                                     message);
    }
    else
    {
        // not connected: Just update our data:
        d->m_accounts[account_handle].data[Decibel::name_presence_message] = message;
    }
    return result;
}

QString AccountManager::presenceMessage(const uint account_handle) const
{
    QString result;
    if (d->gotAccount(account_handle) &&
        d->m_accounts[account_handle].data.contains(Decibel::name_presence_message))
    {
        result = d->m_accounts[account_handle].data[Decibel::name_presence_message].toString();
    }
    return result;
}

int AccountManager::setPresenceAndMessage(const uint account_handle,
                                          const int presence_state,
                                          const QString & message)
{
    if (!d->gotAccount(account_handle))
    { return Decibel::ERROR_NO_SUCH_ACCOUNT; }
    int result = 0;
    result = d->manageConnection(account_handle,
                               QtTapioca::ContactBase::Presence(presence_state),
                               message);
    if (presence_state == QtTapioca::ContactBase::Offline)
    {
        // not connected anymore, so we won't get any signals from telepathy
        // about the message change...
        d->m_accounts[account_handle].data[Decibel::name_presence_message] = message;
    }
    return result;
}


QString AccountManager::serviceName(const uint account_handle) const
{
    if (d->connectionOf(account_handle) == 0) { return QString(); }
    return d->connectionOf(account_handle)->serviceName();
}

QString AccountManager::objectPath(const uint account_handle) const
{
    if (d->connectionOf(account_handle) == 0) { return QString(); }
    return d->connectionOf(account_handle)->objectPath();
}

void AccountManager::onOwnPresenceUpdated(QtTapioca::Connection * connection,
                                          const int presence_state,
                                          const QString & message)
{
    int account_handle = d->findAccountWithConnection(connection);
    if (0 == account_handle) { return; }

    Q_ASSERT(account_handle != 0);
    Q_ASSERT(d->gotAccount(account_handle));

    d->updatePresenceLocally(account_handle,
                             QtTapioca::ContactBase::Presence(presence_state),
                             message);
    emit accountUpdated(account_handle);
}

void AccountManager::onConnectionClosed(QtTapioca::Connection * connection)
{
    int account_handle = d->findAccountWithConnection(connection);
    if (0 == account_handle) { return; }

    Q_ASSERT(d->gotAccount(account_handle));

    d->updatePresenceLocally(account_handle, QtTapioca::ContactBase::Offline, presenceMessage(account_handle));
    d->setConnection(account_handle, 0);

    emit accountUpdated(account_handle);

    Q_ASSERT(currentPresence(account_handle) == QtTapioca::ContactBase::Offline);
}

void AccountManager::doBlockReconnects()
{ d->blockReconnects = true; }
