/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "contactconnector.h"

#include <QtCore/QSettings>
#include <QtCore/QStringList>
#include <QtCore/QDebug>

namespace
{
    static const QString uri_key("URI");

    static const QString filename("~/.decibel_contact_data.ini");
    static const QString array_name("Contacts");
} // namespace

ContactConnector::ContactConnector()
{
    QString use_filename(filename);
    if (filename.startsWith('~'))
    {
        QString home(qgetenv("HOME"));
        use_filename = home + filename.mid(1);
    }
    qDebug() << "SCC: Reading contacts from:" << use_filename;

    QSettings settings(use_filename, QSettings::IniFormat);
    int size = settings.beginReadArray(array_name);
    for (int i = 0; i < size; ++i) {
        settings.setArrayIndex(i);
        QVariantMap current_contact;

        QStringList keys = settings.childKeys();
        QString current_key;
        foreach(current_key, keys)
        { current_contact[current_key] = settings.value(current_key); }

        Q_ASSERT(current_contact.contains(uri_key));

        qDebug() << "SCC:" << (i + 1) << "uri:" << current_contact[uri_key];
        m_contacts.append(current_contact);
    }

    settings.endArray();
}

ContactConnector::~ContactConnector()
{
    QSettings settings(filename, QSettings::IniFormat);
    settings.beginWriteArray(array_name);
    int array_index = 0;
    QVariantMap contact;
    foreach (contact, m_contacts)
    {
        settings.setArrayIndex(array_index);
        ++array_index;
        QString key;
        foreach (key, contact.keys())
        { settings.setValue(key, contact[key]); }
    }
    settings.endArray();
}

bool ContactConnector::gotContact(const uint & contact_id) const
{
    if (m_contacts.size() < int(contact_id)) { return false; }
    if (0 == contact_id) { return false; }
    return true;
}

QVariantMap ContactConnector::contact(const uint & contact_id) const
{
    if (m_contacts.size() < int(contact_id)) { return QVariantMap(); }
    if (0 == contact_id) { return QVariantMap(); }
    return m_contacts[contact_id - 1];
}

QStringList ContactConnector::getURIs(const uint & contact_id,
                                                const QString & proto) const
{
    QStringList result;
    if (contact_id == 0) { return result; }
    if (m_contacts.size() < int(contact_id)) { return result; }
    Q_ASSERT(m_contacts[contact_id - 1].contains(uri_key));

    QString contact_proto = m_contacts[contact_id - 1][uri_key].toString();
    contact_proto = contact_proto.left(contact_proto.indexOf(QString("://")));
    if (contact_proto != proto) { return QStringList(); }

    result << m_contacts[contact_id - 1][uri_key].toString();
    return result;
}

unsigned int ContactConnector::findURI(const QString & uri) const
{
    for (int i = 0; i < m_contacts.size(); ++i)
    {
        Q_ASSERT(m_contacts[i].contains(uri_key));

        if (m_contacts[i][uri_key] == uri) { return (i + 1); }
    }
    return 0;
}

unsigned int ContactConnector::addContact(const QVariantMap & new_contact)
{
    m_contacts.append(new_contact);
    return m_contacts.size();
}

void ContactConnector::setPresence(const uint & contact_id,
                                             const QtTapioca::ContactBase::Presence presence_state,
                                             const QString & message)
{
    if (int(contact_id) > m_contacts.size() ||
        0 == contact_id) { return; }

    qDebug() << m_contacts[contact_id - 1]["Name"] << "has a new presence:"
             << presence_state << message;
}
