/*
 * A Decibel TextChannel GUI
 * Copyright (C) 2008 George Goldberg <grundleborg@googlemail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "chatwindow.h"

#include <QtCore/QDateTime>

ChatWindow::ChatWindow(QtTapioca::Connection * connection,
                       QtTapioca::TextChannel * channel,
                       QWidget *parent) :
    QWidget(parent)
{
    kDebug() << "New ChatWindow Created";

    setAttribute(Qt::WA_DeleteOnClose, true); // delete when closing...

    /*
     * store the thigs passed in as member variables
     */
    m_channel = channel;
    m_connection = connection;

    /*
     * set up the window GUI
     */
    setupUi(this);

    /*
     * set up the model that stores the messages
     */
    m_messagesModel = new QStandardItemModel(this);
    m_messagesView->setModel(m_messagesModel);

    /*
     * make signal/slot connections
     */
    connect(m_channel, SIGNAL(messageReceived(QtTapioca::TextChannel *,QtTapioca::TextChannel::Message)),
            this, SLOT(onMessageReceived()));
    connect(m_channel, SIGNAL(closed()), this, SLOT(onCloseChannel()));
    connect(m_sendButton, SIGNAL(clicked()), this, SLOT(sendMessage()));

    /*
     * once all the GUI stuff is done, show the window
     */
    this->show();

    /*
     * finally, process the initial message that
     * caused this chat window to be created.
     */
    onMessageReceived();
}


ChatWindow::~ChatWindow()
{

}

void ChatWindow::onCloseChannel()
{
    m_connection = 0;
    m_channel = 0;
}

void ChatWindow::onMessageReceived()
{
    /*
     * get a list of all pending messages on this channel
     */
    QList<QtTapioca::TextChannel::Message> message_list = m_channel->pendingMessages();

    /*
     * iterate over the list of messages
     */
    QList<QtTapioca::TextChannel::Message>::const_iterator message_list_end = message_list.constEnd();
    for (QList<QtTapioca::TextChannel::Message>::const_iterator message = message_list.constBegin();
         message != message_list_end;
         ++message)
    {
        /*
         * see what type of message we have received so we can
         * process it appropriately
         */
        if (message->type() == QtTapioca::TextChannel::Message::Normal)
        {

            /*
             * if it is a normal message, then we add it to the messagesModel
             * to be displayed.
             */
            QStandardItem *message_content = new QStandardItem;
            message_content->setData(message->contents(), Qt::DisplayRole);

            QStandardItem *message_time = new QStandardItem;
            QDateTime timestamp;
            timestamp.setTime_t(message->timestamp());
            message_time->setData(timestamp.toString("h:mm"), Qt::DisplayRole);

            QList<QStandardItem*> row;
            row.append(message_time);
            row.append(message_content);
            m_messagesModel->appendRow(row);
        }

        /*
         * aknowledge we received the message
         */
        m_channel->acknowledge(*message);
    }
}

void ChatWindow::sendMessage()
{
    /*
     * send the message that is currently in
     * the messageEdit
     */
    m_channel->sendMessage(m_messageEdit->text());

    /*
     * append the message to the model so we see it
     */
    QStandardItem *message_content = new QStandardItem;
    message_content->setData(m_messageEdit->text(), Qt::DisplayRole);

    QStandardItem *message_time = new QStandardItem;
    QDateTime timestamp = QDateTime::currentDateTime();
    message_time->setData(timestamp.toString("h:mm"), Qt::DisplayRole);

    QList<QStandardItem*> row;
    row.append(message_time);
    row.append(message_content);
    m_messagesModel->appendRow(row);

    /*
    * clear the message edit ready for next use.
    */
    m_messageEdit->clear();
}
