/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "accountmanager.h"
#include "accountconnector.h"
#include "accountmanageradaptor.h"
#include "connectionfacade.h"

#include <QtTapioca/Connection>
#include <QtTapioca/ContactBase>

#include <Decibel/Errors>
#include <Decibel/AccountData>

#include <QtCore/QSettings>
#include <QtCore/QHash>
#include <QtCore/QPointer>
#include <QtCore/QDebug>

namespace
{
    static const QString accountmanager_group("AccountManager");
    static const QString account_array("Accounts");
}

/// @cond INCLUDE_PRIVATE

/**
 * @brief Private class implementing the AccountManager.
 *
 * A private class implementing the AccountManager.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */
class AccountManagerPrivate
{
public:
    /**
     * @brief Constructor.
     */
    AccountManagerPrivate(ConnectionFacade * conn_mgr) :
        m_connectionFacade(conn_mgr),
        m_blockReconnects(false),
        m_connector(new AccountConnector()),
        m_data_is_available(false)
    {
        Q_ASSERT(0 != m_connectionFacade);
        Q_ASSERT(0 != m_connector);
        m_connector->openStorage();
    }

    /**
     * @brief Destructor.
     */
    ~AccountManagerPrivate()
    {
        m_connector->closeStorage();
        delete m_connector;
    }

// Helper functions:

    /**
     * @brief Create an account in the accounts database.
     * @param nv_pairs The account data.
     * @return A handle to the account.
     *
     * This method creates an account in the account database. It assumes
     * the account data is complete.
     */
    uint createAccount(const QVariantMap & nv_pairs)
    {
        Q_ASSERT(nv_pairs.contains(Decibel::name_display_name));
        Q_ASSERT(nv_pairs.contains(Decibel::name_protocol));
        Q_ASSERT(nv_pairs.contains(Decibel::name_presence));
        Q_ASSERT(nv_pairs.contains(Decibel::name_presence_message));
        Q_ASSERT(nv_pairs.contains(Decibel::name_current_presence));
        Q_ASSERT(nv_pairs.contains(Decibel::name_autoreconnect));

        return m_connector->storeAccount(nv_pairs);
    }

    /**
     * @brief Make sure a account has the required presence information.
     * @param nv_pairs The account data.
     * @returns A new set of account data.
     *
     * Make sure a account has the required presence information and fill in
     * the blanks if there are any.
     */
    QVariantMap sanityCheck(const QVariantMap & nv_pairs) const
    {
        QVariantMap pairs(nv_pairs);

        if (!pairs.contains(Decibel::name_protocol))
        {
            return QVariantMap();
        }

        if (!pairs.contains(Decibel::name_presence))
        {
            pairs.insert(Decibel::name_presence,
                         QVariant(QtTapioca::ContactBase::Offline));
        }
        else
        {
            int presence = pairs[Decibel::name_presence].toInt();
            if (presence < QtTapioca::ContactBase::Offline ||
                presence > QtTapioca::ContactBase::Busy)
            {
                pairs[Decibel::name_presence] = QVariant(QtTapioca::ContactBase::Offline);
            }
        }
        if (!pairs.contains(Decibel::name_presence_message))
        { pairs.insert(Decibel::name_presence_message, QVariant(QString())); }
        if (!pairs.contains(Decibel::name_autoreconnect))
        { pairs.insert(Decibel::name_autoreconnect, QVariant(false)); }
        if (!pairs.contains(Decibel::name_display_name))
        {
            // create a sensible display name:
            QString protocol;
            QString account;
            QString server;
            QString port;

            if (pairs.contains(Decibel::name_protocol))
            { protocol = pairs[Decibel::name_protocol].toString(); }
            if (pairs.contains(Decibel::name_tp_account))
            { account = pairs[Decibel::name_tp_account].toString(); }
            if (pairs.contains(Decibel::name_tp_server))
            { server = pairs[Decibel::name_tp_server].toString(); }
            if (pairs.contains(Decibel::name_tp_port))
            { port = pairs[Decibel::name_tp_port].toString(); }

            if (protocol.isEmpty()) { protocol = "unknown"; }
            if (account.isEmpty()) { account = "unknown"; }
            if (server.isEmpty()) { server = "unknown"; }
            if (!port.isEmpty()) { port = ':' + port; }

            QString name(protocol + "://" +
                         account + '@' + server + port + '/');
            pairs.insert(Decibel::name_display_name, QVariant(name));
        }

        if (!pairs.contains(Decibel::name_current_presence))
        {
            pairs.insert(Decibel::name_current_presence,
                         QVariant(QtTapioca::ContactBase::Offline));
        }
        else
        {
            int presence = pairs[Decibel::name_current_presence].toInt();
            if (presence < QtTapioca::ContactBase::Offline ||
                presence > QtTapioca::ContactBase::Busy)
            {
                pairs[Decibel::name_current_presence] =
                        QVariant(QtTapioca::ContactBase::Offline);
            }
        }

        Q_ASSERT(pairs.contains(Decibel::name_protocol));
        Q_ASSERT(pairs.contains(Decibel::name_presence));
        Q_ASSERT(pairs.contains(Decibel::name_current_presence));
        Q_ASSERT(pairs.contains(Decibel::name_autoreconnect));
        Q_ASSERT(pairs.contains(Decibel::name_presence_message));
        Q_ASSERT(pairs.contains(Decibel::name_display_name));
        Q_ASSERT(pairs.size() >= nv_pairs.size() &&
                 pairs.size() <= nv_pairs.size() + 5 );

        return pairs;
    }

    QVariantMap getAccount(const uint account_handle)
    { return sanityCheck(m_connector->getAccount(account_handle)); }

    /**
     * @brief Manage presence information on a connection.
     * @param account_handle A handle to an account.
     * @param presence_state The presence state.
     * @param message A presence message.
     * @return A value > 0 on success and a negative error code otherwise.
     *
     * Change the presence information on an account. This brings an account
     * on/off line as required.
     */
    int manageConnection(const uint account_handle,
                         const QtTapioca::ContactBase::Presence presence_state,
                         const QString message)
    {
        Q_ASSERT(m_connector->hasAccount(account_handle));
        Q_ASSERT(presence_state >= QtTapioca::ContactBase::Offline &&
                 presence_state <= QtTapioca::ContactBase::Busy);
        int result = presence_state;

        QtTapioca::Connection * connection = connectionOf(account_handle);

        if (presence_state == QtTapioca::ContactBase::Offline)
        {
            // We need to disconnect:
            if (0 != connection)
            { connection->disconnect(); }
            // connection pointer is set on signal!
        }
        else
        {
            if (connection == 0)
            {
                // We need to connect if we are not yet connected.
                connection =
                    m_connectionFacade->connectUsingAccount(getAccount(account_handle),
                                                            presence_state, message);
                if (connection == 0)
                {
                    // FIXME: Give proper error code!
                    return QtTapioca::ContactBase::Offline;
                }
                setConnection(account_handle, connection);
            }
            else
            {
                // update presence info
                int new_presence =
                    m_connectionFacade->updatePresence(connection,
                                                       presence_state,
                                                       message);
                if (new_presence < 0) { result = new_presence; }
            }
            Q_ASSERT(0 != connectionOf(account_handle));
        }
        return result;
    }

    /**
     * @brief Associate a account with a connection.
     * @param account_handle A handle to an account.
     * @param connection A pointer to an Connection.
     *
     * Associate an account with a connection.
     */
    void setConnection(const uint account_handle,
                       QtTapioca::Connection * connection)
    {
        Q_ASSERT(0 == connectionOf(account_handle) ||
                 0 == connection);
        if (!m_connector->hasAccount(account_handle)) { return; }
        m_connections[account_handle] = connection;
    }

    /**
     * @brief Get the connection an account is using.
     * @param account_handle A handle to an account.
     * @return A pointer to the connection the account is using. This
     *         connection will be 0 if the account is not on line.
     */
    QtTapioca::Connection * connectionOf(const uint account_handle) const
    {
        if (!m_connector->hasAccount(account_handle)) { return 0; }
        return m_connections[account_handle];
    }

    /**
     * @brief Find the account using a connection.
     * @param connection A pointer to an Connection.
     * @return A handle to an account. This handle is 0 if no account is using
     * that connection.
     *
     * Find the account using a connection.
     */
    uint findAccountWithConnection(QtTapioca::Connection * connection)
    {
        const QList<uint> values(m_connections.keys());
        foreach (uint account_handle, values)
        {
            if (m_connections[account_handle] == connection)
            { return account_handle; }
        }
        return 0;
    }

    /**
     * @brief Update presence information on an account.
     * @param account_handle A handle to an account.
     * @param presence_state The new presence state.
     * @param message The new presence message.
     *
     * Update presence information on an account, bringing it on/off line as
     * required.
     */
    void updatePresenceLocally(const uint account_handle,
                               const QtTapioca::ContactBase::Presence presence_state,
                               const QString message)
    {
        QVariantMap account_data = getAccount(account_handle);
        Q_ASSERT(!account_data.isEmpty());

        account_data[Decibel::name_current_presence] =
            QVariant(presence_state);
        if (!message.isEmpty())
        { account_data[Decibel::name_presence_message] = QVariant(message); }

        account_data = sanityCheck(account_data);

        if (account_data.isEmpty()) { return; }

        m_connector->updateAccount(account_handle, account_data);

        // Get back to intended presence state:
        if (m_blockReconnects) { return; }

        if (account_data[Decibel::name_autoreconnect].toBool())
        {
            const QtTapioca::ContactBase::Presence intended_state =
                static_cast<QtTapioca::ContactBase::Presence>(
                account_data[Decibel::name_presence].toInt());
            if (intended_state != presence_state)
            {
                manageConnection(account_handle, intended_state,
                    account_data[Decibel::name_presence_message].toString());
            }
        }
    }

    /**
     * @brief Update the presence message on an account.
     * @param account_handle A handle to an account.
     * @param message The new presence message.
     *
     * Update the presence message on an account.
     */
    void updatePresenceMessageLocally(const uint account_handle,
                                      const QString & message)
    {
        QVariantMap account_data = getAccount(account_handle);
        account_data[Decibel::name_presence_message] = QVariant(message);
        m_connector->updateAccount(account_handle, account_data);
    }

    /** @brief Account data, addressed by account_handle. */
    QHash<uint, QtTapioca::Connection *> m_connections;

    /** @brief A pointer to the ConnectionFacade */
    QPointer<ConnectionFacade> m_connectionFacade;
    /** @brief A pointer to the D-Bus Adaptor of the AccountManager. */
    QPointer<AccountManagerAdaptor> m_adaptor;

    bool m_blockReconnects;

    AccountConnector * const m_connector;

    bool m_data_is_available;

    friend class AccountManager;
};

/// @endcond

// ****************************************************************************

AccountManager::AccountManager(ConnectionFacade * connection_facade,
                               QObject * parent) :
    QObject(parent),
    d(new AccountManagerPrivate(connection_facade))
{
    Q_ASSERT(d != 0);
    d->m_adaptor = new AccountManagerAdaptor(this);
    Q_ASSERT(d->m_adaptor != 0);

    connect(connection_facade,
        SIGNAL(ownPresenceUpdated(QtTapioca::Connection *,int,QString)),
        SLOT(onOwnPresenceUpdated(QtTapioca::Connection *,int,QString)));
    connect(connection_facade,
        SIGNAL(connectionClosed(QtTapioca::Connection *)),
        SLOT(onConnectionClosed(QtTapioca::Connection *)));
    connect(d->m_connector, SIGNAL(accountDataAvailable(bool)),
            SLOT(doAccountDataAvailable(bool)));
}

AccountManager::~AccountManager()
{ delete d; }

void AccountManager::bringUpAccounts()
{
    const QList<uint> account_ids(d->m_connector->accountIds());
    foreach (uint id, account_ids)
    {
        QVariantMap account_data = d->getAccount(id);
        Q_ASSERT(!account_data.isEmpty());

        qDebug() << "Bringing up" << id << ":"
                 << account_data[Decibel::name_protocol]
                 << "intended presence:"
                 << account_data[Decibel::name_presence]
                 << "current presence:"
                 << account_data[Decibel::name_current_presence]
                 << "presence message:"
                 << account_data[Decibel::name_presence_message];

        if (account_data[Decibel::name_presence] ==
            QtTapioca::ContactBase::Offline)
        { continue; }
        setPresenceAndMessage(id,
                              account_data[Decibel::name_presence].toInt(),
                              account_data[Decibel::name_presence_message].toString());
    }
}

void AccountManager::bringDownAccounts()
{
    const QList<QtTapioca::Connection *> values(d->m_connections.values());
    foreach (QtTapioca::Connection * conn, values)
    { conn->disconnect(); }
}


void AccountManager::doAccountDataAvailable(const bool is_available)
{
    d->m_data_is_available = is_available;
    d->m_blockReconnects = !(is_available);

    emit resetData();

    if (is_available) { bringUpAccounts(); }
    else { bringDownAccounts(); }
}

QString AccountManager::protocol(const uint account_handle) const
{ return d->m_connector->protocol(account_handle); }

bool AccountManager::gotAccount(const uint account_handle) const
{ return d->m_connector->hasAccount(account_handle); }

QtTapioca::Connection *
AccountManager::connectionOf(const uint account_handle) const
{ return d->connectionOf(account_handle); }

QList<uint> AccountManager::listAccounts() const
{ return d->m_connector->accountIds(); }

QVariantMap AccountManager::queryAccount(const uint account_handle) const
{
    if (!gotAccount(account_handle))
    {
        sendErrorReply(Decibel::ErrorNoSuchAccount,
                       tr("Account %1 not found.", "1: account_handle").
                           arg(account_handle));
    }
    return d->getAccount(account_handle);
}

uint AccountManager::addAccount(const QVariantMap & nv_pairs)
{
    QList<uint> matching_accounts = findAccounts(nv_pairs);
    if (!matching_accounts.isEmpty())
    {
        Q_ASSERT(matching_accounts.size() == 1);
        return matching_accounts[0];
    }

    QVariantMap data = d->sanityCheck(nv_pairs);
    if (data.isEmpty())
    {
        sendErrorReply(Decibel::ErrorDataIncomplete,
                       tr("Account data was incomplete"));
        return 0;
    }
    uint account_handle = d->createAccount(data);

    if (account_handle != 0) { emit accountCreated(account_handle); }
    else
    {
        sendErrorReply(Decibel::ErrorInternalError,
                       tr("Failed to store account information."));
        return 0;
    }

    return account_handle;
}

void AccountManager::updateAccount(const uint account_handle,
                                   const QVariantMap & nv_pairs)
{
    if (!d->m_connector->hasAccount(account_handle))
    {
        sendErrorReply(Decibel::ErrorNoSuchAccount,
                       tr("Account %1 not found.", "1: account_handle").
                           arg(account_handle));
        return;
    }

    QVariantMap data(nv_pairs);
    // Remove internal stuff:
    data.remove(Decibel::name_current_presence);
    int presence_state = data[Decibel::name_presence].toInt();
    QString presence_message = data[Decibel::name_presence_message].toString();
    data.remove(Decibel::name_presence);
    data.remove(Decibel::name_presence_message);

    // merge old and new data:
    QVariantMap old_data = queryAccount(account_handle);
    const QStringList values(old_data.keys());
    foreach (const QString & key, values)
    {
        if (!data.contains(key)) { data.insert(key, old_data[key]); }
    }

    // make sure the data is sane:
    data = d->sanityCheck(data);

    if (data.isEmpty())
    {
        sendErrorReply(Decibel::ErrorDataIncomplete,
                       tr("Update invalidates the data."));
        return;
    }

    int old_presence_state(old_data[Decibel::name_presence].toInt());
    QString old_presence_message(old_data[Decibel::name_presence_message].toString());

    d->m_connector->updateAccount(account_handle, data);

    if (presence_state != old_presence_state &&
        presence_message != old_presence_message)
    { setPresenceAndMessage(account_handle, presence_state, presence_message); }
    else if (presence_state != old_presence_state)
    { setPresence(account_handle, presence_state); }
    else if (presence_message != old_presence_message)
    { setPresenceMessage(account_handle, presence_message); }

    emit accountUpdated(account_handle);
}

void AccountManager::deleteAccount(const uint account_handle)
{
    if (d->m_connector->deleteAccount(account_handle))
    { emit accountDeleted(account_handle); }
}

QList<uint> AccountManager::findAccounts(const QVariantMap & nv_pairs) const
{ return d->m_connector->findAccounts(nv_pairs); }

int AccountManager::setPresence(const uint account_handle,
                                const int presence_state)
{
    if (presence_state < QtTapioca::ContactBase::Offline ||
        presence_state > QtTapioca::ContactBase::Busy)
    {
        sendErrorReply(Decibel::ErrorInvalidValue,
                       tr("Presence is not valid."));
        return -1;
    }

    if (!d->m_connector->hasAccount(account_handle))
    {
        sendErrorReply(Decibel::ErrorNoSuchAccount,
                       tr("Account %1 not found.", "1: account_handle").
                           arg(account_handle));
        return -1;
    }

    if (presence(account_handle) == presence_state) { return presence_state; }

    d->m_connector->setValue(account_handle, Decibel::name_presence,
                             QVariant(presence_state));

    QString presence_message = presenceMessage(account_handle);
    int result = d->manageConnection(account_handle,
                                     QtTapioca::ContactBase::Presence(presence_state),
                                     presence_message);
    return result;
}

int AccountManager::presence(const uint account_handle)
{
    if (!d->m_connector->hasAccount(account_handle))
    {
        sendErrorReply(Decibel::ErrorNoSuchAccount,
                       tr("Account %1 not found.", "1: account_handle").
                           arg(account_handle));
        return -1;
    }

    QVariantMap account_data = d->getAccount(account_handle);
    int result = account_data[Decibel::name_presence].toInt();
    Q_ASSERT(result >= QtTapioca::ContactBase::Offline &&
             result <= QtTapioca::ContactBase::Busy);
    return result;
}

int AccountManager::currentPresence(const uint account_handle)
{
    if (!d->m_connector->hasAccount(account_handle))
    {
        sendErrorReply(Decibel::ErrorNoSuchAccount,
                       tr("Account %1 not found.", "1: account_handle").
                           arg(account_handle));
        return -1;
    }

    QVariantMap account_data = d->getAccount(account_handle);
    int result = account_data[Decibel::name_current_presence].toInt();
    Q_ASSERT(result >= QtTapioca::ContactBase::Offline &&
             result <= QtTapioca::ContactBase::Busy);
    return result;
}

int AccountManager::setPresenceMessage(const uint account_handle,
                                       const QString & message)
{
    if (!d->m_connector->hasAccount(account_handle))
    {
        sendErrorReply(Decibel::ErrorNoSuchAccount,
                       tr("Account %1 not found.", "1: account_handle").
                           arg(account_handle));
        return -1;
    }

    int result = 0;
    if (0 != d->connectionOf(account_handle))
    {
        result = d->manageConnection(account_handle,
                                     QtTapioca::ContactBase::Presence(presence(account_handle)),
                                     message);
    }
    else
    {
        // not connected: Just update our data:
        d->m_connector->setValue(account_handle,
                                 Decibel::name_presence_message,
                                 QVariant(QString(message)));
    }
    return result;
}

QString AccountManager::presenceMessage(const uint account_handle) const
{
    QString result;
    QVariantMap account_data = d->getAccount(account_handle);
    if (!account_data.isEmpty() &&
        account_data.contains(Decibel::name_presence_message))
    { result = account_data[Decibel::name_presence_message].toString(); }
    return result;
}

int AccountManager::setPresenceAndMessage(const uint account_handle,
                                          const int presence_state,
                                          const QString & message)
{
    QVariantMap account_data = d->getAccount(account_handle);
    if (account_data.isEmpty())
    {
        sendErrorReply(Decibel::ErrorNoSuchAccount,
                       tr("Account %1 not found.", "1: account_handle").
                           arg(account_handle));
        return -1;
    }

    int result = 0;
    result = d->manageConnection(account_handle,
                                 QtTapioca::ContactBase::Presence(presence_state),
                                 message);

    if (presence_state == QtTapioca::ContactBase::Offline)
    {
        // not connected anymore, so we won't get any signals from telepathy
        // about the message change...
        d->m_connector->setValue(account_handle, Decibel::name_presence,
                                 QVariant(presence_state));
        d->m_connector->setValue(account_handle,
                                 Decibel::name_presence_message,
                                 QVariant(message));
    }
    return result;
}


QString AccountManager::serviceName(const uint account_handle) const
{
    if (d->connectionOf(account_handle) == 0) { return QString(); }
    return d->connectionOf(account_handle)->serviceName();
}

QString AccountManager::objectPath(const uint account_handle) const
{
    if (d->connectionOf(account_handle) == 0) { return QString(); }
    return d->connectionOf(account_handle)->objectPath();
}

void AccountManager::onOwnPresenceUpdated(QtTapioca::Connection * connection,
                                          const int presence_state,
                                          const QString & message)
{
    int account_handle = d->findAccountWithConnection(connection);
    if (0 == account_handle) { return; }

    Q_ASSERT(account_handle != 0);
    Q_ASSERT(d->m_connector->hasAccount(account_handle));

    d->updatePresenceLocally(account_handle,
                             QtTapioca::ContactBase::Presence(presence_state),
                             message);
    emit accountUpdated(account_handle);
}

void AccountManager::onConnectionClosed(QtTapioca::Connection * connection)
{
    int account_handle = d->findAccountWithConnection(connection);
    if (0 == account_handle) { return; }

    Q_ASSERT(d->m_connector->hasAccount(account_handle));

    d->updatePresenceLocally(account_handle, QtTapioca::ContactBase::Offline,
                             presenceMessage(account_handle));
    d->setConnection(account_handle, 0);

    emit accountUpdated(account_handle);

    Q_ASSERT(currentPresence(account_handle) == QtTapioca::ContactBase::Offline);
}

void AccountManager::doBlockReconnects()
{ d->m_blockReconnects = true; }
