/* kbd-interface.cpp
 *  functions for implementing the customize keyboard dialog
 *
 *  for Denemo, thu GNU graphical music notation editor
 *  (c) 2000, 2001, 2002 Matthew Hiller 
 */

#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "kbd-custom.h"

#define CLIST_WIDTH 180
#define CLIST_HEIGHT 180

struct callbackdata
{
  GtkWidget *kbd_window;
  GtkWidget *commands_list;
  GtkWidget *binding_list;
  GtkWidget *hbox1;
  GtkWidget *buttontable;
  GtkWidget *statusbar1;
  gint statusbar1_context_id;
  gint command_selected;
  gint keybinding_number_selected;
  struct keymap *the_keymap;
};

void
set_binding_list (GtkWidget * commands_list, gint row, gint column,
		  GdkEventButton * event, struct callbackdata *cbdata)
{
  GList *iterator;
  struct keybinding_info *ki;
  gint i;
  gchar *rowtext[1];

  gtk_clist_freeze (GTK_CLIST (cbdata->binding_list));
  gtk_clist_clear (GTK_CLIST (cbdata->binding_list));
  for (iterator = cbdata->the_keymap->commands[row], i = 0; iterator;
       iterator = iterator->next, i++)
    {
      ki = (struct keybinding_info *)iterator->data;
      rowtext[0] = g_strdup_printf ("%s (%d)", gdk_keyval_name (ki->keyval),
				    ki->state);
      gtk_clist_append (GTK_CLIST (cbdata->binding_list), rowtext);
      g_free (rowtext[0]);
    }
  gtk_clist_thaw (GTK_CLIST (cbdata->binding_list));
  cbdata->command_selected = row;
  cbdata->keybinding_number_selected = -1;
}

void
set_keybinding_number (GtkWidget * binding_list, gint row, gint column,
		       GdkEventButton * event, struct callbackdata *cbdata)
{
  cbdata->keybinding_number_selected = row;
}

static void
set_widget_sensitivity (struct callbackdata *cbdata, gboolean boolean)
{
  gtk_widget_set_sensitive (cbdata->hbox1, boolean);
  gtk_widget_set_sensitive (cbdata->buttontable, boolean);
}

static void
jump (GtkWidget * kbd_window, GdkEventKey * event,
      struct callbackdata *cbdata)
{
  /* This check for modifier values on the event may not be right,
     if the contents of gdkkeysyms.h are OS-dependent. I don't believe
     they are. */
  if (event->keyval < GDK_Shift_L || event->keyval > GDK_Hyper_R)
    {
      struct keybinding_info *ki;

      if ((ki = lookup_keybinding (cbdata->the_keymap, event->keyval,
				  event->state)))
	{
	  gtk_clist_select_row (GTK_CLIST (cbdata->commands_list),
				ki->command_number, 0);
	  gtk_clist_moveto (GTK_CLIST (cbdata->commands_list),
			    ki->command_number, 0, 0.5, 0.0);
	  gtk_statusbar_push (GTK_STATUSBAR (cbdata->statusbar1),
			      cbdata->statusbar1_context_id, "");
	}
      else
	gtk_statusbar_push (GTK_STATUSBAR (cbdata->statusbar1),
			    cbdata->statusbar1_context_id,
			    _("Binding not found"));
      set_widget_sensitivity (cbdata, TRUE);
      gtk_signal_disconnect_by_func (GTK_OBJECT (kbd_window), 
      				     GTK_SIGNAL_FUNC(jump), cbdata);
    }
}

static void
add (GtkWidget * kbd_window, GdkEventKey * event, struct callbackdata *cbdata)
{
  /* This check for modifier values on the event may not be right,
     if the contents of gdkkeysyms.h are OS-dependent. I don't believe
     they are. */
  if ((event->keyval < GDK_Shift_L || event->keyval > GDK_Hyper_R)
      && cbdata->command_selected != -1)
    {
      gint command_returned;
      gchar *text;

      /* If the key combination entered was already bound, remark
         upon this on the status bar */
      if ((command_returned = add_keybinding (cbdata->the_keymap,
					      event->keyval,
					      event->state,
					      cbdata->command_selected)) !=
	  -1)
	{
	  text =
	    g_strdup_printf (_("Previous mapping of %s (%d) to %s removed"),
			     gdk_keyval_name (event->keyval),
			     MASK_FILTER (event->state),
			     _(denemo_commands[command_returned].name));

	  gtk_statusbar_push (GTK_STATUSBAR (cbdata->statusbar1),
			      cbdata->statusbar1_context_id, text);
	  g_free (text);
	}
      else
	gtk_statusbar_push (GTK_STATUSBAR (cbdata->statusbar1),
			    cbdata->statusbar1_context_id, "");
      set_widget_sensitivity (cbdata, TRUE);
      set_binding_list (cbdata->commands_list, cbdata->command_selected,
			0, NULL, cbdata);
      gtk_signal_disconnect_by_func (GTK_OBJECT (kbd_window), 
      				     GTK_SIGNAL_FUNC(add), cbdata);
    }
}

void
listen_then_jump (GtkWidget * button_keybinding_search,
		  struct callbackdata *cbdata)
{
  set_widget_sensitivity (cbdata, FALSE);
  gtk_statusbar_push (GTK_STATUSBAR (cbdata->statusbar1),
		      cbdata->statusbar1_context_id,
		      _("Type the keystroke to look for"));
  gtk_signal_connect (GTK_OBJECT (cbdata->kbd_window), "key_press_event",
		      GTK_SIGNAL_FUNC (jump), cbdata);
}

/* Some duplicated code in this function, but nothing terrible */

void
listen_then_add (GtkWidget * button_add_keybinding,
		 struct callbackdata *cbdata)
{
  if (cbdata->command_selected != -1)
    {
      set_widget_sensitivity (cbdata, FALSE);
      gtk_statusbar_push (GTK_STATUSBAR (cbdata->statusbar1),
			  cbdata->statusbar1_context_id,
			  _("Type the keystroke to add for this command"));
      gtk_signal_connect (GTK_OBJECT (cbdata->kbd_window), "key_press_event",
			  GTK_SIGNAL_FUNC (add), cbdata);
    }
}

void
remove_binding (GtkWidget * button_remove_binding,
		struct callbackdata *cbdata)
{
  if (cbdata->command_selected != -1
      && cbdata->keybinding_number_selected != -1)
    {
      struct keybinding_info *ki;

      ki = (struct keybinding_info *)g_list_nth (cbdata->the_keymap->commands[cbdata->command_selected],
		       cbdata->keybinding_number_selected)->data;
      remove_keybinding (cbdata->the_keymap, ki->keyval, ki->state);
      set_binding_list (cbdata->commands_list, cbdata->command_selected, 0,
			NULL, cbdata);
    }
}

void
configure_keyboard_dialog (struct scoreinfo *si, guint callback_action,
			   GtkWidget * widget)
{
  GtkWidget *kbd_window;
  GtkWidget *vbox3;
  GtkWidget *hbox1;
  GtkWidget *vbox4;
  GtkWidget *commandlabel;
  GtkWidget *scrolled_window1;
  GtkWidget *commands_list;
  GtkWidget *hbuttonbox2;
  GtkWidget *button_keybinding_search;
  GtkWidget *vbox5;
  GtkWidget *binding_list_label;
  GtkWidget *scrolled_window2;
  GtkWidget *binding_list;
  GtkWidget *hbuttonbox3;
  GtkWidget *button_add_binding;
  GtkWidget *button_remove_binding;
  GtkWidget *hrule;
  GtkWidget *buttontable;
  GtkWidget *button_ok;
  GtkWidget *button_ok_save;
  GtkWidget *button_save;
  GtkWidget *button_save_as;
  GtkWidget *button_load;
  GtkWidget *button_load_from;
  GtkWidget *statusbar1;
  struct keymap *the_keymap = si->prefs->the_keymap;
  GtkTooltips *tooltips;
  gint i;
  gchar *rowtext[1];
  static struct callbackdata cbdata;

  tooltips = gtk_tooltips_new ();

  cbdata.the_keymap = the_keymap;
#if GTK_MAJOR_VERSION > 1
  kbd_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
#else
  kbd_window = gtk_window_new (GTK_WINDOW_DIALOG);
#endif
  gtk_window_set_title (GTK_WINDOW (kbd_window), _("Customize Keybindings"));
  cbdata.kbd_window = kbd_window;

  vbox3 = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox3);
  gtk_container_add (GTK_CONTAINER (kbd_window), vbox3);

  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox1);
  gtk_box_pack_start (GTK_BOX (vbox3), hbox1, TRUE, TRUE, 0);
  cbdata.hbox1 = hbox1;

  vbox4 = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox4);
  gtk_box_pack_start (GTK_BOX (hbox1), vbox4, TRUE, TRUE, 0);

  commandlabel = gtk_label_new (_("Available Commands:"));
  gtk_widget_show (commandlabel);
  gtk_box_pack_start (GTK_BOX (vbox4), commandlabel, FALSE, FALSE, 0);
  gtk_label_set_justify (GTK_LABEL (commandlabel), GTK_JUSTIFY_LEFT);

  scrolled_window1 = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window1),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (vbox4), scrolled_window1, TRUE, TRUE, 0);
  gtk_widget_show (scrolled_window1);

  commands_list = gtk_clist_new (1);
  gtk_clist_set_selection_mode (GTK_CLIST (commands_list),
				GTK_SELECTION_SINGLE);
  gtk_clist_set_reorderable (GTK_CLIST (commands_list), FALSE);

  /* Actually initialize the elements of the list */

  for (i = 0; i < denemo_commands_size; i++)
    {
      rowtext[0] = _(denemo_commands[i].name);
      gtk_clist_append (GTK_CLIST (commands_list), rowtext);
    }
  gtk_widget_set_usize (commands_list, CLIST_WIDTH, CLIST_HEIGHT);
  gtk_clist_set_column_auto_resize (GTK_CLIST (commands_list), 0, TRUE);
  gtk_widget_show (commands_list);
  gtk_container_add (GTK_CONTAINER (scrolled_window1), commands_list);
  cbdata.commands_list = commands_list;

  hbuttonbox2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (hbuttonbox2);
  gtk_box_pack_start (GTK_BOX (vbox4), hbuttonbox2, FALSE, TRUE, 0);

  button_keybinding_search = gtk_button_new_with_label (_("Search"));
  gtk_widget_show (button_keybinding_search);
  gtk_box_pack_start (GTK_BOX (hbuttonbox2), button_keybinding_search, FALSE,
		      TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button_keybinding_search, GTK_CAN_DEFAULT);
  gtk_tooltips_set_tip (tooltips, button_keybinding_search,
			_
			("Click here, then enter a keystroke. Denemo will jump to the command it corresponds to in the Available Commands list."),
			NULL);

  vbox5 = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox5);
  gtk_box_pack_start (GTK_BOX (hbox1), vbox5, TRUE, TRUE, 0);

  binding_list_label = gtk_label_new (_("Bindings for Selected Command:\n"));
  gtk_widget_show (binding_list_label);
  gtk_box_pack_start (GTK_BOX (vbox5), binding_list_label, FALSE, FALSE, 0);
  gtk_label_set_justify (GTK_LABEL (binding_list_label), GTK_JUSTIFY_LEFT);

  scrolled_window2 = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window2),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_widget_show (scrolled_window2);
  gtk_box_pack_start (GTK_BOX (vbox5), scrolled_window2, TRUE, TRUE, 0);

  binding_list = gtk_clist_new (1);
  gtk_clist_set_reorderable (GTK_CLIST (binding_list), FALSE);
  gtk_widget_set_usize (binding_list, CLIST_WIDTH, CLIST_HEIGHT);
  gtk_clist_set_column_auto_resize (GTK_CLIST (binding_list), 0, TRUE);
  gtk_widget_show (binding_list);
  gtk_container_add (GTK_CONTAINER (scrolled_window2), binding_list);
  cbdata.binding_list = binding_list;

  hbuttonbox3 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (hbuttonbox3);
  gtk_box_pack_start (GTK_BOX (vbox5), hbuttonbox3, FALSE, TRUE, 0);

  button_add_binding = gtk_button_new_with_label (_("Add Binding"));
  gtk_widget_show (button_add_binding);
  gtk_box_pack_start (GTK_BOX (hbuttonbox3), button_add_binding, FALSE,
		      TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button_add_binding, GTK_CAN_DEFAULT);

  button_remove_binding
    = gtk_button_new_with_label (_("Delete Selected Binding"));
  gtk_widget_show (button_remove_binding);
  gtk_box_pack_start (GTK_BOX (hbuttonbox3), button_remove_binding, FALSE,
		      TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button_remove_binding, GTK_CAN_DEFAULT);

  hrule = gtk_hseparator_new ();
  gtk_widget_show (hrule);
  gtk_box_pack_start (GTK_BOX (vbox3), hrule, FALSE, FALSE, 0);

  buttontable = gtk_table_new (3, 2, FALSE);
  gtk_widget_show (buttontable);
  gtk_box_pack_start (GTK_BOX (vbox3), buttontable, FALSE, FALSE, 0);
  cbdata.buttontable = buttontable;

  button_ok_save = gtk_button_new_with_label (_("OK and Save As Default"));
  gtk_widget_show (button_ok_save);
  gtk_table_attach_defaults (GTK_TABLE (buttontable), button_ok_save,
			     0, 1, 0, 1);
  GTK_WIDGET_SET_FLAGS (button_ok_save, GTK_CAN_DEFAULT);

  button_ok = gtk_button_new_with_label (_("OK (no save)"));
  gtk_widget_show (button_ok);
  gtk_table_attach_defaults (GTK_TABLE (buttontable), button_ok, 1, 2, 0, 1);
  GTK_WIDGET_SET_FLAGS (button_ok, GTK_CAN_DEFAULT);

  button_save = gtk_button_new_with_label (_("Save to Default Keymap File"));
  gtk_widget_show (button_save);
  gtk_table_attach_defaults (GTK_TABLE (buttontable), button_save,
			     0, 1, 1, 2);
  GTK_WIDGET_SET_FLAGS (button_save, GTK_CAN_DEFAULT);

  button_save_as =
    gtk_button_new_with_label (_("Save As Alternate Keymap File"));
  gtk_widget_show (button_save_as);
  gtk_table_attach_defaults (GTK_TABLE (buttontable), button_save_as,
			     1, 2, 1, 2);
  GTK_WIDGET_SET_FLAGS (button_save_as, GTK_CAN_DEFAULT);

  button_load = gtk_button_new_with_label (_("Load Default Keymap File"));
  gtk_widget_show (button_load);
  gtk_table_attach_defaults (GTK_TABLE (buttontable), button_load,
			     0, 1, 2, 3);
  GTK_WIDGET_SET_FLAGS (button_load, GTK_CAN_DEFAULT);

  button_load_from =
    gtk_button_new_with_label (_("Load Alternate Keymap File"));
  gtk_widget_show (button_load_from);
  gtk_table_attach_defaults (GTK_TABLE (buttontable), button_load_from,
			     1, 2, 2, 3);
  GTK_WIDGET_SET_FLAGS (button_load_from, GTK_CAN_DEFAULT);

  statusbar1 = gtk_statusbar_new ();
  cbdata.statusbar1_context_id
    = gtk_statusbar_get_context_id (GTK_STATUSBAR (statusbar1), "kbd_custom");
  gtk_widget_show (statusbar1);
  gtk_box_pack_start (GTK_BOX (vbox3), statusbar1, FALSE, FALSE, 0);
  cbdata.statusbar1 = statusbar1;

  cbdata.command_selected = -1;
  cbdata.keybinding_number_selected = -1;

  /* Now that the GUI's set up, set up the callbacks */

  gtk_signal_connect (GTK_OBJECT (commands_list), "select_row",
		      GTK_SIGNAL_FUNC (set_binding_list), &cbdata);
  gtk_signal_connect (GTK_OBJECT (button_keybinding_search), "clicked",
		      GTK_SIGNAL_FUNC (listen_then_jump), &cbdata);
  gtk_signal_connect (GTK_OBJECT (binding_list), "select_row",
		      GTK_SIGNAL_FUNC (set_keybinding_number), &cbdata);
  gtk_signal_connect (GTK_OBJECT (button_add_binding), "clicked",
		      GTK_SIGNAL_FUNC (listen_then_add), &cbdata);
  gtk_signal_connect (GTK_OBJECT (button_remove_binding), "clicked",
		      GTK_SIGNAL_FUNC (remove_binding), &cbdata);

  gtk_signal_connect_object (GTK_OBJECT (button_ok), "clicked",
			     GTK_SIGNAL_FUNC(gtk_widget_destroy), 
			     GTK_OBJECT (kbd_window));
  gtk_signal_connect (GTK_OBJECT (button_ok_save), "clicked",
		      GTK_SIGNAL_FUNC (save_standard_keymap_file_wrapper),
		      the_keymap);
  gtk_signal_connect_object (GTK_OBJECT (button_ok_save), "clicked",
			     GTK_SIGNAL_FUNC(gtk_widget_destroy), 
			     GTK_OBJECT (kbd_window));
  gtk_signal_connect (GTK_OBJECT (button_save), "clicked",
		      GTK_SIGNAL_FUNC (save_standard_keymap_file_wrapper),
		      the_keymap);
  gtk_signal_connect (GTK_OBJECT (button_save_as), "clicked",
		      GTK_SIGNAL_FUNC (save_keymap_dialog), the_keymap);
  gtk_signal_connect (GTK_OBJECT (button_load), "clicked",
		      GTK_SIGNAL_FUNC (load_standard_keymap_file_wrapper),
		      the_keymap);
  gtk_signal_connect (GTK_OBJECT (button_load_from), "clicked",
		      GTK_SIGNAL_FUNC (load_keymap_dialog), the_keymap);

  /* And show the window */
  gtk_window_set_modal (GTK_WINDOW (kbd_window), TRUE);
  gtk_window_set_position (GTK_WINDOW (kbd_window), GTK_WIN_POS_MOUSE);
  gtk_widget_show (kbd_window);
}
