/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*-
 *
 * Copyright (C) 2008 David Zeuthen <davidz@redhat.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <glib-object.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include "devkit-device.h"

/**
 * SECTION:devkit-device
 * @short_description: Details about a device
 * @include: devkit-gobject/devkit-gobject.h
 *
 * This class wraps details about a device. Instances of this class are returned
 * to signals handlers for the #DevkitClient::device-event signal and in the result
 * for enumeration functions such as devkit_client_enumerate_by_subsystem().
 **/

struct _DevkitDevicePrivate
{
        gchar       *subsystem;
        gchar       *native_path;
        gchar       *device_file;
        gchar      **device_file_symlinks;
        GHashTable  *properties;
};

G_DEFINE_TYPE (DevkitDevice, devkit_device, G_TYPE_OBJECT)

static void
devkit_device_finalize (GObject *object)
{
        DevkitDevice *device = DEVKIT_DEVICE (object);

        g_free (device->priv->subsystem);
        g_free (device->priv->native_path);
        g_free (device->priv->device_file);
        g_strfreev (device->priv->device_file_symlinks);
        g_hash_table_unref (device->priv->properties);

        if (G_OBJECT_CLASS (devkit_device_parent_class)->finalize)
                (* G_OBJECT_CLASS (devkit_device_parent_class)->finalize) (object);
}

static void
devkit_device_class_init (DevkitDeviceClass *klass)
{
        GObjectClass *gobject_class = (GObjectClass *) klass;

        gobject_class->finalize = devkit_device_finalize;

        g_type_class_add_private (klass, sizeof (DevkitDevicePrivate));
}

static void
devkit_device_init (DevkitDevice *device)
{
        device->priv = G_TYPE_INSTANCE_GET_PRIVATE (device,
                                                    DEVKIT_TYPE_DEVICE,
                                                    DevkitDevicePrivate);
}

DevkitDevice *
_devkit_device_new (const gchar   *subsystem,
                    const gchar   *native_path,
                    const gchar   *device_file,
                    const gchar  **device_file_symlinks,
                    GHashTable    *properties);

static void
_copy_props (gpointer key, gpointer value, gpointer user_data)
{
        g_hash_table_insert (user_data, g_strdup (key), g_strdup (value));
}

DevkitDevice *
_devkit_device_new (const gchar   *subsystem,
                    const gchar   *native_path,
                    const gchar   *device_file,
                    const gchar  **device_file_symlinks,
                    GHashTable    *properties)
{
        DevkitDevice *device;

        device =  DEVKIT_DEVICE (g_object_new (DEVKIT_TYPE_DEVICE, NULL));
        device->priv->subsystem = g_strdup (subsystem);
        device->priv->native_path = g_strdup (native_path);
        device->priv->device_file = g_strdup (device_file);
        device->priv->device_file_symlinks = g_strdupv ((gchar **) device_file_symlinks);
        device->priv->properties = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
        g_hash_table_foreach (properties, _copy_props, device->priv->properties);

        return device;
}

/**
 * devkit_device_get_subsystem:
 * @device: a #DevkitDevice.
 *
 * Get the OS specific subsystem for @device.
 *
 * Returns: the OS-specific subsystem for @device.
 **/
const gchar *
devkit_device_get_subsystem (DevkitDevice *device)
{
        return device->priv->subsystem;
}

/**
 * devkit_device_get_native_path:
 * @device: a #DevkitDevice.
 *
 * Get the OS specific native path for @device.
 *
 * Returns: the OS-specific native path for @device.
 **/
const gchar *
devkit_device_get_native_path (DevkitDevice *device)
{
        return device->priv->native_path;
}

/**
 * devkit_device_get_device_file:
 * @device: a #DevkitDevice.
 *
 * Get the UNIX device file, if any, for @device.
 *
 * Returns: the UNIX device file for @device or %NULL if no UNIX device file exists.
 **/
const gchar *
devkit_device_get_device_file (DevkitDevice *device)
{
        return (const gchar *) device->priv->device_file;
}

/**
 * devkit_device_get_device_file_symlinks:
 * @device: a #DevkitDevice.
 *
 * Get a list of symlinks (in <literal>/dev</literal>) that points to the UNIX device for @device.
 *
 * Returns: a %NULL terminated string array of symlinks. This array is
 * owned by @device and should not be freed by the caller.
 **/
const gchar **
devkit_device_get_device_file_symlinks (DevkitDevice             *device)
{
        return (const gchar **) device->priv->device_file_symlinks;
}

typedef struct {
        DevkitDevice            *device;
        gboolean                 short_circuited;
        DevkitDeviceForeachFunc  func;
        gpointer                 user_data;
} PropForeachData;

static void
prop_foreach (const gchar *key,
              const gchar *value,
              gpointer     user_data)
{
        PropForeachData *data = user_data;

        if (data->short_circuited)
                return;

        data->short_circuited = data->func (data->device, key, value, data->user_data);
}

/**
 * devkit_device_properties_foreach:
 * @device: a #DevkitDevice.
 * @func: a callback function of type #DevkitDeviceForeachFunc
 * @user_data: data to pass to @func
 *
 * Iterates over all properties of @device.
 *
 * Returns: %TRUE only if @func short-circuited the iteration.
 **/
gboolean
devkit_device_properties_foreach (DevkitDevice             *device,
                                  DevkitDeviceForeachFunc   func,
                                  gpointer                  user_data)
{
        PropForeachData data;

        data.device = device;
        data.short_circuited = FALSE;
        data.func = func;
        data.user_data = user_data;

        g_hash_table_foreach (device->priv->properties, (GHFunc) prop_foreach, &data);

        return data.short_circuited;
}

/**
 * devkit_device_has_property:
 * @device: a #DevkitDevice.
 * @key: name of property.
 *
 * Check if a the property with the given key exists.
 *
 * Returns: %TRUE only if the value for @key exist.
 **/
gboolean
devkit_device_has_property (DevkitDevice             *device,
                            const gchar              *key)
{
        return g_hash_table_lookup (device->priv->properties, key) != NULL;
}

/**
 * devkit_device_get_property:
 * @device: a #DevkitDevice.
 * @key: name of property.
 *
 * Look up the value for @key on @device.
 *
 * Returns: the value for @key or %NULL if @key doesn't exist on @device.
 **/
const gchar *
devkit_device_get_property (DevkitDevice *device,
                            const gchar  *key)
{
        return g_hash_table_lookup (device->priv->properties, key);
}

/**
 * devkit_device_get_property_as_int:
 * @device: a #DevkitDevice.
 * @key: name of property.
 *
 * Look up the value for @key on @device and convert it to an integer.
 *
 * If @key doesn't exist a warning will be printed on stderr.
 *
 * Returns: the value for @key or #G_MAXINT if @key doesn't exist or isn't an integer.
 **/
gint
devkit_device_get_property_as_int (DevkitDevice *device,
                                   const gchar  *key)
{
        gint result;
        const gchar *s;

        result = G_MAXINT;
        s = g_hash_table_lookup (device->priv->properties, key);
        if (s == NULL) {
                g_warning ("no property %s on %s", key, device->priv->native_path);
                goto out;
        }

        result = strtol (s, NULL, 0);
out:
        return result;
}

/**
 * devkit_device_get_property_as_uint64:
 * @device: a #DevkitDevice.
 * @key: name of property.
 *
 * Look up the value for @key on @device and convert it to an unsigned 64-bit integer.
 *
 * If @key doesn't exist a warning will be printed on stderr.
 *
 * Returns: the value for @key or #G_MAXUINT64 if @key doesn't exist or isn't a #guint64.
 **/
guint64
devkit_device_get_property_as_uint64 (DevkitDevice *device,
                                      const gchar  *key)
{
        guint64 result;
        const gchar *s;

        result = G_MAXUINT64;
        s = g_hash_table_lookup (device->priv->properties, key);
        if (s == NULL) {
                g_warning ("no property %s on %s", key, device->priv->native_path);
                goto out;
        }

        result = strtoll (s, NULL, 0);
out:
        return result;
}

/**
 * devkit_device_get_property_as_boolean:
 * @device: a #DevkitDevice.
 * @key: name of property.
 *
 * Look up the value for @key on @device and convert it to an boolean.
 *
 * If @key doesn't exist a warning will be printed on stderr.
 *
 * Returns: the value for @key or %FALSE if @key doesn't exist or isn't a #gboolean.
 **/
gboolean
devkit_device_get_property_as_boolean (DevkitDevice *device,
                                       const gchar  *key)
{
        gboolean result;
        const gchar *s;

        result = FALSE;
        s = g_hash_table_lookup (device->priv->properties, key);
        if (s == NULL) {
                g_warning ("no property %s on %s", key, device->priv->native_path);
                goto out;
        }

        if (strcmp (s, "1") == 0 || strcmp (s, "true") == 0 || strcmp (s, "TRUE") == 0 || strcmp (s, "True") == 0)
                result = TRUE;
out:
        return result;
}

/**
 * devkit_device_dup_property_as_str:
 * @device: a #DevkitDevice.
 * @key: name of property.
 *
 * Look up the value for @key on @device and return a duplicate string.
 *
 * Returns: the value of @key on @device or %NULL if @key doesn't exist. Caller must free this string with g_free().
 **/
gchar *
devkit_device_dup_property_as_str (DevkitDevice *device,
                                   const gchar  *key)
{
        return g_strdup (g_hash_table_lookup (device->priv->properties, key));
}

/**
 * devkit_device_dup_property_as_strv:
 * @device: a #DevkitDevice.
 * @key: name of property.
 *
 * Look up the value for @key on @device and return the result of
 * splitting it into tokens split at white space points.
 *
 * Returns: the value of @key on @device split into tokens or %NULL if @key doesn't exist. Caller must free this string array with g_strfreev().
 **/
gchar **
devkit_device_dup_property_as_strv (DevkitDevice *device,
                                    const gchar  *key)
{
        gchar **result;
        const gchar *s;

        result = NULL;
        s = g_hash_table_lookup (device->priv->properties, key);
        if (s == NULL)
                goto out;

        result = g_strsplit (s, " \t", 0);
out:
        return result;
}
