/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/frame/logging_facility.hpp
 *
 * @brief [LEVEL: beta] The @ref diagnostics::Logging_Facility facade
 *
 * $Id: logging_facility.hpp,v 1.18 2005/06/23 09:54:20 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @test diagnostics/frame/logging_facility.t.cpp
 */

#ifndef DIAGNOSTICS__FRAME__LOGGING_FACILITY_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__FRAME__LOGGING_FACILITY_HPP__INCLUDE_GUARD

// used in the interface by value
#include <diagnostics/frame/level.hpp>

// used in the interface by value
#include <diagnostics/frame/type.hpp>

// used for the macro interface
#include <diagnostics/frame/pp_config.hpp>

// used in the interface by reference
#include <string>

DIAGNOSTICS_NAMESPACE_BEGIN;

/**
 * @class Logging_Facility diagnostics/frame/logging_facility.hpp
 *
 * @brief The Logging_Facility facade is used to send log messages
 * into the logging framework. 
 *
 * The Logging_Facility is designed to be a minimal interface which
 * has to be known by a user of the diagnostic framework. The macros
 * and guards of the files in macros/ are built atop Logging_Facility
 * and do not use @ref Logging_Config. 
 *
 * @nosubgrouping 
 */
class Logging_Facility
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Static Modifiers
     * @{
     */
public:
    /**
     * @brief The main interface into the logging framework: logging a
     * message. Calls to @ref Logging_Config and to
     * Logging_Facility::log are mutexed.
     *
	 * @a level is the level of the log message.
	 *
	 * @a type is the type of log message.
	 *
     * @a nr_what is a numerical field which is for unconstrained use
     * by the client. It is intended to allow the client quick
     * classification of the log-messages. I use it in most cases for
     * describing a further log-level of production level log-messages
     * and for encoding the packge which is generating the log-message. 
     * However, this is a personal convention.
     *
     * @a str_what is a string field which is for unconstrained use by
     * the client. All components of this library produce str_what
     * fields with entries of the form "FIELD=\"content\"".  Again,
     * this is a convention -- which I personally adopted for other
     * packages, too.
     *
     * @a base_file_name contains the name of the file which was the
     * root of the include graph, i.e., if an error in an included
     * pice of code occurs, @a base_file_name is the name of file
     * which included the code. @a file_name is set to __FILE__ and @a
     * line to __LINE__.
     *
     * These arguments are used to construct a @ref
     * Record which is then presented to each @ref Logger
     * which is registered to the logging framework (via @ref
     * Logging_Config::register_logger).
	 *
	 * The missing parts of the @ref Record, i.e., pid, tid, date, and
	 * hostname are supplemented by the implementation of log. 
     *
     * @throw never -- internal errors are reported via
     * @a panic_log
     *
     * @attention The first call @ref Logging_Facility::log or to one
     * of methods of @ref Logging_Config might cause a call to abort,
     * if the initialization of the logging framework fails. To
     * enforce initialization, call @ref Logging_Config::init.
     */
    static void log(Level_t const level,
		    Type_t const type,
		    int const nr_what,
		    ::std::string const & str_what,
		    char const * const base_file_name,
		    char const * const file_name,
		    int const line);

    /**
     * @brief panic_log is used inside of the logging framework and by
     * implementations @ref Logger to generate error messages.
     * 
     * For the arguments, see @ref log. The effect of calling @ref
     * panic_log is simple: The arugments are printed via ::std::cerr. 
	 *
	 * @attention This call is not mutexed
     *
     * @throw never -- and if it throws, everything is lost. 
     */
    static void panic_log(::std::string const & what,
			  char const * const base_file_name,
			  char const * const file_name,
			  int const line);
    // @}
};


DIAGNOSTICS_NAMESPACE_END;


/**
 * @brief basis for all logging -- adds the basefile, file and line
 * and sends calls @ref diagnostics::Logging_Facility::log.
 *
 * @param LEVEL must be of type @ref diagnostics::Level_t
 * @param TYPE must of type @ref diagnostics::Type_t
 * @param NR_WHAT must be convertible to int (numerical field which is uninterpreted by the frame)
 * @param STR_WHAT must be convertible to ::std::string (string field which is uninterpreted by the frame)
 */
#define DIAGNOSTICS_BASE_LOG(LEVEL,TYPE,NR_WHAT,STR_WHAT) \
::DIAGNOSTICS_NAMESPACE::Logging_Facility::log( \
(LEVEL), \
(TYPE),  \
(NR_WHAT), \
(STR_WHAT), \
DIAGNOSTICS_BASE_FILE, \
__FILE__, \
__LINE__)


/**
 * @brief basis for all panic logging -- adds the basefile, file and line
 * and sends calls @ref diagnostics::Logging_Facility::panic_log.
 *
 * @param WHAT must be convertible to ::std::string
 */
#define DIAGNOSTICS_PANIC_LOG(WHAT) \
::DIAGNOSTICS_NAMESPACE::Logging_Facility::panic_log( \
 WHAT, \
DIAGNOSTICS_BASE_FILE, \
__FILE__, \
__LINE__)


#endif

// vim:ts=4:sw=4
