/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/test_logger.cpp
 *
 * @brief [LEVEL: beta] Implementation of @ref
 * diagnostics::unittest::Test_Logger Class
 *
 * $Id: test_logger.cpp,v 1.15 2005/06/23 09:54:27 esdentem Exp $
 * 
 * @author Christian Schallhart
 */

#include <diagnostics/unittest/test_system/test_logger.hpp>

#include <diagnostics/unittest/test_system_exception.hpp>

#include <diagnostics/unittest/test_system/test_run_result.hpp>

// for DIAGNOSTICS_PANIC_LOG
#include <diagnostics/frame/logging_facility.hpp>


#include <diagnostics/util/to_string.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

Test_Logger::Test_Logger(Level_t const build_level,
			 Level_t const target_level,
			 Test_Run_Results_t & test_run_results) 
    : m_build_level(build_level),
      m_target_level(target_level),
      m_test_run_results(test_run_results)
{
    if(build_level==LEVEL_TEST) throw Test_System_Exception("build_level!=LEVEL_TEST failed");
    if(target_level==LEVEL_TEST) throw Test_System_Exception("target_level!=LEVEL_TEST failed");
    if(build_level==LEVEL_SYSTEM) throw Test_System_Exception("build_level!=LEVEL_SYSTEM failed");
    if(target_level==LEVEL_SYSTEM) throw Test_System_Exception("target_level!=LEVEL_SYSTEM failed");
    if(build_level<target_level) throw Test_System_Exception("build_level>=target_level");
}

Test_Logger::~Test_Logger()
{
}

Test_Logger::Abstract_State_t Test_Logger::abstract_state() const
{
    if(m_test_run_results.size()==0) return STATE_PASSIVE;
    return (m_test_run_results.back()->abstract_state()==Test_Run_Result::STATE_INCOMPLETE) 
	? STATE_ACTIVE
	: STATE_PASSIVE;
}


void Test_Logger::log(Record const & record)
{
    using ::diagnostics::internal::to_string;

    if(m_test_run_results.size()==0) {
	if(record.type()==TYPE_TESTCASE_ENTER) {
	    m_test_run_results.push_back(new Test_Run_Result(m_build_level,m_target_level));
	    m_test_run_results.back()->add_record(record);
	}
	else if(record.type()==TYPE_TESTCASE_EXIT)
	    DIAGNOSTICS_PANIC_LOG("TYPE_TESTCASE_EXIT without TYPE_TESTCASE_ENTER: " + to_string(record));
	return;
    }

    switch(m_test_run_results.back()->abstract_state()){
	case Test_Run_Result::STATE_COMPLETE:
	    if(record.type()==TYPE_TESTCASE_ENTER) {
		m_test_run_results.push_back(new Test_Run_Result(m_build_level,m_target_level));
		m_test_run_results.back()->add_record(record);
	    }
	    else if(record.type()==TYPE_TESTCASE_EXIT)
		DIAGNOSTICS_PANIC_LOG("TYPE_TESTCASE_EXIT without TYPE_TESTCASE_ENTER: " + to_string(record));
	    break;
	case Test_Run_Result::STATE_INCOMPLETE:
	    if(record.level()==LEVEL_TEST 
	       || record.type()==TYPE_FAILED_ASSERTION
	       || record.type()==TYPE_UNEXPECTED_EXCEPTION
	       || (record.type()==TYPE_FAILED_CHECK && record.level()>m_target_level)) 
		{
		    try{
			m_test_run_results.back()->add_record(record);
		    }
		    catch(Test_System_Exception & e){
			DIAGNOSTICS_PANIC_LOG(::std::string("excep.name=\"") + e.name()+"\" execp.what=\""+e.what()+"\"");
		    }
		    catch(...){
			DIAGNOSTICS_PANIC_LOG("Unknown Exception");
		    }
		}
	    break;
	case Test_Run_Result::STATE_EMPTY:
	    DIAGNOSTICS_PANIC_LOG("empty TEST_RUN_RESULT -- should never happen: " + to_string(record));
	    break;
    }
}

void Test_Logger::close()
{
}


UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;
// vim:ts=4:sw=4
