/* $Progeny: pci.c 4245 2004-04-09 10:49:39Z imurdock $ */

/* pci.c - Scan the PCI bus
 *
 * AUTHOR: John R. Daily <jdaily@progeny.com>
 *
 * Copyright 2002 Hewlett-Packard Company
 * Copyright 2001, 2002 Progeny Linux Systems, Inc.
 * Copyright (C) 1998-2000 MandrakeSoft
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>

#include <sysdep.h>
#include <utils.h>

/*
 * We just need a path length sufficiently long for paths like
 * /proc/bus/pci/02/07.0
 */
#ifdef _MAX_PATH_LEN
#undef _MAX_PATH_LEN
#endif
#define _MAX_PATH_LEN 256

/*
 * This function is specific to each sysdep.
 */
static void
_discover_sysdep_init(discover_sysdep_data_t *node)
{
    node->busclass = _discover_xmalloc(5);
    node->vendor = _discover_xmalloc(5);
    node->model = _discover_xmalloc(5);
}

discover_sysdep_data_t *
_discover_get_pci_raw(void)
{
    FILE *devices_file, *individual_device_file;
    char *line = NULL;
    size_t len = 0;
    discover_sysdep_data_t *head = NULL, *node, *last = NULL;
    unsigned int i;
    unsigned int bus;
    unsigned int slot;
    unsigned int function;
    unsigned int sfcombo;  /* Slot/function combination byte */
    char file_path[_MAX_PATH_LEN];
    u_int16_t blocks[6];

    if ((devices_file = fopen(PATH_PROC_PCI, "r"))) {
        while (getline(&line, &len, devices_file) >= 0) {
            if (line[0] == '\n' || line[0] == '#') {
                continue;
            }

            node = _discover_sysdep_data_new();
            _discover_sysdep_init(node);

            if (sscanf(line, "%02x%02x\t%4s%4s", &bus, &sfcombo,
                       node->vendor, node->model) != 4) {
                perror("Failed to read expected data from devices file");
                exit(-1);
            }

            /*
             * The first 5 bits of the 2nd bite refer to the slot in the
             * PCI bus where this device can be found.  The last 3 bits
             * describe the function this card is performing in this
             * context.
             */
            slot = (sfcombo & 0xf8) >> 3;
            function = (sfcombo & 0x7);


            /*
             * Now we can read the busclass from the individual PCI file
             * in /proc/bus/pci/<bus>/<function>.<slot> and extract the
             * device type.
             */
            if (snprintf(file_path, _MAX_PATH_LEN, "%s/%02x/%02x.%1x",
                        PATH_PROC_PCI_DIR,
                        bus, slot, function) >= _MAX_PATH_LEN) {
                fprintf(stderr, "Path to PCI device file too long.\n");
                exit(-1);
            }

            individual_device_file = fopen(file_path, "r");

            if (!individual_device_file) {
                perror("Unable to locate device file");
                exit(-1);
            }

            if (fread(blocks, sizeof(u_int16_t), 6,
                      individual_device_file) != 6) {
                perror("Unable to read device type from file");
                exit(-1);
            }

            fclose(individual_device_file);

            snprintf(node->busclass, 5, "%.4x", blocks[5]);

            if (head == NULL) {
                head = node;
                last = head;
            } else {
                last->next = node;
                last = node;
            }
        }
        free(line);
        fclose(devices_file);
    }
    return head;
}


/*
 * Local variables:
 * c-file-style: "progeny"
 * indent-tabs-mode: nil
 * End:
 */
/* vim: set cin fo=tcroq sw=4 et sts=4 tw=75: */
