/**
 * @file sysdep.c
 * @brief Interact with system dependencies
 *
 * This file holds the routines that interact with the system-dependent
 * interface.  This file essentially bridges the gap between the XML data
 * and the hardware contained within the machine.
 */

/* $Progeny: sysdep.c 4368 2004-05-13 19:57:37Z licquia $
 *
 * AUTHOR: John R. Daily <jdaily@progeny.com>
 *
 * Copyright 2002 Progeny Linux Systems, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
#include <assert.h>

#include <stdio.h>
#include <string.h>

#include <discover/discover.h>
#include <discover/discover-conf.h>
#include <discover/discover-xml.h>
#include <discover/sysdep.h>

#include <discover/device.h>
#include <discover/utils.h>


/** Function pointers */
typedef discover_sysdep_data_t *(raw_sysdep_function_t)(void);

static raw_sysdep_function_t *raw_map[] = {
    _discover_get_ata_raw,
    _discover_get_pci_raw,
    _discover_get_pcmcia_raw,
    _discover_get_scsi_raw,
    _discover_get_usb_raw
};


/*
 * Utility functions
 */

/** Create a new instance of sysdep data. */
discover_sysdep_data_t *
_discover_sysdep_data_new(void)
{
    discover_sysdep_data_t *node =
        _discover_xmalloc(sizeof(discover_sysdep_data_t));
    node->busclass = NULL;
    node->vendor = NULL;
    node->model = NULL;
    node->next = NULL;
    return node;
}

/** Release the memory that the sysdep data was holding. */
void
_discover_free_sysdep_data(discover_sysdep_data_t *head)
{
    discover_sysdep_data_t *node;

    while (head) {
        node = head->next;
        if (head->vendor) {
            free(head->vendor);
        }
        if (head->model) {
            free(head->model);
        }
        if (head->busclass) {
            free(head->busclass);
        }
        free(head);
        head = node;
    }
}

/*
 * Functions that access the sysdeps
 */
static discover_device_t *devices[BUS_COUNT];

discover_device_t *
discover_get_devices(discover_bus_t bus, discover_error_t *status)
{
    discover_device_t *device, *last;
    discover_device_t *xml_devices;
    discover_bus_map_t *busmap;
    discover_sysdep_data_t *head, *node;

    assert(status);

    status->code = DISCOVER_SUCCESS;
    device = last = NULL;

    busmap = discover_conf_get_bus_map(bus, status);
    if (status->code != 0) {
        return NULL;
    }

    if (busmap->scan_never) {
        status->code = DISCOVER_EBUSDISABLED;
        return NULL;
    }

    if (devices[bus]) {
        return devices[bus];
    }

    xml_devices = discover_xml_get_devices(bus, status);
    if (!xml_devices) {
        return NULL;
    }

    /*
     * Allow overrides of this function.
     */
    if (busmap->get_raw) {
        head = node = busmap->get_raw();
    } else {
        head = node = raw_map[bus]();
    }

    while (node) {
        /*
         * The variable names are ambiguous:
         *   node   -  Hardware data retrieved from the OS
         *   device -  Matching hardware data from our database
         */
        device =
            discover_xml_get_matching_devices(xml_devices, node->vendor,
                                              node->model, status);

        if (!device) {
            device = discover_device_new();
            device->model_id = strdup(node->model);
            device->vendor_id = strdup(node->vendor);
        }

        /*
         * If we get a busclass from the hardware, treat it as
         * canonical.
         */
        if (node->busclass != NULL) {
            if (device->busclass != NULL) {
                free(device->busclass);
            }
            device->busclass = strdup(node->busclass);
        }

        if (last) {
            last->next = device;
            last = device;
        } else {
            devices[bus] = last = device;
        }

        node = node->next;
    }

    _discover_free_sysdep_data(head);

    return devices[bus];
}

void
discover_free_devices(void)
{
    int i;

    for (i = 0; i < BUS_COUNT; i++) {
        discover_device_free(devices[i], 0);
        devices[i] = NULL;
    }
}

/*
 * Local variables:
 * c-file-style: "progeny"
 * indent-tabs-mode: nil
 * End:
 */
/* vim: set cin fo=tcroq sw=4 et sts=4 tw=75: */
