#! /usr/bin/env python

#Common imports
import os, sys
import unittest
import DiskSearchTest
sys.path.append(os.path.join(os.path.dirname(os.path.abspath(__file__)), '..'))

#import specific to the test
from libdisksearch import DiskList
from libdisksearch import Disk
import ConstantsForTests
import tempfile

class TestCase(DiskSearchTest.TestCase):
    def setUp(self):
        self.diskList = DiskList.DiskList()
        self.dirname = None
        self.filename = None

    def tearDown(self):
        self.rmtree(self.dirname)
        self.rmfile(self.filename)

    def testInit(self):
        mediaTypes = (
            ('CD', 'CD'),
            ('DVD', 'DVD'),
            ('DISK', 'Disk'),
            ('ZIP', 'ZIP disk'),
            ('CARD', 'Memory card'),
            ('TAPE', 'Tape'),
        )
        self.assertListsEqual(self.diskList.media_types, mediaTypes)
        self.assertEqual(self.diskList.modified, 0)
        self.assertEqual(len(self.diskList.content), 0)
        self.assertListsEqual(self.diskList.content, [])

    def testRead_disk_listEmpty (self):
        self.diskList.read_disk_list('nonexistent')
        self.assertListsEqual(self.diskList.content, [])

    def testRead_disk_listNormal (self):
        self.filename = os.tempnam('/tmp', 'ds')
        fh = open(self.filename, 'w')
        fh.write(ConstantsForTests.database)
        fh.close()
        self.diskList.read_disk_list(self.filename)
        self.assertEqual(len(self.diskList.content), 1)
        self.assertListsEqual(self.diskList.content[0].content, ConstantsForTests.diskContentList)

    def testSave_disk_list (self):
        disk1 = Disk.Disk('name123', 'type456', 'remark789', '1234567890', ConstantsForTests.diskContentList)
        self.filename = os.tempnam('/tmp', 'ds')
        self.diskList.add_scanned_disk(disk1)
        self.diskList.save_disk_list(self.filename)
        self.assertFileContains(self.filename, ConstantsForTests.database)

    def createDirectories(self, dirname):
        dir1 = os.path.join(dirname, "example-1")
        dir2 = os.path.join(dirname, "example-2")
        dir21 = os.path.join(dirname, "example-2", "nested_1")
        dir211 = os.path.join(dirname, "example-2", "nested_1", "nested_2")
        os.mkdir(dir1)
        os.mkdir(dir2)
        os.mkdir(dir21)
        os.mkdir(dir211)
        self.createFile(os.path.join(dir1, 'file_1'), 5)
        self.createFile(os.path.join(dir1, 'file_2'), 6)
        self.createFile(os.path.join(dir2, 'file_8'), 11)
        self.createFile(os.path.join(dir21, 'file_3'), 7)
        self.createFile(os.path.join(dir21, 'file_4'), 7)
        self.createFile(os.path.join(dir211, 'file_5'), 8)
        self.createFile(os.path.join(dir211, 'file_6'), 9)
        self.createFile(os.path.join(dirname, 'file_7'), 10)

    def testAdd_disk(self):
        self.dirname = tempfile.mkdtemp('tmp', 'dsd', '/tmp')
        self.createDirectories(self.dirname)
        self.diskList.add_disk ('name123', 'type456', 'remark987654', self.dirname)
        self.assertEqual(len(self.diskList.content), 1)
        self.assertStringsEqual(self.diskList.content[0].name, 'name123')
        self.assertStringsEqual(self.diskList.content[0].type, 'type456')
        self.assertStringsEqual(self.diskList.content[0].remark, 'remark987654')
        self.assertListsEqual(self.diskList.content[0].content, ConstantsForTests.diskContentList)

    def testScan_directory(self):
        self.dirname = tempfile.mkdtemp('tmp', 'dsd', '/tmp')
        self.createDirectories(self.dirname)
        expected = ConstantsForTests.diskContentList
        content = []
        self.diskList.scan_directory(self.dirname, content)
        self.assertListsEqual(content, expected)

    def testAdd_scanned_disk(self):
        disk1 = Disk.Disk('BBB', '', '', '', '')
        disk2 = Disk.Disk('aaa', '', '', '', '')
        disk3 = Disk.Disk('abc', '', '', '', '')
        self.diskList.add_scanned_disk(disk1)
        self.assertEqual(len(self.diskList.content), 1)
        self.assertStringsEqual(self.diskList.content[0].name, 'BBB')
        self.diskList.add_scanned_disk(disk2)
        self.assertEqual(len(self.diskList.content), 2)
        self.assertStringsEqual(self.diskList.content[0].name, 'aaa')
        self.assertStringsEqual(self.diskList.content[1].name, 'BBB')
        self.diskList.add_scanned_disk(disk3)
        self.assertEqual(len(self.diskList.content), 3)
        self.assertStringsEqual(self.diskList.content[0].name, 'aaa')
        self.assertStringsEqual(self.diskList.content[1].name, 'abc')
        self.assertStringsEqual(self.diskList.content[2].name, 'BBB')

    def testRemove_disk(self):
        disk1 = Disk.Disk('BBB', '', '', '', '')
        disk2 = Disk.Disk('aaa', '', '', '', '')
        disk3 = Disk.Disk('abc', '', '', '', '')
        self.diskList.add_scanned_disk(disk1)
        self.diskList.add_scanned_disk(disk2)
        self.diskList.add_scanned_disk(disk3)
        self.assertEqual(len(self.diskList.content), 3)
        self.diskList.remove_disk('abc')
        self.assertEqual(len(self.diskList.content), 2)
        self.assertStringsEqual(self.diskList.content[0].name, 'aaa')
        self.assertStringsEqual(self.diskList.content[1].name, 'BBB')

def suite():
    suite = unittest.TestSuite()
    suite.addTest(unittest.makeSuite(TestCase))
    return suite

if __name__ == '__main__':
    unittest.main()
