#
# act.py - DITrack 'act' command
#
# Copyright (c) 2006-2007 The DITrack Project, www.ditrack.org.
#
# $Id: act.py 1251 2007-02-02 06:39:51Z gli $
# $HeadURL: https://127.0.0.1/ditrack/src/tags/0.5/DITrack/Command/act.py $
#
# Redistribution and use in source and binary forms, with or without 
# modification, are permitted provided that the following conditions are met:
#
#  * Redistributions of source code must retain the above copyright notice, 
# this list of conditions and the following disclaimer.
#  * Redistributions in binary form must reproduce the above copyright notice, 
# this list of conditions and the following disclaimer in the documentation 
# and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
# POSSIBILITY OF SUCH DAMAGE.
#

import copy
import email
import os
import sys

# DITrack modules
import DITrack.Command.generic
import DITrack.DB
import DITrack.Edit
import DITrack.UI
import DITrack.Util.common

class Handler(DITrack.Command.generic.Handler):
    canonical_name = "act"

    # XXX: replace ISSUENUM with ISSUEID later
    description = """Perform actions on an issue (or multiple issues).
usage: %s ISSUENUM [ISSUENUM...]""" % canonical_name

    def run(self, opts, globals):
        self.check_options(opts)

        if len(opts.fixed) < 2:
            self.print_help(globals)
            sys.exit(1)

        # We'll need an editor.
        globals.get_editor()

        db = DITrack.Util.common.open_db(globals, opts)

        globals.get_username(opts, db)

        present_categories = {}
        present_vsets = {}

        issue = {}
        prev_issue = {}
        for id in opts.fixed[1:]:

            try:
                issue[id] = db.issue_by_id(id)
            except (KeyError, ValueError):
                # Diagnostics printed by issue_by_id().
                pass

            if db.is_valid_issue_name(id):
                DITrack.Util.common.err("Non-local identifier expected '%s'" % id)

            prev_issue[id] = copy.deepcopy(issue[id])

            present_categories[issue[id].info["Category"]] = 1

            present_vsets[
                db.cfg.category[
                issue[id].info["Category"]
                ].version_set] = 1

        issue_numbers = issue.keys()
        issue_numbers.sort()

        same_category = (len(present_categories.keys()) == 1)
        same_vset = (len(present_vsets.keys()) == 1)
        single_issue = (len(issue) == 1)

        # Build up the menu.
        mi_abort = DITrack.UI.MenuItem("a", "abort, discarding changes")
        mi_ch_due_in = DITrack.UI.MenuItem("d", "change due version")
        mi_close = DITrack.UI.MenuItem("c", "close the issue")
        mi_edit_info = DITrack.UI.MenuItem("h", "edit the issue header")
        mi_edit_text = DITrack.UI.MenuItem("e", "edit comment text")
        mi_quit = DITrack.UI.MenuItem("q", "quit, saving changes")
        mi_reassign = DITrack.UI.MenuItem("o", "reassign the issue owner")
        mi_reopen = DITrack.UI.MenuItem("r", "reopen the issue")

        menu = DITrack.UI.Menu("Choose an action for the issue(s)",
            [
            mi_abort,
            mi_ch_due_in,
            mi_close,
            mi_edit_info,
            mi_edit_text,
            mi_quit,
            mi_reassign,
            mi_reopen
            ])

        save_changes = 0

        comment_text = ""

        mi_ch_due_in.enabled = same_vset
        mi_edit_info.enabled = single_issue

        while 1:

            # Conditionally enable/disable menu items.
            mi_close.enabled = filter(lambda x: x.info["Status"] == "open",
                issue.itervalues())

            mi_reopen.enabled = filter(
                lambda x: x.info["Status"] == "closed",
                issue.itervalues())

            print
            print "Acting on:"

            for id in issue_numbers:
                print "i#%s: %s" % (id, issue[id].info["Title"])

            print

            r = menu.run()
            if r == mi_abort:
                break

            elif r == mi_close:

                mi_c_abort = DITrack.UI.MenuItem("a", "abort closing")
                mi_c_dropped = DITrack.UI.MenuItem("d", "dropped")
                mi_c_fixed = DITrack.UI.MenuItem("f", "fixed")
                mi_c_invalid = DITrack.UI.MenuItem("i", "invalid")

                resolution_menu = DITrack.UI.Menu(
                    "Choose the resolution",
                    [
                        mi_c_abort,
                        mi_c_dropped,
                        mi_c_fixed,
                        mi_c_invalid
                    ])

                r = resolution_menu.run()

                if r != mi_c_abort:
                    if r == mi_c_dropped:
                        resolution = "dropped"
                    elif r == mi_c_fixed:
                        resolution = "fixed"
                    elif r == mi_c_invalid:
                        resolution = "invalid"

                    assert(resolution)

                    for i in issue.values():

                        i.replace_header("Status", "closed")

                        assert "Resolution" not in i.info
                        i.info["Resolution"] = resolution

            elif r == mi_ch_due_in:
                assert(same_vset)

                any_issue = issue[issue_numbers[0]]
                if single_issue:
                    sys.stdout.write("Current due version: %s\n" %
                        any_issue.info["Due-in"])

                assert("Category" in any_issue.info)
                assert(any_issue.info["Category"] in db.cfg.category)

                due_menu = DITrack.UI.EnumMenu("Choose new due version",
                    db.cfg.category[
                        any_issue.info["Category"]].versions.future)

                v = due_menu.run()

                if not v: break

                for i in issue.values():
                    i.replace_header("Due-in", v)

            elif r == mi_edit_info:
                id = issue_numbers[0]
                
                info = email.Message.Message()

                for k, v in issue[id].info.iteritems():
                    info.add_header(k, v)

                header = DITrack.Edit.edit_text(globals, info.as_string())

                issue[id].info = email.message_from_string(header)

            elif r == mi_edit_text:
                comment_text = DITrack.Edit.edit_text(globals,
                        comment_text)

            elif r == mi_quit:
                save_changes = 1
                break

            elif r == mi_reassign:

                if single_issue:
                    print "Current issue owner: " + issue[issue_numbers[0]].info["Owned-by"]

                users = db.cfg.users.keys()
                users.sort()
                owner_menu = DITrack.UI.EnumMenu("Choose new issue owner",
                    users)

                v = owner_menu.run()

                if not v: break

                for i in issue.values():
                    i.replace_header("Owned-by", v)

            elif r == mi_reopen:
                for i in issue.values():
                    if "Resolution" in i.info:
                        del i.info["Resolution"]

                    i.replace_header("Status", "open")

            else:
                raise NotImplementedError

        if save_changes:

            local_names = []

            for id in issue_numbers:

                try:
                    name, comment = db.new_comment(id, prev_issue[id], 
                        issue[id], comment_text, globals.username, 
                        globals.fmt_timestamp())

                    local_names.append((id, name))

                except DITrack.DB.NoDifferenceCondition:
                    continue

                sys.stdout.write("Comment %s added to issue %s\n" % (name, id))

            if not opts.var["no_commits"]:
                # Now commit newly added comments. We do it in a separate step 
                # to simplify the solution for now. If something goes wrong 
                # (like no disk space or connectivity issues), a user may 
                # choose to commit the changes later.

                for issue_id, comment_name in local_names:

                    # XXX: for now we don't deal with commenting local issues.
                    assert db.is_valid_issue_number(issue_id)
                    
                    firm_id = db.commit_comment(issue_id, comment_name)

                    # XXX: print 'Local ... in r234'.
                    sys.stdout.write("Local i#%s.%s committed as i#%s.%s\n" % \
                        (issue_id, comment_name, issue_id, firm_id))
