#
# DB.py - database interface
#
# Copyright (c) 2006-2007 The DITrack Project, www.ditrack.org.
#
# $Id: DB.py 1326 2007-02-26 03:46:34Z vss $
# $HeadURL: https://127.0.0.1/ditrack/src/tags/0.5/DITrack/DB.py $
#
# Redistribution and use in source and binary forms, with or without 
# modification, are permitted provided that the following conditions are met:
#
#  * Redistributions of source code must retain the above copyright notice, 
# this list of conditions and the following disclaimer.
#  * Redistributions in binary form must reproduce the above copyright notice, 
# this list of conditions and the following disclaimer in the documentation 
# and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
# POSSIBILITY OF SUCH DAMAGE.
#

import copy
import email
import email.Message
import logging
import os.path
import pprint
import re
import shelve
import string
import sys
import traceback

# DITrack modules
import DITrack.SVN

# Root database directory property containing format version.
VERSION_PROP = "ditrack:format"

# Current database format version.
FORMAT_VERSION = "2"

# Local modifications area directory and file name
LMA_DIR = ".ditrack"
LMA_FILE = "LMA"

# Comment file name regular expression.
comment_fname_re = re.compile("^comment(\\d+)$")

#
# Corrupted DB exceptions
# 
class CorruptedDBError(Exception):
    message = "Database is corrupted"

class CorruptedDB_DuplicateUserError(CorruptedDBError):
    message = CorruptedDBError.message + ": duplicate user account entry"

class CorruptedDB_UnparseableCategoriesError(CorruptedDBError):
    message = CorruptedDBError.message \
        + ": unparseable categories configuration"

class CorruptedDB_UnparseableVersionsError(CorruptedDBError):
    message = CorruptedDBError.message \
        + ": unparseable version sets configuration"

class CorruptedDB_UnparseableFiltersError(CorruptedDBError):
    message = CorruptedDBError.message \
        + ": unparseable filter configuration"

#
# Other exceptions
#
class NotDirectoryError(Exception):
    message = "Database path is not an [existing] directory"

class NotDatabaseError(Exception):
    message = "Path specified is not an issue database root"

class InvalidVersionError(Exception):
    message = "Database format is not supported"

class IssueExistsError(Exception):
    message = "Cannot add new issue: correponding directory already exists"

class IssueIdSyntaxError(Exception):
    message = "Invalid issue identifier syntax"

class FilterExpressionError(Exception):
    message = "Syntax error in filter expression"

class FilterIsPredefinedError(Exception):
    message = "This looks like a predefined filter name"

#
# Not errors, but various conditions.
#
class NoDifferenceCondition(Exception):
    message = "No difference between passed entities"

#
# Functions
#

def get_caller():
    """
    Returns the grand-caller of the function in a form of string "[FILE:LINE]".
    """

    tb = traceback.extract_stack(limit=3)[0]

    s = os.path.split(tb[0])

    fname = os.path.join(os.path.split(s[0])[1], s[1])

    return "[%s:%d]" % (fname, tb[1])

def DEBUG(str):
    """
    Sends STR to logging.debug(), prepending it with the file name and line of
    the caller.
    """

    logging.debug("%s %s" % (get_caller(), str))

def entity_id_split(str):
    """
    Splits entity id STR into issue and comment parts. Ensures the string 
    passed conforms to the syntax (otherwise ValueError is raised). Returns
    tuple (ISSUE, COMMENT), where COMMENT may be None. The values returned are
    strings. Names are uppercase.
    """

    s = str.strip().upper().split(".")

    if len(s) < 1 or len(s) > 2 or len(s[0]) == 0:
        raise ValueError
    
    issue_numeric = False

    def ensure(str, allowed):
        for x in str:
            if x not in allowed:
                raise ValueError

    if s[0][0] in string.digits:
        # Numeric id
        ensure(s[0], string.digits)

        issue = s[0]
        issue_numeric = True
    else:
        # Name
        ensure(s[0], string.uppercase)
        issue = s[0]

    if len(s) > 1 and s[1]:
        if s[1][0] in string.digits:
            if not issue_numeric:
                # Case of 'A.1': we cannot have committed comment to 
                # uncommitted issue.
                raise ValueError

            ensure(s[1], string.digits)
            comment = s[1]
        else:
            # Name
            ensure(s[1], string.uppercase)
            comment = s[1]

    else:
        if "." in str:
            # Case of '1.' or 'A.'.
            raise ValueError

        comment = None

    return issue, comment

def next_entity_name(names):
    """
    Generates next entity name, given the set of existing ones.

    The sequence is A, B .. Z, AA, AB, .. AZ, BA, BB, ... ZZ
    """

    if names:
        names = copy.copy(names)
        names.sort()
        prev = names[-1]
    else:
        return "A"

    assert(len(prev))

    res = ""
    carry = True
    for c in prev[::-1]:
        assert(c >= "A")
        assert(c <= "Z")

        was_carry = carry
        carry = False

        if was_carry:
            if c == "Z":
                res = "A" + res
                carry = True
            else:
                res = chr(ord(c) + 1) + res
        else:
            res = c + res

    if carry:
        res = "A" + res

    return res

#
# Classes
#
class VersionSet:
    """
    A representation of a single version set.
    """

    def __init__(self, past, current, future):
        self.past = past
        self.current = current
        self.future = future

class VersionCfg:
    """
    A representation of version sets configuration.
    """

    def __contains__(self, key):
        return self.items.has_key(key)

    def __getitem__(self, key):
        return self.items[key]

    def __init__(self, path):
        f = open(path)
        versions = email.message_from_file(f)
        f.close()

        if len(versions.get_payload()):
            raise CorruptedDB_UnparseableVersionsError("Empty line in version "
                "set configuration file")

        self.items = {}
        for s in versions.keys():

            if s in self.items:
                raise CorruptedDB_UnparseableVersionsError("Duplicate version "
                    "set '" + s + "' definition")

            v = versions[s].strip().split("/")
            if len(v) != 3:
                raise CorruptedDB_UnparseableVersionsError("Invalid version "
                    "set '" + s + "' definition")

            self.items[s] = VersionSet(
                v[0].strip().split(),
                v[1].strip().split(),
                v[2].strip().split()
                )

class Category:
    def __init__(self, version_set, versions, default_owner, enabled):
        """
        VERSION_SET
            Version set name.

        XXX: do we really need to pass both version_set and versions?
        """

        self.version_set = version_set
        self.versions = versions
        self.default_owner = default_owner
        self.enabled = enabled

class CategoryCfg(object):
    """
    Representation of per-database category configuration.
    """

    def __getitem__(self, key):
        return self.items[key]

    def __init__(self, cfgfile, users, versions):
        """
        Parse category configuration file 'cfgfile'.

        Defined versions and users are passed in 'versions' and 'users'
        parameters respectively. They are used only for consistency checks and
        are not stored anywhere inside the object.
        """

        f = open(cfgfile)
        sections = f.read().split("\n\n")
        f.close()

        category = {}
        for s in sections:
            if not len(s):
                continue

            c = email.message_from_string(s)

            if "Category" not in c:
                raise CorruptedDB_UnparseableCategoriesError(
                    "Invalid category definition (no 'Category' field)"
                    )

            if not len(c["Category"]):
                raise CorruptedDB_UnparseableCategoriesError(
                    "Invalid category definition (blank 'Category' field)"
                    )

            if " " in c["Category"]:
                raise CorruptedDB_UnparseableCategoriesError(
                    "Invalid category '" + c["Category"] + "' definition: "
                    "blank characters in name"
                    )


            if c["Category"] in category:
                raise CorruptedDB_UnparseableCategoriesError("Duplicate "
                    "category '" + c["Category"] + "' definition")

            if "Version-set" not in c:
                raise CorruptedDB_UnparseableCategoriesError(
                    "Invalid category '" + c["Category"] + "' definition: "
                    "no version set defined"
                    )

            if c["Version-set"] not in versions:
                raise CorruptedDB_UnparseableCategoriesError(
                    "Invalid category '" + c["Category"] + "' definition: "
                    "unknown version set"
                    )

            if "Default-owner" not in c:
                raise CorruptedDB_UnparseableCategoriesError(
                    "Invalid category '" + c["Category"] + "' definition: "
                    "no default owner defined"
                    )

            if c["Default-owner"] not in users:
                raise CorruptedDB_UnparseableCategoriesError(
                    "Invalid category '" + c["Category"] + "' definition: "
                    "unknown user assigned as default owner"
                    )

            if c.has_key("Status"):
                status = c["Status"].strip()
                if status == "enabled":
                    enabled = True
                elif status == "disabled":
                    enabled = False
                else:
                    raise CorruptedDB_UnparseableCategoriesError(
                        "Invalid category '" + c["Category"] + "' definition: "
                        "invalid status"
                        )
            else:
                enabled = True

            category[c["Category"]] = Category(
                c["Version-set"],
                versions[c["Version-set"]],
                c["Default-owner"],
                enabled
                )

        self.items = category

    def __iter__(self):
        return self.items.__iter__()

    def __len__(self):
        return len(self.items)

    def keys(self):
        return self.items.keys()

class UserCfg:
    """
    A representation of user accounts configuration.
    """

    def __contains__(self, key):
        return self.items.has_key(key)

    def __init__(self, fname):
        """
        Parses user accounts configuration file 'fname'.
        """

        users = {}
        f = open(fname)
        while 1:
            str = f.readline()
            if not str: break

            user = str.strip()

            if user in users:
                raise CorruptedDB_DuplicateUserError("Duplicate user entry '"
                    + user + "' in" " '" + fname + "'")

            users[user] = user

        f.close()

        self.items = users

    def __len__(self):
        return len(self.items)

    def keys(self):
        return self.items.keys()

    has_key = __contains__


class FilterCfg:
    """
    A representation of filters configuration.
    """

    def __contains__(self, key):
        return self.items.has_key(key)

    def __getitem__(self, key):
        return self.items[key]

    def __init__(self, path):
        f = open(path)
        filters = email.message_from_file(f)
        f.close()

        if len(filters.get_payload()):
            raise CorruptedDB_UnparseableFiltersError("Empty line in filter "
                "configuration file")

        self.items = {}
        for s in filters.keys():

            if s in self.items:
                raise CorruptedDB_UnparseableFiltersError("Duplicate filter "
                    + s + "' definition")

            filter = filters[s].strip()

            if not filter:
                raise CorruptedDB_UnparseableFiltersError("Empty filter "
                    + s + "' definition")
            try:
                self.items[s] = Filter(filter)
            except (FilterIsPredefinedError, FilterExpressionError):
                raise CorruptedDB_UnparseableFiltersError("Syntax error in filter "
                    + s + "' definition")

    def __len__(self):
        return len(self.items)

    has_key = __contains__   

class Filter:
    def __init__(self, str):
        "Initialize filter by parsing the expression passed"

        # Is it a predefined filter name?
        inplace_re = re.compile("[,=]")

        if not inplace_re.search(str):
            # This looks like a predefined filter name
            raise FilterIsPredefinedError(str)

        # Split into subclauses.
        clauses = filter(lambda x: len(x), str.split(","))

        self.conditions = []
        condition_re = re.compile("^(?P<param>[^!=]*)(?P<condition>!?=)(?P<value>.*)$")

        for c in clauses:
            m = condition_re.match(c)
            if not m:
                raise FilterExpressionError(c)

            condition = m.groupdict()

            if len(condition) != 3:
                raise FilterExpressionError(c)
            
            # Perform substitutions.
            condition['value'] = self.substitute(condition['value'])

            self.conditions.append(condition)

    def substitute(self, str):
        
        out = ""
        idx = 0
        l = len(str)
        while idx < l:
            dollar = str.find("$", idx)
            if dollar < 0:
                out = out + str[idx:]
                break
            elif dollar == (l - 1):
                break
            elif str[dollar + 1] == "$":
                out = out + "$"
                idx = dollar + 2
            else:
                end = dollar + 1
                while (end < l) and (str[end].isalnum() or str[end] == "_"):
                    end = end + 1

                if end != (dollar + 1):
                    var = str[dollar + 1:end]
                    if os.environ.has_key(var):
                        out = out + os.environ[var]

                idx = end

        return out


    def matches(self, issue):
        for c in self.conditions:
            assert len(c) == 3

            if c['param'] not in issue.info:
                return False

            if c['condition'] == "=":
                if issue.info[c['param']] != c['value']:
                    return False
            elif c['condition'] == "!=":
                if issue.info[c['param']] == c['value']: 
                    return False
            else:
                raise NotImplemented, c['condition']

        return True

class Configuration:
    """
    Database configuration object (everything under /etc in a database).

    Exported:

    category
        Categories configuration object, CategoryCfg.

    path
        A mapping of strings identifying various objects to their respective
        locations. The keys currently supported are:

        /
            Root of the issue database.

        categories
            Location of categories configuration file.

        data
            Root of the issue data.

        filters
            Location of predefined filters configuration file.

        users
            Location of user accounts configuration file.

        version
            Location of versions configuration file.

    users
        User accounts configuration object, UserCfg.

    versions
        Version sets configuration object, VersionCfg.
    """

    def __init__(self, path):

        # Prepare pathnames mapping.
        self.path = {}
        self.path["/"] = path
        self.path["categories"] = os.path.join(path, "etc", "categories")
        self.path["data"] = os.path.join(path, "data")
        self.path["filters"] = os.path.join(path, "etc", "filters")
        self.path["users"] = os.path.join(path, "etc", "users")
        self.path["versions"] = os.path.join(path, "etc", "versions")

        self.users = UserCfg(self.path["users"])

        self.versions = VersionCfg(self.path["versions"])

        self.category = CategoryCfg(self.path["categories"], self.users,
            self.versions)

        try:
            self.filters = FilterCfg(self.path["filters"])
        except DITrack.DB.CorruptedDB_UnparseableFiltersError:
            raise DITrack.DB.CorruptedDB_UnparseableFiltersError()
        
class Issue:
    """
    A representation of an issue record.
    """
    
    def __contains__(self, key):
        """
        Checks whether a comment indexed by the string KEY is present in the
        issue.
        """

        present = (key in self.local_names) or (key in self.firm_names)

        if present:
            assert(key in self.comment)

        return present

    def __getitem__(self, key):
        """
        Returns comment object indexed by KEY (which should be a string).
        """

        # XXX: Make sure the key is a string.

        if key not in self:
            raise KeyError

        return self.comment[key]

    def __init__(self):
        self.comment = {}
        self.firm_names = []
        self.local_names = []

    def __len__(self):
        """
        Returns total number of attached comments.
        """

        return len(self.firm_names) + len(self.local_names)

    def add_comment(self, comment, is_local, update_info=True):
        """
        Adds comment COMMENT to the list of comments and names it according to
        IS_LOCAL. If UPDATE_INFO is true, updates the issue info afterwards. 
        Returns simple comment id as a string.
        """

        DEBUG("Adding comment: is_local=%s, update_info=%s (called from %s)" % 
            (is_local, update_info, get_caller()))

        # XXX
        assert(is_local)

        name = next_entity_name(self.local_names)
        assert(name not in self.comment)

        self.comment[name] = comment
        self.local_names.append(name)

        if update_info:
            self.update_info()

        return name

    def comments(self, firm=True, local=True):
        """
        Returns a list of tuples (ID, COMMENT) for all existing comments. IDs
        are strings. First all firm comments go in order, then the local ones.

        FIRM and LOCAL prescribe which comments to include into the result.
        """

        assert firm or local

        if firm:
            # We assume the list is sorted.
            names = self.firm_names
        else:
            names = []

        result = [(id, self.comment[id]) for id in names]

        if local:
            names = self.local_names
            names.sort()

            result.extend([(id, self.comment[id]) for id in names])

        return result
        
    def create(cls, title, opened_by, opened_on, owned_by, category,
        version_reported, version_due, description):
        """
        Create new issue instance from scratch.
        """

        issue = cls()

        # Create an initial comment.
        comment = Comment.create(
            text=description,
            added_by=opened_by,
            added_on=opened_on,
            delta=[
                ("Opened-by", None, opened_by),
                ("Opened-on", None, opened_on),
                ("Owned-by", None, owned_by),
                ("Title", None, title),
                ("Category", None, category),
                ("Status", None, "open"),
                ("Reported-for", None, version_reported),
                ("Due-in", None, version_due)
            ])

        # Now append the comment to the issue.
        issue.add_comment(comment, is_local=True)

        return issue

    create = classmethod(create)

    def diff(self, other):
        """
        Returns a difference in headers between SELF and the OTHER issue, by
        checking the headers. The changed headers are returned as a list of
        tuples (HEADER, OLD, NEW), where the HEADER is the header name and the
        OLD and NEW are the old and new values respectively (any of these may
        be None, but not both).
        """

        delta = []

        for h in self.info.keys():

            if h in other.info:
                if self.info[h] != other.info[h]:
                    delta.append((h, self.info[h], other.info[h]))   
            else:
                    delta.append((h, self.info[h], None))

        for h in other.info.keys():

            if h not in self.info:
                delta.append((h, None, other.info[h]))

        return delta


    def load(cls, path):
        """
        Load an issue from path PATH (should point to a directory).
        """

        DEBUG("Loading an issue from '%s' (called from %s)" % 
            (path, get_caller()))

        issue = cls()

        comments = {}
        keys = []

        for fn in os.listdir(path):
            m = comment_fname_re.match(fn)
            if m:
                n = m.group(1)
                fname = os.path.join(path, "comment" + n)
                comments[int(n)] = Comment.load(fname)
                keys.append(int(n))

        keys.sort()

        issue.firm_names = []

        # k is numeric here.
        for k in keys:
            key = "%d" % k
            issue.comment[key] = comments[k]
            issue.firm_names.append(key)

        issue.update_info()

        return issue

    load = classmethod(load)

    def merge_local_comments(self, local):
        """
        Merges all local comments from the LOCAL issue to SELF, retaining
        the names as is. Names collision will raise an assertion fault.

        NB! Does NOT update the info, use the update_info() method for that.
        """

        for name in local.local_names:

            DEBUG("Merging local comment '%s'" % name)
            assert name not in self.local_names, name
 
            self.comment[name] = local.comment[name]

        self.local_names.extend(local.local_names)

        self.local_names.sort()

    def remove_comment(self, name):
        """
        Removes a local comment named NAME from the issue. The comment should 
        exist, otherwise KeyError is raised. The info is *not* updated.
        """

        if name not in self.local_names:
            raise KeyError, name

        assert name in self.comment, "name='%s'" % name

        del self.comment[name]

        self.local_names.remove(name)

    def replace_header(self, name, value):
        """
        Replaces the issue header NAME with VALUE. The header should exist.
        """
        del self.info[name]
        self.info[name] = value

    def update_info(self):
        """
        Goes through all the comments and updates info accordingly (applies
        deltas in sequence). May be used repeatedly.
        """

        self.info = {}

        names = self.local_names
        names.sort()

        names = self.firm_names + names

        DEBUG("Updating the info (called from %s)" % get_caller())

        for name in names:
            DEBUG("Processing comment '%s'" % name)
            self.comment[name].apply_delta(self.info)

        DEBUG("Updated info: %s" % self.info)

    def write_info(self):
        """
        Prints out current issue headers.
        """
    
        keys = self.info.keys()
        keys.sort()

        for k in keys:
            sys.stdout.write("%s: %s\n" % (k, self.info[k]))

class Comment:
    """
    A representation of a comment record.
    """

    def apply_delta(self, info):
        """
        Applies own delta to the info dictionary INFO.

        XXX: should deal with conflicts
        """

        # XXX: should check for duplicate headers in delta

        DEBUG("Applying the delta")

        for header, old, new in self.delta:

            DEBUG("Delta: header='%s' old='%s' new='%s'" % (header, old, new))

            # XXX
            if old:

                assert header in info, "header = '%s'" % header

                assert(info[header] == old)

                if new:
                    info[header] = new
                else:
                    del info[header]

            else:
                assert(new is not None)
                assert(header not in info)

                info[header] = new

    def create(cls, text, added_on, added_by, delta, logmsg=""):

        comment = cls()

        comment.text = text
        comment.added_on = added_on
        comment.added_by = added_by
        comment.delta = delta
        comment.logmsg = logmsg

        return comment

    create = classmethod(create)

    def load(cls, path):
        """
        Load a comment from path PATH.
        """

        f = open(path)
        data = email.message_from_file(f)
        f.close()

        # XXX: we should also handle user headers here (somehow).

        # To ignore case difference, put all headers in our own map.
        header = {}
        orig = {}
        for h in data.keys():
            header[h.lower()] = data[h]
            orig[h.lower()] = h

        delta = []

        for h in header.keys():
            h = h.lower()

            if h.startswith("dt-old-"):

                name = h[7:]
                nh = "dt-new-" + name
                assert(nh in header)

                if not (data[h] or data[nh]):
                    continue

                delta.append((orig[h][7:], data[h], data[nh]))

        return Comment.create(
            text=data.get_payload(),
            added_on=data["Added-on"],
            added_by=data["Added-by"],
            delta=delta
            )

    load = classmethod(load)

    def write(self, dest=sys.stdout, headers_only=False, display_headers=True):
        """
        Write out the comment into file object DEST.
        """
        
        if display_headers:
            output = []

            output.append("Added-on: %s\n" % self.added_on)
            output.append("Added-by: %s\n" % self.added_by)

            for header, old, new in self.delta:

                assert((old is not None) or (new is not None))

                old = old or ""
                new = new or ""

                output.append("DT-Old-%s: %s\n" % (header, old))
                output.append("DT-New-%s: %s\n" % (header, new))

            output.sort()

            dest.writelines(output)

        if not headers_only:

            if display_headers:
                dest.write("\n")

            dest.write("%s" % self.text)

# XXX: factor out common code from Issue and Comment.

class LocalModsArea:
    """
    Class representing local modifications area.

    MEMBERS:
        data    A mapping of existing LMA issues: maps identifiers (as strings)
                to issue objects.
    """

    def __contains__(self, key):
        """
        Check if the issue KEY (string) is contained in the LMA.
        """
        return key in self.data;

    def __del__(self):
        self.data.close()

    def __getitem__(self, key):
        return self.data[key]

    def __init__(self, path):
        
        datadir = os.path.join(path, LMA_DIR)

        if not os.path.exists(datadir):
            os.mkdir(datadir, 0755)

        fname = os.path.join(datadir, LMA_FILE)

        self.data = shelve.open(fname)

    def _comment_keys(self, issue_id):
        """
        Return a sorted list of comment names for all comments of the
        issue ISSUE_ID in the LMA.
        """

        prefix = "%s." % issue_id

        ids = filter(lambda x: x.startswith(prefix), self.data.keys())

        ids.sort()

        return ids

    def comments(self, issue_id):
        """
        Return a sorted list of tuples (ID, COMMENT) for all comments of the
        issue ISSUE_ID in the LMA.
        """

        return [(x, self.data[x]) for x in self._comment_keys(issue_id)]

    def issues(self, firm=True, local=True):
        """
        Return a sorted list of tuples (ID, ISSUE) for all issues in the 
        LMA. The FIRM and LOCAL parameters control which kind of issues to
        include into the resulting list. Firm issues precede local ones in the
        result.

        Either FIRM or LOCAL should be True (or both).
        """
        assert firm or local, "firm=%s, local=%s" % (firm, local)

        # XXX: replace with is_valid_XXX() once those are available for the
        # whole module.
        def is_firm(s):
            try:
                int(s)
            except ValueError:
                return False

            return True

        keys = filter(
            lambda x: (firm and is_firm(x)) or (local and not is_firm(x)),
            self.data.keys()
        )

        keys.sort()

        return [(k, self.data[k]) for k in keys]

    def new_comment(self, issue_id, comment):
        """
        Add issue ISSUE_ID (string) COMMENT to the LMA. Returns newly assigned 
        comment name.
        """

        DEBUG("Creating a new comment in the LMA for issue '%s'" % issue_id)

        try:
            issue = self.data[issue_id]
        except KeyError:
            # The very first local comment to the issue.
            issue = Issue()

        # Don't update the info, since this issue is merely a collection of
        # comments.
        #
        # XXX: make it a list then?
        name = issue.add_comment(comment, is_local=True, update_info=False)

        self.data[issue_id] = issue

        return name

    def new_issue(self, issue):

        # Figure out a name for the new issue
        name = next_entity_name([k for k, i in self.issues(firm=False)])

        self.data[name] = issue

        return name

    def remove_comment(self, issue_number, comment_name):
        """
        Removes comment COMMENT_NAME from the issue ISSUE_NUMBER in the LMA.
        The issue and comment should exist. If the issue has no comments after
        the removal, removes the issue altogether.
        """

        assert issue_number in self.data, "issue_number='%s'" % issue_number

        issue = self.data[issue_number]

        issue.remove_comment(comment_name)

        if len(issue):
            self.data[issue_number] = issue
        else:
            del self.data[issue_number]

    def remove_issue(self, name):
        del self.data[name]

class WC_Operation:
    """
    Class representing a single operation in a working copy.
    """

    def __init__(self):
        raise NotImplementedError

    def fill_txn(self, txn):
        """
        Include the change into given Subversion transaction TXN.
        """
        raise NotImplementedError, self

class WC_Op_Modification(WC_Operation):
    """
    Modification of an existing file.
    """

    def __init__(self, fname):
        self.fname = fname

    def fill_txn(self, txn):

        # Just record the file name in the transaction.
        txn.include(self.fname)

class WC_Op_NewDirectory(WC_Operation):
    """
    New directory creation.
    """

    def __init__(self, fname):
        self.fname = fname

    def fill_txn(self, txn):
        txn.add(self.fname)

class WC_Op_NewFile(WC_Operation):
    """
    New file addition.
    """

    def __init__(self, fname):
        self.fname = fname

    def fill_txn(self, txn):
        txn.add(self.fname)

class WorkingCopy:
    """
    Class encapsulating working copy operations.
    """

    def __getitem__(self, key):
        """
        Read an issue named by KEY from the working copy.
        """
    
        # We deal with issues only (not comments).
        assert("." not in key)

        issue_dir = self._issue_path(key)

        return Issue.load(issue_dir)


    def __init__(self, path, svn_path):
        """
        PATH is a path to an issue database. SVN_PATH is a path to Subversion
        command-line client.
        """

        self.path = path

        self.data_path = os.path.join(path, "data")
        self.next_num_path = os.path.join(path, "meta", "next-id")
        self.svn = DITrack.SVN.Client(svn_path)

    def _get_next_comment_number(self, path):
        """
        Look up the issue directory PATH (it has to exist) and return next
        available comment number.
        """

        assert os.path.exists(path), "path='%s'" % path

        numbers = []

        for fn in os.listdir(path):
            m = comment_fname_re.match(fn)
            if m:
                numbers.append(int(m.group(1)))
                
        numbers.sort()

        if numbers:
            return numbers[-1] + 1
        else:
            return 0

    def _get_next_issue_number(self):
        """
        Look up the next available issue number and return it.
        """

        if not os.path.exists(self.next_num_path):
            raise CorruptedDBError(self.next_num_path + " doesn't exist")

        f = open(self.next_num_path)
        s = f.readline()
        if not s:
            f.close()
            raise CorruptedDBError(self.next_num_path + " is empty")

        try:
            num = int(s.strip())
        except ValueError:
            raise CorruptedDBError("Contents of " + self.next_num_path + 
                " are invalid")

        f.close()

        return num

    def _issue_path(self, id):
        """
        Returns directory path of issue ID. Raises KeyError if no such issue
        exists.
        """

        issue_dir = os.path.join(self.data_path, "i%s" % id)

        if not os.path.exists(issue_dir):
            raise KeyError

        return issue_dir

    def _set_next_issue_number(self, num):
        """
        Update the next available issue number. Returns WC_Operation obejct.
        """

        f = open(self.next_num_path, "w")
        f.write("%s\n" % num)
        f.close()

        return WC_Op_Modification(self.next_num_path)

    def comments(self, issue_id):
        """
        Return a sorted list of tuples (ID, COMMENT) for all comments of the
        issue ISSUE_ID in the working copy.
        """
        
        # XXX: do we need this?
        issue_path = self._issue_path(issue_id)

        comments = {}

        for fn in os.listdir(issue_path):
            m = comment_fname_re.match(fn)
            if m:
                n = m.group(1)
                fname = os.path.join(issue_path, "comment" + n)
                comments[int(n)] = Comment.load(fname)

        keys = comments.keys()
        keys.sort()

        return [(x, comments[x]) for x in keys]

    def commit(self, changes, logmsg):
        """
        Attempts to commit the CHANGES list of operations with log message 
        LOGMSG.
        """

        txn = self.svn.start_txn()

        for op in changes:
            op.fill_txn(txn)

        # XXX: if failed to commit, revert local changes.
        txn.commit(logmsg)

    def issues(self):
        """
        Return a sorted list of tuples (ID, ISSUE) for all issues in the 
        working copy.
        """

        issue_re = re.compile("^i(\\d+)$")

        lst = []
        for fn in os.listdir(self.data_path):
            m = issue_re.match(fn)
            if m:
                lst.append(int(m.group(1)))

        lst.sort()

        res = []
        for id in lst:
            path = os.path.join(self.data_path, "i%s" % id)
            issue = Issue.load(path)
            res.append((id, issue))

        return res

    def new_comment(self, issue_num, comment):
        """
        Writes a comment COMMENT for the issue ISSUE_NUM to the working copy 
        and returns a tuple (NUMBER, OPS) where the NUMBER is a newly assigned 
        comment number and the OPS is a list of corresponding operations.
        """

        issue_dir = os.path.join(self.data_path, "i%s" % issue_num)

        # Fetch next comment number
        num = self._get_next_comment_number(issue_dir)

        fname = os.path.join(issue_dir, "comment%d" % num)

        # Write out the comment.
        f = open(fname, 'w')
        comment.write(dest=f)
        f.close()

        # Remember that
        ops = [WC_Op_NewFile(fname)]

        return num, ops

    def new_issue(self, issue):
        """
        Writes issue ISSUE data to the working copy and returns a tuple 
        (NUMBER, OPS) where the NUMBER is a newly assigned issue number and 
        the OPS is a list of corresponding operations.
        """

        # Fetch next issue number
        num = self._get_next_issue_number()

        issue_dir = os.path.join(self.data_path, "i%s" % num)

        ops = []

        # Update next issue numer
        ops.append(self._set_next_issue_number(num + 1))

        # Create the issue directory
        os.mkdir(issue_dir, 0755)

        # Remember that we've created the directory
        ops.append(WC_Op_NewDirectory(issue_dir))

        # XXX: for now we assume that the very first comment in this issue is
        # named 'A'. This may not be actually the case. Also, we write out one
        # comment only (as 'comment0').

        assert("A" in issue)

        fname = os.path.join(issue_dir, "comment0")

        # Write out the info
        f = open(fname, 'w')

        issue["A"].write(dest=f)

        f.close()

        # Remember that
        ops.append(WC_Op_NewFile(fname))

        return num, ops

class Database:
    """
    Class representing complete database object.
    """

    def __init__(self, globals, path):

        # Check that it's a directory.
        if not os.path.isdir(path):
            raise NotDirectoryError
        
        # Get database version
        v = DITrack.SVN.propget(VERSION_PROP, path)
        
        if not v or not len(v):
            raise NotDatabaseError
        
        # Check if this is supported version.
        if v != FORMAT_VERSION:
            raise InvalidVersionError
        
        # Read up the configuration.
        self.cfg = Configuration(path)

        # Set up local modifications area.
        self.lma = LocalModsArea(path)

        # Working copy interface.
        self.wc = WorkingCopy(path, globals.svn_path)

    def __getitem__(self, key):
        """
        Return an issue by KEY (string id). Raises KeyError is the issue is
        neither in the WC, nor in the LMA.
        """

        # XXX: make sure the key is a string.

        try:
            issue = self.wc[key]
        except KeyError:
            issue = None

        local = None

        try:
            local = self.lma[key]
        except KeyError:
            if not issue:
                raise

        assert(issue or local)

        if issue:
            if local:
                issue.merge_local_comments(local)
        else:
            issue = local

        return issue

    def comments(self, issue_id, from_wc=True, from_lma=True):
        """
        Return a list of tuples (ID, COMMENT) for issue ISSUE_ID comments 
        present in the database.

        Comments from working copy and LMA are included as prescribed by 
        FROM_WC and FROM_LMA parameters respectively. The list returned is 
        sorted. All WC comments precede LMA comments.
        """

        res = []
        if self.is_valid_issue_number(issue_id) and from_wc:
            res = self.wc.comments(issue_id)

        if from_lma:
            res.extend(self.lma.comments(issue_id))

        return res

    def commit_comment(self, issue_number, comment_name):
        """
        Commits comment COMMENT_NAME of issue ISSUE_NUMBER from the LMA. An
        assertion will fail if the ISSUE_NUMBER is not a firm one. Returns
        simple newly assigned comment number as a string.
        """

        DEBUG("Committing comment '%s' of issue '%s' (called from %s)" % 
            (comment_name, issue_number, get_caller()))

        for x in issue_number:
            assert x in string.digits

        lma_issue = self.lma[issue_number]
        comment = lma_issue[comment_name]

        number, changes = self.wc.new_comment(issue_number, comment)

        # XXX: if the commit has failed, we need to revert the changes back.
        self.wc.commit(changes, "i#%s: %s" % (issue_number, comment.logmsg))

        # Now, when the changes are committed, remove the comment from the LMA.
        self.lma.remove_comment(issue_number, comment_name)

        return number

    def commit_issue(self, name):
        """
        Commits issue NAME from LMA. Returns newly assigned issue number.
        """

        DEBUG("Committing issue '%s' (called from %s)" % (name, get_caller()))

        for x in name:
            assert(x in string.uppercase)

        issue = self.lma[name]
        number, changes = self.wc.new_issue(issue)

        # XXX: if the commit has failed, we need to revert the changes back.
        self.wc.commit(changes, "i#%d added: %s" % 
            (number, issue.info["Title"]))

        # Now, when the changes are committed, remove the issue from the LMA.
        self.lma.remove_issue(name)

        return number

    def issue_by_id(self, id, err=True):
        """
        Fetches an issue by ID from the database, checking if 1) the identifier
        is valid; 2) the issue actually exists. If either of these checks
        fails, raises IssueIdSyntaxError or KeyError respectively. If ERR is
        true, prints out diagnostics before raising the exception.
        """

        if not self.is_valid_issue_id(id):
            if err:
                DITrack.Util.common.err("Invalid identifier: '%s'" % id)
            raise IssueIdSyntaxError, id

        try:
            issue = self[id]
        except KeyError:
            if err:
                DITrack.Util.common.err("No such entity: '%s'" % id)
            raise KeyError, id

        return issue

    def issues(self, from_wc=True, from_lma=True):
        """
        Return a list of tuples (ID, ISSUE) for issues present in the database.

        Issues from working copy and LMA are included as prescribed by FROM_WC
        and FROM_LMA parameters respectively. The list returned is sorted. All
        WC issues precede LMA issues.
        """
        
        DEBUG("from_wc=%s, from_lma=%s" % (from_wc, from_lma))

        res = {}
        if from_wc:
            for id, issue in self.wc.issues():
                res[id] = issue

        DEBUG("Firm issues: %s" % pprint.pformat(res))
        if from_lma:
            for id, issue in self.lma.issues():
                DEBUG("LMA issue entity: %s" % pprint.pformat(id))
                if self.is_valid_issue_name(id):
                    # It's a local issue, just store it.
                    res[id] = issue
                else:

                    # We are asked to return local issues only, don't bother
                    # merging comments to firm ones.
                    if not from_wc:
                        continue

                    # It's a local comment to a firm issue. We need to merge
                    # issue headers.
                    id = int(id)
                    assert id in res, pprint.pformat(id)

                    DEBUG("Merging local comments for issue '%s'" % id)
                    res[id].merge_local_comments(issue)
                    res[id].update_info()

        keys = res.keys()
        keys.sort()
        return [(k, res[k]) for k in keys]

    def is_valid_issue_number(self, id):

        issue_number_re = re.compile("^\\d+$")
        if issue_number_re.match(id):
            return int(id) != 0
        else:
            return False

    def is_valid_issue_name(self, id):
        """
        XXX: rename into valid_simple_name()
        XXX: move out of the class

        Checks if the ID passed is a syntactically valid simple entity name.

        'Simple' means 'not compound', e.g. "A" is simple, "A.B" is not.
        """

        issue_name_re = re.compile("^[A-Z]+$")
        return issue_name_re.match(id)

    def is_valid_issue_id(self, id):
        """
        Checks if the passed identifier ID is a syntactically correct 
        identifier (i.e. a valid number or name).
        """
        
        return self.is_valid_issue_number(id) or self.is_valid_issue_name(id)

    def lma_issues(self, firm=True, local=True):
        """
        Returns a list of tuples (ID, ISSUE) for all issue entities present
        in the LMA. The FIRM and LOCAL parameters control which kind of issues
        to include into the resulting list (either FIRM or LOCAL should be
        True; or both). The returned list is sorted, firm issues always precede
        local ones.
        """

        assert (firm or local), "firm=%s, local=%s" % (firm, local)

        return self.lma.issues(firm=firm, local=local)

    def new_comment(self, issue_num, issue_before, issue_after, text, added_by,
        added_on):
        """
        Add a new comment reflecting the change in issue ISSUE_NUM from
        ISSUE_BEFORE to ISSUE_AFTER with specified TEXT to the LMA. Returns 
        tuple (NAME, COMMENT). ADDED_BY and ADDED_ON are the comment's author
        and addition date respectively. 
        
        If there is no difference and the TEXT passed is empty, raises 
        NoDifferenceCondition.
        """
        
        delta = issue_before.diff(issue_after)

        if (not delta) and (not text):
            raise NoDifferenceCondition

        delta_map = {}
        for h, o, n in delta:
            delta_map[h] = n

        logmsg = []

        if "Status" in delta_map:

            if delta_map["Status"] == "closed":

                s = "closed"

                if "Resolution" in delta_map:
                    s += " as %s" % delta_map["Resolution"]

                logmsg.append(s)

            if delta_map["Status"] == "open":
                logmsg.append("reopened")

        if "Owned-by" in delta_map:
            logmsg.append("reassigned to %s" % delta_map["Owned-by"])

        if "Due-in" in delta_map:
            logmsg.append("moved to %s" % delta_map["Due-in"])

        if len(text):
            logmsg.append("comment added")


        comment = Comment.create(text, added_on=added_on, added_by=added_by,
            delta=delta, logmsg=", ".join(logmsg))

        name = self.lma.new_comment(issue_num, comment)

        return name, comment

    def new_issue(self, title, opened_by, opened_on, owned_by, category,
        version_reported, version_due, description):
        """
        Add a new issue to LMA of the database.

        Returns tuple (name, issue), where NAME is newly assigned name and
        ISSUE is newly created issue object.
        """

        if not owned_by:
            owned_by = self.cfg.category[category].default_owner

        issue = Issue.create(
            title=title,
            opened_by=opened_by,
            opened_on=opened_on,
            owned_by=owned_by,
            category=category,
            version_reported=version_reported,
            version_due=version_due,
            description=description
            )

        name = self.lma.new_issue(issue)

        return name, issue

    def remove_comment(self, issue_id, comment_name):
        """
        Removes local comment from the database (e.g. from the LMA). 
        
        ValueError is raised if the ISSUE_ID is not a syntactically valid 
        issue number of if COMMENT_NAME is not a syntactically valid name.

        The COMMENT_NAME parameter should refer to an existing LMA entity
        (e.g. a comment of an existing issue), otherwise KeyError is raised.
        """

        if not self.is_valid_issue_id(issue_id):
            raise ValueError

        if not self.is_valid_issue_name(comment_name):
            raise ValueError

        if not issue_id in self.lma:
            raise KeyError

        # XXX: check that KeyError will be raised if COMMENT_NAME is not 
        # contained in the LMA issue
        self.lma.remove_comment(issue_id, comment_name)
