/*
**  Copyright (c) 2007 Sendmail, Inc. and its suppliers.
**    All rights reserved.
*/

#ifndef _DKIM_UTIL_H_
#define _DKIM_UTIL_H_

#ifndef lint
static char dkim_util_h_id[] = "@(#)$Id: dkim-util.h,v 1.4 2007/07/31 03:23:07 msk Exp $";
#endif /* !lint */

/* system includes */
#include <sys/types.h>
#include <sys/param.h>

/* libsm includes */
#include <sm/gen.h>
#include <sm/types.h>
#include <sm/cdefs.h>

/* libdkim includes */
#include "dkim.h"

/* macros */
#define	DKIM_MALLOC(x,y)	dkim_malloc((x)->dkim_libhandle, \
				            (x)->dkim_closure, y)
#define	DKIM_FREE(x,y)		dkim_mfree((x)->dkim_libhandle, \
				           (x)->dkim_closure, y)

/*
**  DKIM_MALLOC -- allocate memory
**
**  Parameters:
**  	libhandle -- DKIM library context in which this is performed
**  	closure -- opaque closure handle for the allocation
**  	nbytes -- number of bytes desired
**
**  Return value:
**  	Pointer to allocated memory, or NULL on failure.
*/

extern void *dkim_malloc __P((DKIM_LIB *, void *, size_t));

/*
**  DKIM_MFREE -- release memory
**
**  Parameters:
**  	dkim -- DKIM context in which this is performed
**  	closure -- opaque closure handle for the allocation
**  	ptr -- pointer to memory to be freed
**
**  Return value:
**  	None.
*/

extern void dkim_mfree __P((DKIM_LIB *, void *, void *));

/*
**  DKIM_STRDUP -- duplicate a string
**
**  Parameters:
**  	dkim -- DKIM handle
**  	str -- string to clone
**  	len -- bytes to copy (0 == copy to NULL byte)
**
**  Return value:
**  	Pointer to a new copy of "str" allocated from within the appropriate
**  	context, or NULL on failure.
*/

extern unsigned char *dkim_strdup __P((DKIM *, const unsigned char *, size_t));

/*
**  DKIM_TMPFILE -- open a temporary file
**
**  Parameters:
**  	dkim -- DKIM handle
**  	fp -- descriptor (returned)
**  	keep -- if FALSE, unlink() the file once created
**
**  Return value:
**  	A DKIM_STAT_* constant.
*/

extern DKIM_STAT dkim_tmpfile __P((DKIM *, int *, bool));

/*
**  DKIM_DSTRING_NEW -- make a new dstring
**
**  Parameters:
**  	dkim -- DKIM handle
**  	len -- initial number of bytes
**  	maxlen -- maximum allowed length (0 == unbounded)
**
**  Return value:
**  	A DKIM_DSTRING handle, or NULL on failure.
*/

extern struct dkim_dstring *dkim_dstring_new __P((DKIM *, int, int));

/*
**  DKIM_DSTRING_FREE -- destroy an existing dstring
**
**  Parameters:
**  	dstr -- DKIM_DSTRING handle to be destroyed
**
**  Return value:
**  	None.
*/

extern void dkim_dstring_free __P((struct dkim_dstring *));

/*
**  DKIM_DSTRING_COPY -- copy data into a dstring
**
**  Parameters:
**  	dstr -- DKIM_DSTRING handle to update
**  	str -- input string
**
**  Return value:
**  	TRUE iff the copy succeeded.
**
**  Side effects:
**  	The dstring may be resized.
*/

extern bool dkim_dstring_copy __P((struct dkim_dstring *, char *));

/*
**  DKIM_DSTRING_CAT -- append data onto a dstring
**
**  Parameters:
**  	dstr -- DKIM_DSTRING handle to update
**  	str -- input string
**
**  Return value:
**  	TRUE iff the update succeeded.
**
**  Side effects:
**  	The dstring may be resized.
*/

extern bool dkim_dstring_cat __P((struct dkim_dstring *, char *));

/*
**  DKIM_DSTRING_CAT1 -- append one byte onto a dstring
**
**  Parameters:
**  	dstr -- DKIM_DSTRING handle to update
**  	c -- input character
**
**  Return value:
**  	TRUE iff the update succeeded.
**
**  Side effects:
**  	The dstring may be resized.
*/

extern bool dkim_dstring_cat1 __P((struct dkim_dstring *, int));

/*
**  DKIM_DSTRING_GET -- retrieve data in a dstring
**
**  Parameters:
**  	dstr -- DKIM_STRING handle whose string should be retrieved
**
**  Return value:
**  	Pointer to the NULL-terminated contents of "dstr".
*/

extern char *dkim_dstring_get __P((struct dkim_dstring *));


/*
**  DKIM_DSTRING_LEN -- retrieve length of data in a dstring
**
**  Parameters:
**  	dstr -- DKIM_STRING handle whose string should be retrieved
**
**  Return value:
**  	Number of bytes in a dstring.
*/

extern int dkim_dstring_len __P((struct dkim_dstring *));

/*
**  DKIM_DSTRING_BLANK -- clear out the contents of a dstring
**
**  Parameters:
**  	dstr -- DKIM_STRING handle whose string should be cleared
**
**  Return value:
**  	None.
*/

extern void dkim_dstring_blank __P((struct dkim_dstring *));

#endif /* _DKIM_UTIL_H_ */
