#ifndef ErrorRegistry_h
#define ErrorRegistry_h

#ifndef StringHashMap_h
#include "StringHashMap.h"
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif

#ifndef std_iostream
#define std_iostream
#include <iostream>
#endif

#ifndef Check_h
#include "Check.h"
#endif

#ifndef Error_h
#include "Error.h"
#endif

#define REGISTER_DESCRIPTION(Err) \
    static doctorj::ErrorRegistrant register_##Err(new Err())

using namespace std;

namespace doctorj 
{
    struct ErrorSettings;

    class NoSuchErrorCodeException
    {
    public:
        NoSuchErrorCodeException(const string& code);

        virtual ~NoSuchErrorCodeException();

        virtual string code() const;

    private:
        string code_;
    };

    class ErrorRegistrant
    {
    public:
        ErrorRegistrant(Check* const check);
    };

    /**
     * Writes long descriptions of various errors, which are identified by code.
     */
    class ErrorRegistry
    {
    public:
        /**
         * Creates the registry.
         */
        ErrorRegistry();
        
        /**
         * Deletes all the stored errors.
         */
        virtual ~ErrorRegistry();

        /**
         * Writes the description to the output stream, for the error with the
         * given code.
         * 
         * @throws NoSuchErrorCodeException Thrown when there is no description for
         *                                  the given code.
         */
        static void describe(ostream& os, const string& code) throw(NoSuchErrorCodeException);

        /**
         * Writes the description for each code to the output stream. The codes
         * are in alphanumerical order.
         */
        static void describeAll(ostream& os);

        /**
         * Adds the given error.
         */
        void add(Check* const check);

        /**
         * Returns the errors registered.
         */
        static vector< Check* > errors();

        /**
         * Sets the suppression state of the given error code.
         */
        static void setSuppressed(const string& code, bool suppress);

        /**
         * Returns whether the given error should be suppressed.
         */
        static bool isSuppressed(const string& code);

        /**
         * Returns the identifier for the given error.
         */
        virtual int identifier(const string& errorTitle) const;
        
        /**
         * Returns the category of these types of errors.
         */
        virtual char category() const = 0;
        
    protected:

        typedef std::map<std::string, ErrorSettings*, std::greater<std::string> > StringToErrorSettings;

        /**
         * Map from the error title to the error.
         */
        static StringToErrorSettings errors_;

        /**
         * Dynamically used to assign identifiers to errors being registered.
         */
        int identifier_;

        typedef std::map<std::string, bool, std::greater<std::string> > StringToBool;
        
        /**
         * Map from the error codes to whether they are suppressed.
         */
        static StringToBool suppressions_;

    };
}


#endif //! ErrorRegistry_h
