#ifndef CommentSpellCheck_h
#include "CommentSpellCheck.h"
#endif

#ifndef StringUtilities_h
#include "StringUtilities.h"
#endif

#ifndef DocUserPreferences_h
#include "DocUserPreferences.h"
#endif

#ifndef Platform_h
#include "Platform.h"
#endif

#ifndef iterext_h
#include "iterext.h"
#endif

#ifndef std_algorithm
#define std_algorithm
#include <algorithm>
#endif

using namespace doctorj;
using namespace std;

CommentSpellCheck* CommentSpellCheck::instance_ = NULL;

CommentSpellCheck* CommentSpellCheck::get()
{
    if (!instance_) {
        instance_ = new CommentSpellCheck();
    }
    return instance_;
}

CommentSpellCheck::CommentSpellCheck() : doCheck_(false)
{
    DocUserPreferences* prefs = DocUserPreferences::get();

    EACHC(vector<string>, prefs->dictionaries, dit) {
        string d = *dit;
        doCheck_ = addDictionary(d) || doCheck_;
    }

    EACHC(vector<string>, prefs->validWords, vwit) {
        string w = *vwit;
        addWord(w);
        // but that's not enough to say that we should do the check.
    }
}

CommentSpellCheck::~CommentSpellCheck()
{
}

void CommentSpellCheck::skipSection(const string& section)
{
    if (consume(string("<")  + section + string(">"))) {
        consumeTo(string("</") + section + string(">"));
    }
}

void CommentSpellCheck::skipLink()
{
    if (consume("{@link")) {
        consumeTo("}");
    }
}

void CommentSpellCheck::skipBlanks()
{
    while (pos_ < len_ && desc_[pos_] != '<' && desc_.substr(pos_, 2) != "{@" && !isalnum(desc_[pos_])) {
        ++pos_;
    }
}

void CommentSpellCheck::skipToWord()
{
    skipSection("code");
    skipSection("pre");
    skipLink();
    consume("&nbsp;");
}

void CommentSpellCheck::check(const string& description)
{
    if (doCheck_) {
        return runCheck(description);
    }
}

void CommentSpellCheck::runCheck(const string& desc)
{
    desc_ = desc;
    len_ = desc_.length();
    pos_ = 0;
    
    while (pos_ < len_) {
        skipToWord();
        if (pos_ < len_) {
            if (isalpha(desc_[pos_])) {
                checkCurrentWord();
            }
            else {
                // not at an alpha character. Might be some screwy formatting or
                // a nonstandard tag.
                skipThroughWord();
            }
        }
    }
}

void CommentSpellCheck::checkWord(const string& word, int position)
{
    multimap<int, string> nearMatches;
    bool valid = checker_.isCorrect(word, &nearMatches);
    if (!valid) {
        wordMisspelled(word, position, nearMatches);
    }
}

void CommentSpellCheck::wordMisspelled(const string& word, int position, const multimap<int, string>& nearMatches)
{
    int nPrinted = 0;
    const int printGoal = 15;
    for (int i = 0; nPrinted < printGoal && i < 4; ++i) { // 4 == max edit distance
        multimap<int, string>::const_iterator it   = nearMatches.lower_bound(i);
        multimap<int, string>::const_iterator stop = nearMatches.upper_bound(i);
        while (it != stop) {
            // This is not debugging output -- this is actually wanted. But I
            // often run "glark '^\s*cout' to find all my couts, so we'll hide
            // this very sneakily:
            /* escond */ cout << "    near match '" << it->second << "': " << it->first << endl;
            ++nPrinted;
            ++it;
        }
    }
}

bool CommentSpellCheck::consume(const string& what)
{
    skipBlanks();
    if (StringUtilities::startsWith(desc_.substr(pos_), what)) {
        pos_ += what.length();
        return true;
    }
    else {
        return false;
    }
}

void CommentSpellCheck::consumeTo(const string& what)
{
    int len = desc_.length();
    while (pos_ < len_ && !StringUtilities::startsWith(desc_.substr(pos_), what)) {
        ++pos_;
    }
}

void CommentSpellCheck::checkCurrentWord()
{
    string word;
    word += desc_[pos_];
    int startingPosition = pos_;
    bool canCheck = true;

    ++pos_;

    // spell check words that do not have:
    //     - mixed case (varName)
    //     - embedded punctuation ("wouldn't", "pkg.foo")
    //     - numbers (M16, BR549)
    while (pos_ < len_) {
        char ch = desc_[pos_];
        // cout << "testing " << ch << endl;
        if (isspace(ch)) {
            break;
        }
        else if (islower(ch)) {
            word += ch;
            ++pos_;
        }
        else if (isupper(ch)) {
            // cout << "got uppercase in word" << endl;
            skipThroughWord();
            return;
        }
        else if (isdigit(ch)) {
            skipThroughWord();
            return;
        }
        else if (ispunct(ch)) {
            // we can check it if there's nothing but punctuation up to the next
            // space or end of description.
            if (pos_ + 1 == len_) {
                // that's OK to check
                break;
            }
            else {
                ++pos_;
                while (pos_ < len_ && ispunct(desc_[pos_])) {
                    // skipping through punctuation
                    ++pos_;
                }
                if (pos_ == len_ || isspace(desc_[pos_])) {
                    // punctuation ended the word, so we can check this
                    break;
                }
                else {
                    // punctuation did NOT end the word, so we can NOT check this
                    skipThroughWord();
                    return;
                }
            }
        }
        else {
            // cout << "CommentSpellCheck: ERROR: character not handled: " << ch << " (" << (int)ch << ")" << endl;
            ++pos_;
            canCheck = false;
        }
    }
    
    // has to be more than one character:
    if (canCheck && pos_ - startingPosition > 1) {
        checkWord(word, startingPosition);
    }
}

void CommentSpellCheck::skipThroughWord()
{
    ++pos_;
    while (pos_ < len_ && !isspace(desc_[pos_])) {
        ++pos_;
    }
}
        
bool CommentSpellCheck::addDictionary(const string& dictName) 
{
    if (Platform::isReadableFile(dictName)) {
        checker_.addDictionary(dictName);
        return true;
    }
    else {
        cerr << "WARNING: dictionary " << dictName << " could not be read." << endl;
        return false;
    }
}
void CommentSpellCheck::addWord(const string& word) 
{
    checker_.addWord(word);
}
