#ifndef FieldDocumentationAnalyzer_h
#include "FieldDocumentationAnalyzer.h"
#endif

#ifndef DocumentationErrors_h
#include "DocumentationErrors.h"
#endif

#ifndef AST_h
#include "AST.h"
#endif

#ifndef JavadocTags_h
#include "JavadocTags.h"
#endif

using namespace doctorj;

FieldDocumentationAnalyzer::FieldDocumentationAnalyzer(Reporter* const reporter,
                                                       AstFieldDeclaration* const fd,
                                                       AstModifierList* const modifiers) :
        ItemDocumentationAnalyzer(reporter, modifiers), 
        field_(fd)
{
    check();
}

FieldDocumentationAnalyzer::FieldDocumentationAnalyzer()
{
}

FieldDocumentationAnalyzer::~FieldDocumentationAnalyzer()
{
}

bool FieldDocumentationAnalyzer::handleTag(AstTaggedComment* const tc) 
{
    string tag = tc->tag();
    if (tag == JavadocTags::SERIAL) {
        checkSerial(tc);
        return true;
    }
    else if (tag == JavadocTags::SERIALFIELD) {
        checkSerialField(tc);
        return true;
    }
    else {
        return ItemDocumentationAnalyzer::handleTag(tc);
    }
}

string FieldDocumentationAnalyzer::type() const
{
    return "field";
}

AstItem* FieldDocumentationAnalyzer::getSubject()
{
    return field_->getVariableDeclaratorList();
}

AstNoncode* FieldDocumentationAnalyzer::leadingNoncode()
{
    return field_->leadingNoncode();
}

void FieldDocumentationAnalyzer::checkSerial(AstTaggedComment* const tc) 
{
    if (tc->countTargets() <= 0) {
        ErrorSerialWithoutDescription err(reporter(), tc);
        err.process();
    }
}

void FieldDocumentationAnalyzer::checkSerialField(AstTaggedComment* const tc) 
{
//     if (tc->countTargets() <= 0) {
//         ErrorSerialFieldWithoutDataDescription err(reporter(), tc);
//         err.process();
//     }

    int nTargets = tc->countTargets();
    if (nTargets <= 0) {
        ErrorSerialFieldWithoutNameTypeOrDescription err(reporter(), tc);
        err.process();
    }
    else if (nTargets == 1) {
        ErrorSerialFieldWithoutTypeOrDescription err(reporter(), tc);
        err.process();
    }
    else if (nTargets == 2) {
        ErrorSerialFieldWithoutDataDescription err(reporter(), tc);
        err.process();
    }
}

void FieldDocumentationAnalyzer::complainVariable(AstVariableDeclarator* const vd,
                                                  const string& docCmt)
{
    AstVariableDeclaratorId* vdi = vd->getVariableDeclaratorId();
    AstIdentifier*           id  = vdi->getIdentifier();
    
    if (hasPublicAccess()) {
        ErrorFieldUndocumentedUncommented err(reporter(), id, type(), docCmt, true);
        err.process();
    }
    else {
        ErrorFieldUndocumentedUncommented err(reporter(), id, type(), docCmt, false);
        err.process();
    }
}

void FieldDocumentationAnalyzer::complainUndocumented()
{
    AstVariableDeclaratorList* vars = field_->getVariableDeclaratorList();
    int count = vars->getVariableDeclaratorCount();
    for (int i = 0; i < count; ++i) {
        AstVariableDeclarator* var = vars->getVariableDeclarator(i);
        complainVariable(var, "documented");
    }
}

void FieldDocumentationAnalyzer::complainUncommented()
{
    AstVariableDeclaratorList* vars = field_->getVariableDeclaratorList();
    int count = vars->getVariableDeclaratorCount();
    for (int i = 0; i < count; ++i) {
        AstVariableDeclarator* var = vars->getVariableDeclarator(i);
        complainVariable(var, "commented");
    }
}

ErrorInvalidJavadocTag* FieldDocumentationAnalyzer::makeUnknownTagError(AstItem* const item, 
                                                                        const string& tag) const
{
    return new ErrorInvalidJavadocTagField(reporter(), item, tag);
}
