#ifndef ItemDocumentationAnalyzer_h
#include "ItemDocumentationAnalyzer.h"
#endif

#ifndef DocumentationErrors_h
#include "DocumentationErrors.h"
#endif

#ifndef AST_h
#include "AST.h"
#endif

#ifndef JavadocCommentSpellCheck_h
#include "JavadocCommentSpellCheck.h"
#endif

#ifndef JavadocTags_h
#include "JavadocTags.h"
#endif

#ifndef StringUtilities_h
#include "StringUtilities.h"
#endif

#ifndef SpellChecker_h
#include "SpellChecker.h"
#endif

#include <ctype.h>

#ifndef std_vector
#define std_vector
#include <vector>
#endif

#ifndef std_map
#define std_map
#include <map>
#endif

#ifndef iterext_h
#include "iterext.h"
#endif

#ifndef DocDebug_h
#include "DocDebug.h"
#endif

using namespace std;

static const int MIN_SUMMARY_WORDS = 4;

static const int MAX_EDIT_DISTANCE = 4;

using namespace doctorj;

static bool validForSpellCheck(const string& tag)
{
    return tag != JavadocTags::AUTHOR;
}

static bool usesTarget(const string& tag)
{
    // technically, JavadocTags::RETURN does not use the target, but we'll catch
    // that error elsewhere.
    return tag != JavadocTags::DEPRECATED;
}   

ItemDocumentationAnalyzer::ItemDocumentationAnalyzer(Reporter* const reporter, AstModifierList* const modifiers) :
        JavadocAnalyzer(reporter, modifiers)
{
}

ItemDocumentationAnalyzer::ItemDocumentationAnalyzer()
{
}

ItemDocumentationAnalyzer::~ItemDocumentationAnalyzer()
{
}

void ItemDocumentationAnalyzer::check() 
{
    if (AstNoncode* nc = leadingNoncode()) {
        checkComments(nc);
    }
    else {
        complainUncommented();
    }
}

void ItemDocumentationAnalyzer::checkComments(AstNoncode* const nc) 
{
    vector<AstComment*> cmts = nc->comments();
    vector<AstComment*>::reverse_iterator stop = cmts.rend();
    AstJavadocComment* jc = NULL;

    for (vector<AstComment*>::reverse_iterator it = cmts.rbegin(); !jc && it != stop; ++it) {
        AstComment* cmt = *it;
        jc = dynamic_cast<AstJavadocComment*>(cmt);
    }

    if (jc) {
        checkJavadoc(jc);
    }
    else {
        complainUndocumented();
    }
}

void ItemDocumentationAnalyzer::complainUncommented()
{
    if (hasPublicAccess()) {
        ErrorNoCommentForPublic err(reporter(), getSubject(), type());
        err.process();
    }
    else {
        ErrorNoComment err(reporter(), getSubject(), type());
        err.process();
    }
}

void ItemDocumentationAnalyzer::complainUndocumented()
{
    if (hasPublicAccess()) {
        ErrorNoJavadocCommentForPublic err(reporter(), getSubject(), type());
        err.process();
    }
    else {
        ErrorNoJavadocComment err(reporter(), getSubject(), type());
        err.process();
    }
}

void ItemDocumentationAnalyzer::checkJavadoc(AstJavadocComment* const jc) 
{
    checkSummarySentence(jc);
    checkSpelling(jc);

    DEBUG_DOCUMENTATION_ANALYZER(cout << "ItemDocumentationAnalyzer::checkJavadoc: " << jc->text() << endl);

    JavadocTags javadocTags;

    vector<AstTaggedComment*> tcs = jc->taggedComments();
    vector<AstTaggedComment*>::iterator stop = tcs.end();

    // check for misordered tags
    map<AstTaggedComment*, int> indices; // index == their proper order in the Javadoc comment

    vector<AstTaggedComment*> reordered;

    for (vector<AstTaggedComment*>::iterator it = tcs.begin(); it != stop; ++it) {
        AstTaggedComment* tc = *it;
        string tag = tc->tag();
        DEBUG_DOCUMENTATION_ANALYZER(cout << "ItemDocumentationAnalyzer::checkJavadoc: checking tag '" << tag << "'" << endl);
        int idx = javadocTags.indexForTag(tag);
        indices[tc] = idx;
    }
    
    for (vector<AstTaggedComment*>::iterator it = tcs.begin(); it != stop; ++it) {
        AstTaggedComment* tc = *it;
        string tag = tc->tag();

        DEBUG_DOCUMENTATION_ANALYZER(cout << "ItemDocumentationAnalyzer::checkJavadoc: checking tag '" << tag << "'" << endl);

        map<AstTaggedComment*, int>::const_iterator ipos = indices.find(tc);
        int index = ipos == indices.end() ? -1 : ipos->second;

        bool inserted = false;
        
        if (index >= 0 && index != JavadocTags::CUSTOM_TAG) {
            for (vector<AstTaggedComment*>::iterator rit = reordered.begin(), rstop = reordered.end(); rit != rstop; ++rit) {
                AstTaggedComment* rtc = *rit;
                map<AstTaggedComment*, int>::const_iterator ripos = indices.find(rtc);
                int rindex = ripos == indices.end() ? -1 : ripos->second;
                DEBUG_DOCUMENTATION_ANALYZER(cout << "ItemDocumentationAnalyzer::checking reordered tag '" << rtc->tag() << "', index " << rindex << endl);
                if (index < rindex) {
                    reordered.insert(rit, tc);
                    inserted = true;
                    break;
                }
            }
        }

        if (!inserted) {
            reordered.push_back(tc);
        }
    }

    if (reordered != tcs) {
        DEBUG_DOCUMENTATION_ANALYZER(cout << "ItemDocumentationAnalyzer::checkJavadoc: tags appear to be reordered" << endl);
        ErrorMisorderedTags err(reporter(), jc, tcs, reordered);
        err.process();
    }

    // check all tags
    for (vector<AstTaggedComment*>::iterator it = tcs.begin(); it != stop; ++it) {
        AstTaggedComment* tc = *it;
        string tag = tc->tag();

        DEBUG_DOCUMENTATION_ANALYZER(cout << "ItemDocumentationAnalyzer::checkJavadoc: checking tag '" << tag << "'" << endl);

        int idx = javadocTags.indexForTag(tag);
        
        DEBUG_DOCUMENTATION_ANALYZER(cout << "ItemDocumentationAnalyzer::checkJavadoc: about to handle unknown tag '" << tag << "'" << endl);
        if (idx != JavadocTags::CUSTOM_TAG && !handleTag(tc)) {
            checkUnknownTag(tc);
        }

        if (AstTaggedDescribedComment* tdc = dynamic_cast<AstTaggedDescribedComment*>(tc)) {
            if (validForSpellCheck(tdc->tag())) {
                JavadocCommentSpellCheck* jcsc = JavadocCommentSpellCheck::get(reporter());
                // for some comment types, we'll skip checking the target

                char*  start = usesTarget(tdc->tag()) ? tdc->targetEnd() + 1 : tdc->descriptionStart();
                char*  end   = tdc->descriptionEnd();
                string desc  = start && end ? string(start, 1 + end - start) : string("");
                jcsc->checkComment(desc, start, tdc->sourceFile());
            }
        }
    }

    // check for the parameters and exceptions that weren't documented.
    checkUndocumented();
}

void ItemDocumentationAnalyzer::checkUnknownTag(AstTaggedComment* const tc) 
{
    // TODO: add check for locally-defined tags and proposed tags
    JavadocTags javadocTags;

    // it is even valid?
    string tag = tc->tag();
    int idx = javadocTags.indexForTag(tag);

    if (idx >= 0) {
        // it's a valid Javadoc tag, but just not for this type of item.
        Error* error = makeUnknownTagError(tc, tag);
        error->process();
        error->destroy();
    }
    else if (checkTagMisspelling(tc, tag)) {
        // nothing to do here--the checkMisspelling method already complained
        // for us.
    }
    else {
        // no such tag
        ErrorInvalidJavadocTag err(reporter(), tc, tag);
        err.process();
    }
}

void ItemDocumentationAnalyzer::checkSummarySentence(AstJavadocComment* const jc) 
{
    // check the first sentence for length
    string desc = jc->description();

    if (desc.length() == 0) {
        ErrorNoSummarySentence err(reporter(), jc);
        err.process();
    }
    else {
        // we want the first period followed by whitespace
        int len = desc.length();
        int dotpos = desc.find('.');
        while (dotpos != -1 && dotpos + 1 < len && !isspace(desc[dotpos + 1])) {
            dotpos = desc.find('.', dotpos + 1);
        }

        if (dotpos == -1) {
            // this would be from a sentence with no ending period.
            AstStringLeaf dl = jc->descriptionLeaf();
            ErrorNoPeriodEndingSummarySentence err(reporter(), &dl);
            err.process();
        }
        else {
            string firstSentence = desc.substr(0, 1 + dotpos);
            int nWords = 1 + StringUtilities::count(firstSentence, ' ');
            if (nWords < MIN_SUMMARY_WORDS) {
                AstStringLeaf dl = jc->descriptionLeaf();
                ErrorShortSummarySentence err(reporter(), &dl);
                err.process();
            }
        }
    }
}

void ItemDocumentationAnalyzer::checkSpelling(AstJavadocComment* const jc) 
{
    JavadocCommentSpellCheck* jcsc = JavadocCommentSpellCheck::get(reporter());
    jcsc->checkComment(jc->description(), jc->descriptionStart(), jc->sourceFile());
}

bool ItemDocumentationAnalyzer::handleTag(AstTaggedComment* const tc) 
{
    string tag = tc->tag();
    if (tag == JavadocTags::DEPRECATED) {
        checkDeprecated(tc);
        return true;
    }
    else if (tag == JavadocTags::SEE) {
        checkSee(tc);
        return true;
    }
    else if (tag == JavadocTags::SINCE) {
        checkSince(tc);
        return true;
    }
    else {
        return false;
    }
}

void ItemDocumentationAnalyzer::checkDeprecated(AstTaggedComment* const tc) 
{
    if (tc->countTargets() <= 0) {
        ErrorNoDeprecatedText err(reporter(), tc);
        err.process();
    }
}

void ItemDocumentationAnalyzer::checkSee(AstTaggedComment* const tc) 
{
    if (tc->countTargets() <= 0) {
        ErrorNoSeeReference err(reporter(), tc);
        err.process();
    }
}

void ItemDocumentationAnalyzer::checkSince(AstTaggedComment* const tc) 
{
    if (tc->countTargets() <= 0) {
        ErrorNoSinceText err(reporter(), tc);
        err.process();
    }
}

void ItemDocumentationAnalyzer::checkUndocumented()
{
    // nothing.
}

bool ItemDocumentationAnalyzer::checkTagMisspelling(AstTaggedComment* const tc, const string& tag) const
{
    DEBUG_DOCUMENTATION_ANALYZER(cout << "checking spelling of '" << tag << "'" << endl);

    JavadocTags javadocTags;
    vector<string> tags = javadocTags.allTags();
    
    map<string, int> editDistances; // key = tag, value = edit distance
    int shortest = -1;          // shortest edit distance

    const char* const cTag = tag.c_str();
    size_t tagLen = tag.length();
    
    // how far off is each one?
    SpellChecker cchk;
    EACHC(vector<string>, tags, it) {
        string other = *it;
        int edist = cchk.editDistance(cTag, tagLen, other.c_str(), other.length());
        DEBUG_DOCUMENTATION_ANALYZER(cout << "edit distance between '" << tag << "' and '" << other << "' = " << edist << endl);
        // the edit distance is misleading for very short words
        if (edist >= 0 && edist <= MAX_EDIT_DISTANCE && 
            edist < (int)tag.length() && 
            edist < (int)other.length()) {
            editDistances[other] = edist;
            if (shortest == -1 || edist < shortest) {
                shortest = edist;
            }
        }
    }

    if (shortest == -1) {
        return false;
    }
    else {
        vector<string> possibles;
        for (map<string, int>::const_iterator it = editDistances.begin(),
                 stop = editDistances.end(); it != stop; ++it) {
            if (it->second == shortest) {
                string other = it->first;
                possibles.push_back(it->first);
            }
        }

        AstStringLeaf taglf = tc->tagLeaf();
        ErrorMisspelledJavadocTag err(reporter(), &taglf, tag, possibles);
        err.process();
        return true;
    }
}
