#ifndef ParameterDocumentationAnalyzer_h
#include "ParameterDocumentationAnalyzer.h"
#endif

#ifndef Java_h
#include "Java.h"
#endif

#ifndef AST_h
#include "AST.h"
#endif

#ifndef SpellChecker_h
#include "SpellChecker.h"
#endif

#ifndef DocDebug_h
#include "DocDebug.h"
#endif

#ifndef std_algorithm
#define std_algorithm
#include <algorithm>
#endif

using namespace doctorj;

ParameterDocumentationAnalyzer::ParameterDocumentationAnalyzer(Reporter* const reporter, 
                                                               AstModifierList* const modifiers,
                                                               AstFormalParameterList* const args) :
        JavadocAnalyzer(reporter, modifiers), 
        arguments_(args),
        currentParam_(0)
{
}

ParameterDocumentationAnalyzer::~ParameterDocumentationAnalyzer()
{
}

bool ParameterDocumentationAnalyzer::nearMatch(const string& str1, const string& str2) const
{
    SpellChecker cchk;
    return cchk.nearMatch(str1, str2);
}

vector<AstFormalParameter*> ParameterDocumentationAnalyzer::getUndocumentedParams() const
{
    vector<AstFormalParameter*> undocumented;
    if (arguments_) {
        int nArgs = arguments_->getFormalParameterCount();
        for (int ai = 0; ai < nArgs; ++ai) {
            AstFormalParameter* param   = arguments_->getFormalParameter(ai);
            string argName = param->name();
            if (!isDocumented(argName)) {
                undocumented.push_back(param);
            }
        }
    }
    return undocumented;
}

void ParameterDocumentationAnalyzer::paramDocumented(const string& paramName)
{
    paramsDocumented_.push_back(paramName);
}

void ParameterDocumentationAnalyzer::nextUndocumented()
{
    ++currentParam_;
}

int ParameterDocumentationAnalyzer::getParameterStatus(AstTaggedComment* const tc)
{
    if (arguments_) {
        if (AstTaggedDescribedComment* tdc = dynamic_cast<AstTaggedDescribedComment*>(tc)) {
            return getParameterStatus(tdc);
        }
        else {
            return DOC_PARAM_NAME_MISSING;
        }
    }
    else {
        return DOC_PARAMS_NOT_IN_CODE;
    }
}

AstFormalParameter* ParameterDocumentationAnalyzer::findArgumentByName(const string& name) const
{
    return JavaParameterList::getMatching(arguments_, name);
}

bool ParameterDocumentationAnalyzer::isDocumented(const string& name) const
{
    vector<string>::const_iterator it = 
        std::find(paramsDocumented_.begin(), paramsDocumented_.end(), name);
    return it != paramsDocumented_.end();
}

AstFormalParameter* ParameterDocumentationAnalyzer::findArgumentByNameMisspelled(const string& name) const
{
    if (arguments_) {
        int count = arguments_->getFormalParameterCount();
        SpellChecker cchk;
        for (int i = 0; i < count; ++i) {
            AstFormalParameter* arg = arguments_->getFormalParameter(i);
            if (cchk.nearMatch(arg->name(), name)) {
                return arg;
            }
        }
    }
    return NULL;
}
    
int ParameterDocumentationAnalyzer::getParameterStatus(AstTaggedDescribedComment* const tdc)
{
    // check the description for:
    // first char uppercase => last char period
    // first char lowercase => no last period
    // within description, references name (widget), not class (Widget)

    int nTargets = tdc->countTargets();
    if (nTargets < 1) {
        return DOC_PARAM_NAME_MISSING;
    }
    else { 
        int status = 0;
        if (nTargets < 2) {
            status = DOC_PARAM_DESCRIPTION_MISSING;
        }
        else {
            status = DOC_PARAM_HAS_DESCRIPTION;
        }
        
        string docName = tdc->target();
        AstFormalParameter* cp = currentParameter();

        if (cp == NULL) {
            if (AstFormalParameter* arg = findArgumentByName(docName)) {
                if (isDocumented(arg->name())) {
                    status |= DOC_PARAM_REPEATED;
                }
            }
            else {
                status |= DOC_PARAM_NOT_IN_CODE;
            }
        }
        else if (docName == cp->name()) {
            status |= DOC_PARAM_OK;
            paramDocumented(docName);
            nextUndocumented();
        }
        else if (docName == cp->dataType()) {
            status |= DOC_PARAM_USES_DATA_TYPE;
            // we're giving them the benefit of the doubt:
            paramDocumented(cp->name());
            nextUndocumented();
        }
        else if (nearMatch(docName, cp->name())) {
            // this test follows the data type test, since often parameter names are
            // the lowercased version of the class name, for example "void
            // reset(Socket socket)".
            status |= DOC_PARAM_MISSPELLED;
            paramDocumented(cp->name());
            nextUndocumented();
        }
        else if (AstFormalParameter* arg = findArgumentByName(docName)) {
            if (isDocumented(arg->name())) {
                status |= DOC_PARAM_REPEATED;
            }
            else {
                status |= DOC_PARAM_MISORDERED;
                paramDocumented(arg->name());
                // don't change the current parameter
            }
        }
        else if (arg = findArgumentByNameMisspelled(docName)) {
            status |= DOC_PARAM_MISORDERED_AND_MISSPELLED;
            paramDocumented(arg->name());
            // don't change the current parameter
        }
        else {
            status |= DOC_PARAM_NOT_IN_CODE;
        }
        return status;
    }
}

AstFormalParameter* ParameterDocumentationAnalyzer::currentParameter()
{
    if (arguments_) {
        if (currentParam_ < arguments_->getFormalParameterCount()) {
            DEBUG_DOCUMENTATION_ANALYZER(cout << "ParameterDocumentationAnalyzer:currentParameter: "
                                         << "currentParam_ " << currentParam_
                                         << " < arguments_->getFormalParameterCount() "
                                         << arguments_->getFormalParameterCount() << endl);
            return arguments_->getFormalParameter(currentParam_);
        }
        else {
            DEBUG_DOCUMENTATION_ANALYZER(cout << "ParameterDocumentationAnalyzer:currentParameter: "
                                         << "currentParam_ " << currentParam_
                                         << " < arguments_->getFormalParameterCount() "
                                         << arguments_->getFormalParameterCount() << endl);
            return NULL;
        }
    } 
    else {
        DEBUG_DOCUMENTATION_ANALYZER(cout << "ParameterDocumentationAnalyzer:currentParameter: no arguments" << endl);
        return NULL;
    }
}
