#ifndef Bc_h
#include "Bc.h"
#endif

#ifndef ByteArray_h
#include "ByteArray.h"
#endif

#ifndef FileByteStream_h
#include "FileByteStream.h"
#endif

#ifndef std_algorithm
#define std_algorithm
#include <algorithm>
#endif

#ifndef std_iterator
#define std_iterator
#include <iterator>
#endif

#include <iostream>

#include <math.h>

using namespace std;
using namespace doctorj;

typedef basic_string <wchar_t> wstring;

REGISTER_DEBUG_OPTION('b', "Byte code (decompiling)");

BcByteArraySource::BcByteArraySource(ByteArrayBigEndian* const bytes) : bytes_(bytes)
{
}

BcByteArraySource::~BcByteArraySource()
{
}

u1 BcByteArraySource::getU1()
{
    return bytes_->getU1();
}

u2 BcByteArraySource::getU2()
{
    return bytes_->getU2();
}

u4 BcByteArraySource::getU4()
{
    return bytes_->getU4();
}

char* BcByteArraySource::getBytes(int nBytes)
{
    return bytes_->getBytes(nBytes);
}


BcClassFileReader::BcClassFileReader(FileByteStreamBigEndian* const bytes)
{
    source_ = new BcClassFileSource(bytes);
    parse();
}


BcClassFileReader::BcClassFileReader(ByteArrayBigEndian* const bytes)
{
    source_ = new BcByteArraySource(bytes);
    parse();
}

BcClassFileReader::~BcClassFileReader()
{
    delete source_;
    
    vector<BcConstant*>::iterator it   = pool_.begin();
    vector<BcConstant*>::iterator stop = pool_.end();
    while (it != stop) {
        BcConstant* bc = *it;
        if (bc) {
            delete bc;
            bc = NULL;
        }
        ++it;
    }
}

u1 BcClassFileReader::getU1()
{
    return source_->getU1();
}

u2 BcClassFileReader::getU2()
{
    return source_->getU2();
}

u4 BcClassFileReader::getU4()
{
    return source_->getU4();
}

char* BcClassFileReader::getBytes(int nBytes)
{
    return source_->getBytes(nBytes);
}

void BcClassFileReader::parse()
{
    type_    = NULL;
    field_   = NULL;
    method_  = NULL;

    u4 magic = getU4();
    DEBUG_BYTE_CODE(cout << "magic                                   = " << (void*)magic << endl);
    
    u2 minor = getU2();
    u2 major = getU2();
    DEBUG_BYTE_CODE(cout << "version                                 = " << major << "." << minor << endl);

    parseConstantPool();
    parseClassInfo();
    parseInterfaces();
    parseFields();
    parseMethods();
    parseAttributes();
}

void BcClassFileReader::parseConstantPool()
{
    u2 cpct = getU2();

    pool_.push_back(NULL);      // so it is 1-indexed.
    
    int cp = 1; 
    DEBUG_BYTE_CODE(cout << "#constants                              = " << cpct << endl);
    while (cp < cpct) {
        DEBUG_BYTE_CODE(printf("%8d. ", cp));
        ++cp;
        u1 tag = getU1();

        switch (tag) {
            case 1:
                parseUTF8Constant();
                break;

                // What is 2? I don't see it in the JVM spec.
                
            case 3:
                parseIntegerConstant();
                break;
                
            case 4:
                parseFloatConstant();
                break;

            case 5:
                parseLongConstant();
                // we use up an extra spot
                pool_.push_back(NULL); // sloppy indexing
                ++cp;
                break;

            case 6:
                parseDoubleConstant();
                // we use up an extra spot
                pool_.push_back(NULL); // sloppy indexing
                ++cp;
                break;

            case 7:
                parseClassConstant();
                break;

            case 8:
                parseStringConstant();
                break;

            case 9:
                parseFieldRefConstant();
                break;

            case 10:
                parseMethodRefConstant();
                break;

            case 11:
                parseInterfaceMethodRefConstant();
                break;

            case 12:
                parseNameAndTypeConstant();
                break;
                
            default:
                DEBUG_BYTE_CODE(cout << "not handling constant type " << (int)tag << endl);
                break;
        }
    }
}


void BcClassFileReader::parseUTF8Constant()
{
    DEBUG_BYTE_CODE(cout << "UTF8 constant                 = "); 
    u2     len = getU2();
    char*  sb  = getBytes(len);
    string str = "<<<not ascii>>>"; // until we figure out otherwise

    // Figure out what kind of (so-called) utf8 constant this is.
    //     - Plain old ASCII (\u0001 through \u007F), one byte:
    //         - x: 0.......
    //     - \u0000 and \u0080 through \u07FF, two bytes: 
    //         - x: 110.....
    //         - y: 10......
    //         - value = ((x & 0x1f) << 6) + (y & 0x3f)
    //     - \u0800 through \uFFFF, three bytesA:
    //         - x: 1110....
    //         - y: 10......
    //         - z: 10......
    //         - value = ((x & 0xf) << 12) + ((y & 0x3f) << 6) + (z & 0x3f)

    bool isAscii = true;
    for (char* ptr = sb; ptr < sb + len; ++ptr) {
        char byte = *ptr;
        // printf("byte = %x\n", byte);
        if ((byte & 0x80) == 0) {
            // plain old ASCII
        }
        else {
            // not ASCII
            isAscii = false;
            break;
        }
    }
    
    if (isAscii) {
        str = string(sb, len);
    }

    DEBUG_BYTE_CODE(cout << "UTF8Constant string = '" << str << "'" << endl);
    delete [] sb;
    BcUTF8Constant* uc = new BcUTF8Constant(str);
    pool_.push_back(uc);
}


void BcClassFileReader::parseIntegerConstant()
{
    DEBUG_BYTE_CODE(cout << "integer constant              = "); 
    u4 value = getU4();
    DEBUG_BYTE_CODE(cout << value << endl);
    pool_.push_back(new BcIntegerConstant(value));
}

void BcClassFileReader::parseFloatConstant()
{
    DEBUG_BYTE_CODE(cout << "float constant                = "); 
    u4               bits = getU4();
    BcFloatConstant* fc   = NULL;
    if (bits == 0x7f800000) {
        // positive infinity
        DEBUG_BYTE_CODE(cout << "positive infinity" << endl);
        fc = new BcFloatConstant(CONST_POSITIVE_INFINITY);
    }
    else if (bits == 0xff800000) {
        // negative infinity
        DEBUG_BYTE_CODE(cout << "negative infinity" << endl);
        fc = new BcFloatConstant(CONST_NEGATIVE_INFINITY);
    }
    else if ((bits >= 0x7f800001 && bits <= 0x7fffffff) ||
             (bits >= 0xff800001 && bits <= 0xffffffff)) {
        // not a number
        DEBUG_BYTE_CODE(cout << "NaN" << endl);
        fc = new BcFloatConstant(CONST_NOT_A_NUMBER);
    }
    else {
        float s     = (bits >> 31) == 0 ? 1 : -1;
        float e     = (bits >> 23) & 0xff;
        float m     = e == 0 ? ((bits & 0x7fffff) << 1) : ((bits & 0x7fffff) | 0x800000);
        float value = s * m * pow(2, (e - 150));
        DEBUG_BYTE_CODE(cout << value << endl);
        fc = new BcFloatConstant(value);
    }
    pool_.push_back(fc);
}

void BcClassFileReader::parseLongConstant()
{
    DEBUG_BYTE_CODE(cout << "long constant                 = ");
    u4 hibys = getU4();
    u4 lobys = getU4();
    // how common is "long long"?
    long long value = ((long long)hibys << 32) + lobys;
    DEBUG_BYTE_CODE(cout << value << endl);
    pool_.push_back(new BcLongConstant(value));
}

void BcClassFileReader::parseDoubleConstant()
{
    DEBUG_BYTE_CODE(cout << "double constant               = ");
    u4 hibts = getU4();
    u4 lobts = getU4();
    long long bits  = ((long long)hibts << 32) + lobts;
    BcDoubleConstant* dc = NULL;
    if (bits == 0x7ff0000000000000LL) {
        // positive infinity
        DEBUG_BYTE_CODE(cout << "positive infinity" << endl);
        dc = new BcDoubleConstant(CONST_POSITIVE_INFINITY);
    }
    else if (bits == 0xfff0000000000000LL) {
        // negative infinity
        DEBUG_BYTE_CODE(cout << "negative infinity" << endl);
        dc = new BcDoubleConstant(CONST_NEGATIVE_INFINITY);
    }
    else if ((bits >= 0x7ff0000000000001LL && bits <= 0x7fffffffffffffffLL) ||
             (bits >= 0xfff0000000000001LL && bits <= 0xffffffffffffffffLL)) {
        // not a number
        DEBUG_BYTE_CODE(cout << "NaN" << endl);
        dc = new BcDoubleConstant(CONST_NOT_A_NUMBER);
    }
    else {
        double s = (bits >> 63) == 0 ? 1 : -1;
        double e = ((bits >> 52) & 0x7ff);
        double m = e == 0 ?
            ((bits & 0xfffffffffffffLL) << 1) :
            ((bits & 0xfffffffffffffLL) | 0x10000000000000LL);
        double value = s * m * pow(2, (e - 1075));
        DEBUG_BYTE_CODE(cout << value << endl);
        dc = new BcDoubleConstant(value);
    }
    pool_.push_back(dc);
}

void BcClassFileReader::parseClassConstant()
{
    DEBUG_BYTE_CODE(cout << "class constant                = ");
    u2 nameindex = getU2();
    DEBUG_BYTE_CODE(cout << nameindex << " (index)" << endl);
    pool_.push_back(new BcClassConstant(nameindex));
}

void BcClassFileReader::parseStringConstant()
{
    DEBUG_BYTE_CODE(cout << "string constant               = ");
    u2 index = getU2();
    DEBUG_BYTE_CODE(cout << index << " (index)" << endl);
    pool_.push_back(new BcStringConstant(index));
}

void BcClassFileReader::parseFieldRefConstant()
{
    DEBUG_BYTE_CODE(cout << "field ref constant            = ");
    u2 classIndex    = getU2();
    u2 nameTypeIndex = getU2();
    DEBUG_BYTE_CODE(cout << classIndex    << " (class index), ");
    DEBUG_BYTE_CODE(cout << nameTypeIndex << " (name type index)" << endl);
    pool_.push_back(new BcFieldRefConstant(classIndex, nameTypeIndex));
}

void BcClassFileReader::parseMethodRefConstant()
{
    DEBUG_BYTE_CODE(cout << "method ref constant           = ");
    u2 classIndex    = getU2();
    u2 nameTypeIndex = getU2();
    DEBUG_BYTE_CODE(cout << classIndex    << " (class index), ");
    DEBUG_BYTE_CODE(cout << nameTypeIndex << " (name type index)" << endl);
    pool_.push_back(new BcMethodRefConstant(classIndex, nameTypeIndex));
}

void BcClassFileReader::parseInterfaceMethodRefConstant()
{
    DEBUG_BYTE_CODE(cout << "interface method ref constant = ");
    u2 classIndex    = getU2();
    u2 nameTypeIndex = getU2();
    DEBUG_BYTE_CODE(cout << classIndex    << " (class index), " << endl);
    DEBUG_BYTE_CODE(cout << nameTypeIndex << " (name type index)" << endl);
    pool_.push_back(new BcInterfaceMethodRefConstant(classIndex, nameTypeIndex));
}

void BcClassFileReader::parseNameAndTypeConstant()
{
    DEBUG_BYTE_CODE(cout << "name/type constant            = ");
    u2 nameIndex = getU2();
    u2 descIndex = getU2();
    DEBUG_BYTE_CODE(cout << nameIndex << " (name index), ");
    DEBUG_BYTE_CODE(cout << descIndex << " (desc index)" << endl);
    pool_.push_back(new BcNameAndTypeConstant(nameIndex, descIndex));
}

string BcClassFileReader::className(int nameIndex)
{
    if (nameIndex >= pool_.size()) {
        cout << "ERROR: BcClassFileReader::className(" << nameIndex << "): beyond pool size " << pool_.size() << endl;
    }
    BcConstant*      bc = pool_[nameIndex];
    BcClassConstant* cc = dynamic_cast<BcClassConstant*>(bc);
    if (cc) {
        int ni = cc->index();
        return BcDescriptors::javaType(stringAt(ni));
    }
    else {
        DEBUG_BYTE_CODE(cout << "ERROR in BcClassFileReader::className(" << nameIndex << "): not a class constant" << endl);
        return "<<<notfound>>>";
    }
}

string BcClassFileReader::stringAt(int index)
{
    BcConstant* nmbc = pool_[index];
    BcUTF8Constant* un = dynamic_cast<BcUTF8Constant*>(nmbc);
    if (un) {
        return un->value();
    }
    else {
        DEBUG_BYTE_CODE(cout << "ERROR in BcClassFileReader::stringAt(" << index << "): not a UTF8 constant" << endl);
        return "<<<notfound>>>";
    }
}


void spewAccess(u2 accflag)
{
    if ((accflag & 0x0001) != 0) {
        DEBUG_BYTE_CODE(cout << "public " << endl);
    }
    if ((accflag & 0x0002) != 0) {
        DEBUG_BYTE_CODE(cout << "private " << endl);
    }
    if ((accflag & 0x0004) != 0) {
        DEBUG_BYTE_CODE(cout << "protected " << endl);
    }
    if ((accflag & 0x0008) != 0) {
        DEBUG_BYTE_CODE(cout << "static " << endl);
    }
    if ((accflag & 0x0010) != 0) {
        DEBUG_BYTE_CODE(cout << "final " << endl);
    }
    if ((accflag & 0x0020) != 0) {
        DEBUG_BYTE_CODE(cout << "super " << endl);
    }
    if ((accflag & 0x0040) != 0) {
        DEBUG_BYTE_CODE(cout << "volatile " << endl);
    }
    if ((accflag & 0x0080) != 0) {
        DEBUG_BYTE_CODE(cout << "transient " << endl);
    }
    if ((accflag & 0x0200) != 0) {
        DEBUG_BYTE_CODE(cout << "interface " << endl);
    }
    if ((accflag & 0x0400) != 0) {
        DEBUG_BYTE_CODE(cout << "abstract " << endl);
    }
}

void BcClassFileReader::parseClassInfo()
{
    u2     accFlags   = getU2();
    u2     nameIndex  = getU2();
    string name       = className(nameIndex);
    u2     superIndex = getU2();
    string superName = className(superIndex);
    type_ = new BcTypeDeclaration(name, accFlags, superName);
}

void BcClassFileReader::parseInterfaces()
{
    u2 nInterfaces = getU2();
    for (int i = 0; i < nInterfaces; ++i) {
        u2 index = getU2();
        string name = className(index);
        type_->addInterface(name);
    }
}

void BcClassFileReader::parseFields()
{
    u2 nFields = getU2();
    for (int i = 0; i < nFields; ++i) {
        parseField();
    }
}

void BcClassFileReader::parseAttribute()
{
    u2     nameIndex = getU2();
    u4     length    = getU4();
    string type      = stringAt(nameIndex);
    
    if (type == "Code") {
        parseCode();
    }
    else if (type == "SourceFile") {
        parseSourcefile();
    }
    else if (type == "LineNumberTable") {
        parseLineNumberTable();
    }
    else if (type == "LocalVariableTable") {
        parseLocalVariableTable();
    }
    else if (type == "Deprecated") {
        if (field_) {
            field_->setDeprecated();
        }
        else if (method_) {
            method_->setDeprecated();
        }
        else {
            type_->setDeprecated();
        }
    }
    else if (type == "ConstantValue") {
        parseConstantValue();
    }
    else if (type == "InnerClasses") {
        parseInnerClasses();
    }
    else if (type == "Synthetic") {
        if (field_) {
            field_->setSynthetic();
        }
        else if (method_) {
            method_->setSynthetic();
        }
        else {
            DEBUG_BYTE_CODE(cout << "ERROR: synthetic is applicable only to fields or methods" << endl);
        }
    }
    else if (type == "Exceptions") {
        parseExceptions();
    }
    else {
        DEBUG_BYTE_CODE(cout << "attribute '" << type << "': not understood" << endl);
    }
}
    

void BcClassFileReader::parseField()
{
    u2 accFlags  = getU2();
    u2 nameIndex = getU2();
    u2 descIndex = getU2();
    u2 attrCount = getU2();

    string name  = stringAt(nameIndex); // this is UTF8, not a class constant.
    string desc  = stringAt(descIndex);
    string jtype = BcDescriptors::toDescriptor(desc);

    field_ = new BcField(name, accFlags, jtype);
    type_->addField(field_);

    for (int i = 0; i < attrCount; ++i) {
        parseAttribute();
    }

    field_ = NULL;
}

void BcClassFileReader::parseCode()
{
    // we're just skipping through all the code, for now.
 
    u2 maxStack   = getU2();
    u2 maxLocals  = getU2();
    u4 codeLength = getU4();
    for (int i = 0; i < codeLength; ++i) {
        u1 code = getU1();
        DEBUG_BYTE_CODE(cout << (int)code << " ");
    }
    DEBUG_BYTE_CODE(cout << endl);

    // the exceptions
    u2 excTableLength = getU2();
    for (int i = 0; i < excTableLength; ++i) {
        parseExceptionTableEntry();
    }

    u2 attrCount = getU2();
    for (int i = 0; i < attrCount; ++i) {
        parseAttribute();
    }
}


void BcClassFileReader::parseSourcefile()
{
    // we just throw this away.
    u2 sfIndex = getU2();
    string sourceFile = stringAt(sfIndex);
}

void BcClassFileReader::parseLineNumberTable()
{
    u2 length = getU2();
    for (int i = 0; i < length; ++i) {
        u2 startPC    = getU2();
        u2 lineNumber = getU2();
    }
}

void BcClassFileReader::parseLocalVariableTable()
{
    u2 count = getU2();
    for (int i = 0; i < count; ++i) {
        u2 startPC   = getU2();
        u2 len       = getU2();
        u2 nameIndex = getU2();
        u2 descIndex = getU2();
        u2 index     = getU2();        
    }
}

void BcClassFileReader::parseConstantValue()
{
    u2 cvIndex = getU2();
    BcConstant* c = pool_[cvIndex];

    // this might be the thing itself, or it could just be pointing to it.
    BcScalarConstant* sc = dynamic_cast<BcScalarConstant*>(c);
    if (!sc) {
        BcStringConstant* strc = dynamic_cast<BcStringConstant*>(c);
        // cout << "as string constant = " << strc << endl);
        if (strc) {
            c = pool_[strc->index()];
            sc = dynamic_cast<BcScalarConstant*>(c);
            // cout << "    as UTF8 constant = " << sc << endl;
            if (sc) {
                // cout << "    as UTF8 constant = " << sc->asString() << endl;
            }
        }
    }
    
    if (sc && field_) {
        field_->setConstantValue(sc);
    }
}

void BcClassFileReader::parseInnerClasses()
{
    u2 nClasses = getU2();
    for (int i = 0; i < nClasses; ++i) {
        parseInnerClass();
    }
}

void BcClassFileReader::parseInnerClass()
{
    u2 innerInfoIndex = getU2();
    u2 outerInfoIndex = getU2();
    u2 innerNameIndex = getU2();
    u2 accFlags       = getU2();
}

void BcClassFileReader::parseExceptions()
{
    u2 nExceptions = getU2();
    for (int i = 0; i < nExceptions; ++i) {
        u2     index = getU2();
        string name  = className(index);
        if (method_) {
            method_->addException(name);
        }
        else {
            cout << "ERROR: no current method while processing an exception" << endl;
        }
    }
}

void BcClassFileReader::parseExceptionTableEntry()
{
    u2 startPC   = getU2();
    u2 endPC     = getU2();
    u2 handlerPC = getU2();
    u2 catchType = getU2();
    // not doing anything with this.
}

void BcClassFileReader::parseMethods()
{
    u2 count = getU2();
    for (int i = 0; i < count; ++i) {
        parseMethod();
    }
}

void BcClassFileReader::parseMethod()
{
    u2 accFlags  = getU2();
    u2 nameIndex = getU2();
    u2 descIndex = getU2();
    u2 attrCount = getU2();

    string desc = stringAt(descIndex);
    string retType;
    vector<string> paramTypes;
    expandDescriptor(desc, &retType, &paramTypes);

    string name = stringAt(nameIndex);
    if (name == "<init>") {
        // a constructor
        method_ = new BcConstructor(type_->name(), accFlags);
    }
    else if (name == "<clinit>") {
        // a static initializer
        method_ = new BcClassInit(type_->name(), accFlags);
    }
    else {
        // a plain old method
        method_ = new BcMethod(name, accFlags, retType);
    }
    method_->setParameters(paramTypes);

    type_->addMethod(method_);

    for (int i = 0; i < attrCount; ++i) {
        parseAttribute();
    }

    method_ = NULL;
}

void BcClassFileReader::expandDescriptor(const string& desc, 
                                         string* const retType, 
                                         vector<string>* const parameterTypes)
{
    int    rparen = desc.find(')');
    string pt     = desc.substr(1, rparen - 1);
    string rt     = desc.substr(rparen + 1);
    *retType      = BcDescriptors::toDescriptor(rt);

    int ppos = 0;
    while (ppos < pt.length()) {
        string param = BcDescriptors::toDescriptor(pt, &ppos);
        parameterTypes->push_back(param);
    }
}

void BcClassFileReader::parseAttributes()
{
    u2 attrCount = getU2();
    for (int i = 0; i < attrCount; ++i) {
        parseAttribute();
    }
}

BcTypeDeclaration* BcClassFileReader::getType() const
{
    return type_;
}


BcClassFileSource::BcClassFileSource(FileByteStream* const bytes) : bytes_(bytes)
{
}

BcClassFileSource::~BcClassFileSource()
{
}

u1 BcClassFileSource::getU1()
{
    return bytes_->getU1();
}

u2 BcClassFileSource::getU2()
{
    return bytes_->getU2();
}

u4 BcClassFileSource::getU4()
{
    return bytes_->getU4();
}

char* BcClassFileSource::getBytes(int nBytes)
{
    return bytes_->getBytes(nBytes);
}



BcClassInit::BcClassInit(const string& name, int access)
        : BcMethod(name, access, "void")
{
}

BcClassInit::~BcClassInit()
{
}


BcConstant::BcConstant()
{
}

BcConstant::~BcConstant()
{
}


BcScalarConstant::BcScalarConstant()
{
}

BcScalarConstant::~BcScalarConstant()
{
}


BcUTF8Constant::BcUTF8Constant(const string& v) : BcGenericConstant<string>(v)
{
}

BcUTF8Constant::~BcUTF8Constant()
{
}

    
BcIntegerConstant::BcIntegerConstant(int v) : BcGenericConstant<int>(v)
{
}
     
BcIntegerConstant::~BcIntegerConstant()
{
}


BcFloatConstant::BcFloatConstant(float v) : BcGenericConstant<float>(v)
{
}
     
BcFloatConstant::~BcFloatConstant()
{
}


BcLongConstant::BcLongConstant(long long v) : BcGenericConstant<long long>(v)
{
}
     
BcLongConstant::~BcLongConstant()
{
}


BcDoubleConstant::BcDoubleConstant(double v) : BcGenericConstant<double>(v)
{
}
     
BcDoubleConstant::~BcDoubleConstant()
{
}


BcClassConstant::BcClassConstant(int index) : index_(index)
{
}

BcClassConstant::~BcClassConstant()
{
}

int BcClassConstant::index() const
{
    return index_;
}


BcStringConstant::BcStringConstant(int index) : index_(index)
{
}

BcStringConstant::~BcStringConstant()
{
}

int BcStringConstant::index() const
{
    return index_;
}


BcRefConstant::BcRefConstant(int classIndex, int nameTypeIndex) : 
        classIndex_(classIndex), 
     nameTypeIndex_(nameTypeIndex)
{
}

BcRefConstant::~BcRefConstant()
{
}

int BcRefConstant::classIndex() const
{
    return classIndex_;
}

int BcRefConstant::nameTypeIndex() const
{
    return nameTypeIndex_;
}


BcFieldRefConstant::BcFieldRefConstant(int classIndex, int nameTypeIndex)
        : BcRefConstant(classIndex, nameTypeIndex)
{
}

BcFieldRefConstant::~BcFieldRefConstant()
{
};


BcMethodRefConstant::BcMethodRefConstant(int classIndex, int nameTypeIndex)
        : BcRefConstant(classIndex, nameTypeIndex)
{
}

BcMethodRefConstant::~BcMethodRefConstant()
{
};


BcInterfaceMethodRefConstant::BcInterfaceMethodRefConstant(int classIndex, int nameTypeIndex) 
        : BcRefConstant(classIndex, nameTypeIndex)
{
}

BcInterfaceMethodRefConstant::~BcInterfaceMethodRefConstant()
{
}


BcNameAndTypeConstant::BcNameAndTypeConstant(int nameIndex, int descIndex) :
        nameIndex_(nameIndex),
     descIndex_(descIndex)
{
}

BcNameAndTypeConstant::~BcNameAndTypeConstant()
{
}

int BcNameAndTypeConstant::nameIndex() const
{
    return nameIndex_;
}

int BcNameAndTypeConstant::descIndex() const
{
    return descIndex_;
}

template <>
string BcGenericConstant<string>::asString() const
{
    return "\"" + StringUtilities::toString(value_) + "\"";
}


BcConstantValue::BcConstantValue()
{
}
        
BcConstantValue::~BcConstantValue()
{
}


BcConstructor::BcConstructor(const string& name, int access)
        : BcMethod(name, access, "void")
{
}

BcConstructor::~BcConstructor()
{
}


string BcDescriptors::toDescriptor(const string& type, int* const pos)
{
    // either:
    //     Lsomething/like/this;
    // or
    //     D
    // with any number of leading ['s (for arrays)

    int arrayDepth = 0;
    int p = pos ? *pos : 0;

    // advance past arrays.
    while (type.find("[", p) == p) {
        ++arrayDepth;
        ++p;
    }

    int origpos = p;

    // non-primitive
    if (type[p] == 'L') {
        // it's a class, so get through the first semicolon
        while (type[p++] != ';') {
            // nothing
        }
    }
    else {
        // it's a primitive, which is only one character
        ++p;
    }

    string pt   = type.substr(origpos, p - origpos);

    string desc = "<<unknown>>";
    switch (pt[0]) {
        case 'B': desc = "byte";         break;
        case 'C': desc = "char";         break;
        case 'D': desc = "double";       break;
        case 'F': desc = "float";        break;
        case 'I': desc = "int";          break;
        case 'J': desc = "long";         break;
        case 'L': desc = javaType(pt.substr(1, pt.find(";") - 1)); break;
        case 'S': desc = "short";        break;
        case 'V': desc = "void";         break;
        case 'Z': desc = "boolean";      break;
        default: cerr << "bad desc '" << type[0] << "'" << endl;
    }

    while (arrayDepth > 0) {
        desc += "[]";
        --arrayDepth;
    }

    if (pos) {
        *pos = p;
    }

    return desc;
}


string BcDescriptors::javaType(const string& type)
{
    string jtype = type;
    StringUtilities::subst(&jtype, "/", ".");

    // why did I write the following line in the Ruby version?
    // jn.gsub!(/^.*\$/, "")

    return jtype;
}


BcElement::BcElement(const string& name, 
                     int access, 
                     const string& type /* = string("") */, 
                     bool deprecated /* = false */,
                     bool synthetic  /* = false */) :
        name_(name),
     type_(type),
     access_(access),
     deprecated_(deprecated),
     synthetic_(synthetic)
{
}

BcElement::~BcElement()
{
}


int BcElement::access() const
{
    return access_;
}

string BcElement::name() const
{
    return name_;
}

string BcElement::type() const
{
    return type_;
}

bool BcElement::isDeprecated() const
{
    return deprecated_;
}

bool BcElement::isSynthetic() const
{
    return synthetic_;
}

void BcElement::setDeprecated()
{
    deprecated_ = true;
}

void BcElement::setSynthetic()
{
    synthetic_ = true;
}

string BcElement::expandAccess() const
{
    string str;
    if ((access_ & BC_ACCESS_PUBLIC) != 0) {
        str += "public ";
    }
    if ((access_ & BC_ACCESS_PRIVATE) != 0) {
        str += "private ";
    }
    if ((access_ & BC_ACCESS_PROTECTED) != 0) {
        str += "protected ";
    }
    if ((access_ & BC_ACCESS_STATIC) != 0) {
        str += "static ";
    }
    if ((access_ & BC_ACCESS_FINAL) != 0) {
        str += "final ";
    }
//     if ((access_ & BC_ACCESS_SUPER) != 0) {
//         str += "super ";
//     }
    if ((access_ & BC_ACCESS_VOLATILE) != 0) {
        str += "volatile ";
    }
    if ((access_ & BC_ACCESS_TRANSIENT) != 0) {
        str += "transient ";
    }
//     if ((access_ & BC_ACCESS_INTERFACE) != 0) {
//         str += "interface ";
//     }
    if ((access_ & BC_ACCESS_ABSTRACT) != 0) {
        str += "abstract ";
    }
    return str;
}


BcField::BcField(const string& name, 
                 int access, 
                 const string& type, 
                 bool deprecated /* = false */,
                 bool synthetic  /* = false */)
        : BcElement(name, access, type, deprecated, synthetic), cv_(NULL)
{
}

BcField::~BcField()
{
}

BcScalarConstant* BcField::constantValue() const
{
    return cv_;
}

void BcField::setConstantValue(BcScalarConstant* const cv)
{
    cv_ = cv;
}


BcMethod::BcMethod(const string& name, 
                   int access, 
                   const string& type, 
                   bool deprecated /* = false */,
                   bool synthetic  /* = false */)
        : BcElement(name, access, type, deprecated, synthetic)
{
}

BcMethod::~BcMethod()
{
}

vector<string> BcMethod::parameterTypes() const
{
    return parameterTypes_;
}

void BcMethod::addParameter(const string& pType)
{
    parameterTypes_.push_back(pType);
}

void BcMethod::setParameters(const vector<string>& params)
{
    parameterTypes_ = params;
}

vector<string> BcMethod::exceptions() const
{
    return exceptions_;
}

void BcMethod::addException(const string& exc)
{
    exceptions_.push_back(exc);
}


BcSource::BcSource()
{
}

BcSource::~BcSource()
{
}


BcTypeDeclaration::BcTypeDeclaration(const string& name, 
                                     int access, 
                                     const string& superClass)
        : BcElement(name, access), superClass_(superClass)
{
}

BcTypeDeclaration::~BcTypeDeclaration()
{
}

vector<BcMethod*> BcTypeDeclaration::methods() const
{
    return methods_;
}

vector<BcField*> BcTypeDeclaration::fields() const
{
    return fields_;
}

vector<BcTypeDeclaration*> BcTypeDeclaration::innerTypes() const
{
    return innerTypes_;
}

void BcTypeDeclaration::addMethod(BcMethod* const method)
{
    methods_.push_back(method);
}

void BcTypeDeclaration::addField(BcField* const field)
{
    fields_.push_back(field);
}

void BcTypeDeclaration::addInnerTypes(BcTypeDeclaration* const td)
{
    innerTypes_.push_back(td);
}

vector<string> BcTypeDeclaration::superInterfaces() const
{
    return superInterfaces_;
}

void BcTypeDeclaration::addInterface(const string& i)
{
    superInterfaces_.push_back(i);
}

string BcTypeDeclaration::superClass() const
{
    return superClass_;
}

void BcTypeDeclaration::setSuperClass(const string& sc)
{
    superClass_ = sc;
}

bool BcTypeDeclaration::isClass() const
{
    return (access() & BC_ACCESS_INTERFACE) == 0;
}

static void write(BcMethod* const method, bool showType)
{
    if (method->isDeprecated()) {
        cout << "    /* deprecated */" << endl;
    }
    if (method->isSynthetic()) {
        cout << " /* synthetic */";
    }
    cout << "    ";
    cout << method->expandAccess();
    if (showType) {
        cout << method->type() << " ";
    }
    cout << method->name();
    cout << "(";
    if (method->parameterTypes().size() > 0) {
        vector<string> params = method->parameterTypes();
        copy(params.begin(), params.end() - 1, ostream_iterator<string>(cout, ", "));
        cout << *(params.end() - 1);
    }
    cout << ")";
        
    if (method->exceptions().size() > 0) {
        cout << " throws ";
        vector<string> excs = method->exceptions();
        copy(excs.begin(), excs.end() - 1, ostream_iterator<string>(cout, ", "));
        cout << *(excs.end() - 1);
    }
    
    cout << ";";
    cout << endl;
}

void BcTypeDeclaration::print()
{
    if (isDeprecated()) {
        cout << "/* deprecated */" << endl;
    }
    if (isSynthetic()) {
        cout << "/* synthetic */" << endl;
    }
    cout << expandAccess() << (isClass() ? "class" : "interface") << " " << name() << " extends " << superClass();
    if (superInterfaces_.size() > 0) {
        cout << " implements ";
        copy(superInterfaces_.begin(), superInterfaces_.end() - 1, ostream_iterator<string>(cout, ", "));
        cout << *(superInterfaces_.end() - 1);
    }
    cout << " {" << endl;

    // fields
    vector<BcField*>::iterator fit   = fields_.begin();
    vector<BcField*>::iterator fstop = fields_.end();
    while (fit != fstop) {
        BcField* field = *fit;
        cout << "    " << field->expandAccess() << field->type() << " " << field->name();
        BcScalarConstant* cv = field->constantValue();
        if (cv) {
            string str = cv->asString();
            DEBUG_BYTE_CODE(cout << " = ");
            int len = str.length();
            //$$$ Actually, should just figure out if it is unicode. Convert it?
            if (len > 80) {
                cout << "<<string of length " << len << ">>";
            }
            else {
                cout << cv->asString();
            }
        }
        cout << ";";
        if (field->isDeprecated()) {
            cout << " /* deprecated */";
        }
        if (field->isSynthetic()) {
            cout << " /* synthetic */";
        }
        cout << endl;

        ++fit;
    }

    // methods
    vector<BcMethod*>::iterator mit   = methods_.begin();
    vector<BcMethod*>::iterator mstop = methods_.end();
    while (mit != mstop) {
        BcMethod* method = *mit;
//         if (!dynamic_cast<BcClassInit*>(method)) {
            write(method, !dynamic_cast<BcConstructor*>(method));
        // }
        ++mit;
    }

    cout << "}" << endl;
    
}


