#ifndef Bc_h
#define Bc_h

#ifndef Debug_h
#include "Debug.h"
#endif

#ifndef Types_h
#include "Types.h"
#endif

#ifndef StringUtilities_h
#include "StringUtilities.h"
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif

#ifndef std_string
#define std_string
#include <string>
#endif

#ifdef DEBUGGING
#define DEBUG_BYTE_CODE(a)  if (DebugOptions::get()->isEnabled('b'))    a
#else
#define DEBUG_BYTE_CODE(a)  
#endif

using namespace std;

namespace doctorj
{
    class BcConstant;
    class ByteArrayBigEndian;

    class BcSource
    {
    public:
        BcSource();

        virtual ~BcSource();

        /**
         * Returns an unsigned byte (one byte) from the stream.
         */
        virtual u1 getU1() = 0;

        /**
         * Returns an unsigned short (two bytes) from the stream.
         */
        virtual u2 getU2() = 0;

        /**
         * Returns an unsigned int (four bytes) from the stream.
         */
        virtual u4 getU4() = 0;

        /**
         * Gets n bytes and returns a pointer to the first. The caller must
         * delete the returned bytes, which should be done as "delete []".
         */
        virtual char* getBytes(int nBytes) = 0;

    };


    class BcByteArraySource : public BcSource
    {
    public:
        BcByteArraySource(ByteArrayBigEndian* const bytes);

        virtual ~BcByteArraySource();

        /**
         * Returns an unsigned byte (one byte) from the stream.
         */
        virtual u1 getU1();

        /**
         * Returns an unsigned short (two bytes) from the stream.
         */
        virtual u2 getU2();

        /**
         * Returns an unsigned int (four bytes) from the stream.
         */
        virtual u4 getU4();

        /**
         * Gets n bytes and returns a pointer to the first. The caller must
         * delete the returned bytes, which should be done as "delete []".
         */
        virtual char* getBytes(int nBytes);

    private:

        ByteArrayBigEndian* bytes_;

    };

    enum BcAccessType {
        BC_ACCESS_PUBLIC    = 0x0001,
        BC_ACCESS_PRIVATE   = 0x0002,
        BC_ACCESS_PROTECTED = 0x0004,
        BC_ACCESS_STATIC    = 0x0008,
        BC_ACCESS_FINAL     = 0x0010,
        BC_ACCESS_SUPER     = 0x0020,
        BC_ACCESS_VOLATILE  = 0x0040,
        BC_ACCESS_TRANSIENT = 0x0080,
        BC_ACCESS_INTERFACE = 0x0200,
        BC_ACCESS_ABSTRACT  = 0x0400
    };

    /**
     * The characters common to all Java elements, such as methods, fields, and
     * classes.
     */
    class BcElement
    {
    public:
        BcElement(const string& name, 
                  int access, 
                  const string& type = string(""), 
                  bool deprecated = false,
                  bool synthetic  = false);

        virtual ~BcElement();

        /**
         * Returns the access (bit-or'ed) for this element.
         */
        int access() const;

        /**
         * Returns the name of this element.
         */
        string name() const;

        /**
         * Returns the type of this element. See subclasses for what
         * specifically the type refers to.
         */
        string type() const;

        /**
         * Returns whether this element is deprecated.
         */
        bool isDeprecated() const;

        /**
         * Returns whether this element is synthetic.
         */
        bool isSynthetic() const;

        /**
         * Sets this element as deprecated.
         */
        void setDeprecated();

        /**
         * Sets this element as synthetic.
         */
        void setSynthetic();

        /**
         * Returns access as a string.
         */
        string expandAccess() const;

    private:
        string name_;

        string type_;

        int access_;

        bool deprecated_;

        bool synthetic_;

    };

    class BcScalarConstant;
    
    /**
     * A field extracted from a .class file. Note that <code>type()</code> (from
     * BcElement) refers to the variable type, such as "int" or
     * "java.lang.Integer".
     */
    class BcField : public BcElement
    {
    public:
        BcField(const string& name, 
                int access, 
                const string& type, 
                bool deprecated = false,
                bool synthetic  = false);
        
        virtual ~BcField();

        /**
         * Returns the constant value, or NULL if none.
         */
        BcScalarConstant* constantValue() const;

        /**
         * Sets the constant value. Ownership is taken by the field.
         */
        void setConstantValue(BcScalarConstant* const cv);

    private:
        BcScalarConstant* cv_;

    };


    class BcField;
    class BcMethod;
    class BcSource;
    class BcTypeDeclaration;
    class ByteArrayBigEndian;
    class FileByteStreamBigEndian;
    
    class BcClassFileReader
    {
    public:
        BcClassFileReader(FileByteStreamBigEndian* const bytes);

        BcClassFileReader(ByteArrayBigEndian* const bytes);

        virtual ~BcClassFileReader();

        /**
         * Returns the type that was parsed from the .class.
         */
        BcTypeDeclaration* getType() const;

    protected:
        void parse();

        // constant pool

        void parseConstantPool();

        void parseUTF8Constant();

        void parseIntegerConstant();

        void parseFloatConstant();

        void parseLongConstant();

        void parseDoubleConstant();

        void parseClassConstant();

        void parseStringConstant();

        void parseFieldRefConstant();

        void parseMethodRefConstant();

        void parseInterfaceMethodRefConstant();
        
        void parseNameAndTypeConstant();

        // class info

        void parseClassInfo();

        /**
         * Returns the class name for the given name index. Note that the name
         * index points to a class constant, which in turn points to a UTF8
         * constant. Returns "<<<notfound>>>" if not found.
         */
        string className(int nameIndex);

        /**
         * Returns the string (UTF8) at the given index. Returns
         * "<<<notfound>>>" if not found or the given index is not a UTF8
         * constant.
         */
        string stringAt(int index);

        // interface info (supers)

        /**
         * Parses the interfaces.
         */
        void parseInterfaces();

        // field info

        /**
         * Parses the fields.
         */
        void parseFields();

        /**
         * Parses a single field (as opposed to the married ones).
         */
        void parseField();

        // attributes

        /**
         * Parses a list of (ClassFile) attributes.
         */
        void parseAttributes();

        /**
         * Parses an attribute.
         */
        void parseAttribute();

        /**
         * Parses code for a method.
         */
        void parseCode();

        /**
         * Parses the source file information.
         */
        void parseSourcefile();

        /**
         * Parses a line number table.
         */
        void parseLineNumberTable();

        /**
         * Parses a local variable table.
         */
        void parseLocalVariableTable();

        /**
         * Parses a constant value.
         */
        void parseConstantValue();

        /**
         * Parses inner classes.
         */
        void parseInnerClasses();

        /**
         * Parses a single inner class.
         */
        void parseInnerClass();

        /**
         * Parses exceptions.
         */
        void parseExceptions();

        /**
         * Parses an entry in an exception table.
         */
        void parseExceptionTableEntry();

        // methods

        /**
         * Parses all methods.
         */
        void parseMethods();

        /**
         * Parses a method.
         */
        void parseMethod();

        /**
         * Returns the descriptor string as a return type and a vector of
         * parameter types, all in the original form, such as "int",
         * "java.lang.String".
         */
        void expandDescriptor(const string& desc, 
                              string* const retType, 
                              vector<string>* const parameterTypes);

        /**
         * Returns an unsigned byte (one byte) from the stream.
         */
        virtual u1 getU1();

        /**
         * Returns an unsigned short (two bytes) from the stream.
         */
        virtual u2 getU2();

        /**
         * Returns an unsigned int (four bytes) from the stream.
         */
        virtual u4 getU4();

        /**
         * Gets n bytes and returns a pointer to the first. The caller must
         * delete the returned bytes, which should be done as "delete []".
         */
        virtual char* getBytes(int nBytes);

    private:
        BcSource* source_;

        vector<BcConstant*> pool_;

        BcTypeDeclaration* type_;

        /**
         * The current field being parsed, or NULL if not applicable.
         */
        BcField* field_;

        /**
         * The current method being parsed, or NULL if not applicable.
         */
        BcMethod* method_;
        
    };

    class FileByteStream;

    class BcClassFileSource : public BcSource
    {
    public:
        BcClassFileSource(FileByteStream* const bytes);

        virtual ~BcClassFileSource();

        /**
         * Returns an unsigned byte (one byte) from the stream.
         */
        virtual u1 getU1();

        /**
         * Returns an unsigned short (two bytes) from the stream.
         */
        virtual u2 getU2();

        /**
         * Returns an unsigned int (four bytes) from the stream.
         */
        virtual u4 getU4();

        /**
         * Gets n bytes and returns a pointer to the first. The caller must
         * delete the returned bytes, which should be done as "delete []".
         */
        virtual char* getBytes(int nBytes);

    private:

        FileByteStream* bytes_;

    };


    /**
     * A method extracted from a .class file. Note that <code>type()</code>
     * (from BcElement) refers to the return type.
     */
    class BcMethod : public BcElement
    {
    public:
        BcMethod(const string& name, 
                 int access, 
                 const string& type, 
                 bool deprecated = false,
                 bool synthetic  = false);
        
        virtual ~BcMethod();

        // bytes (code) not supported

        /**
         * Returns a list of the parameter types, which are in Java format,
         * i.e., "java.lang.String", not "Ljava/lang/String;".
         */
        vector<string> parameterTypes() const;

        /**
         * Adds a parameter type.
         */
        void addParameter(const string& pType);

        /**
         * Sets all the parameter types.
         */
        void setParameters(const vector<string>& params);

        /**
         * Returns a list of the exceptions thrown, which are in Java format,
         * i.e., "java.io.IOException", not "Ljava/io/IOException;".
         */
        vector<string> exceptions() const;

        /**
         * Adds a exception.
         */
        void addException(const string& exc);

    private:
        
        /**
         * A list of the parameter types, which are in Java format, i.e.,
         * "java.lang.String", not "Ljava/lang/String;".
         */
        vector<string> parameterTypes_;

        /**
         * a list of the exceptions thrown, which are in Java format, i.e.,
         * "java.io.IOException", not "Ljava/io/IOException;".
         */
        vector<string> exceptions_;

    };


    /**
     * A class initializer extracted from a .class file. Note that
     * <code>type()</code> (from BcElement) has no significance for a class
     * initializer.
     */
    class BcClassInit : public BcMethod
    {
    public:
        BcClassInit(const string& name, // name == class name, not "<clinit>"
                    int access);
        
        virtual ~BcClassInit();

    };

    
    enum NumberType {
        CONST_NORMAL,
        CONST_POSITIVE_INFINITY,
        CONST_NEGATIVE_INFINITY,
        CONST_NOT_A_NUMBER            // NaN
    };

    /**
     * A constant in the constant pool.
     */
    class BcConstant
    {
    public:
        BcConstant();

        virtual ~BcConstant();
    };
    

    /**
     * A single-value constant in the constant pool.
     */
    class BcScalarConstant : public BcConstant
    {
    public:
        BcScalarConstant();

        virtual ~BcScalarConstant();

        virtual string asString() const = 0;
    };
    

    /**
     * A constant in the constant pool. This type of constant has a single
     * value, which represents itself. Compare with the constants which are
     * indices into other positions within the constant pool.
     */
    template <typename Type>
    class BcGenericConstant : public BcScalarConstant
    {
    public:
        BcGenericConstant(const Type& v);

        virtual ~BcGenericConstant();

        Type value() const;

        virtual string asString() const;

    protected:
        BcGenericConstant();

    private:
        Type value_;
    };

    
    class BcUTF8Constant : public BcGenericConstant<string>
    {
    public:
        BcUTF8Constant(const string& v);

        virtual ~BcUTF8Constant();
    };

    
    class BcIntegerConstant : public BcGenericConstant<int>
    {
    public:
        BcIntegerConstant(int v);

        virtual ~BcIntegerConstant();
    };


    /**
     * This can be infinity or not a number (NaN).
     */
    template <typename Type>
    class BcNumberConstant : public BcGenericConstant<Type>
    {
    public:
        BcNumberConstant(Type v);

        BcNumberConstant(NumberType t);

        virtual ~BcNumberConstant();

        bool isPositiveInfinity() const;

        bool isNegativeInfinity() const;

        bool isNotANumber() const; // i.e., NaN

        bool isNormal() const;

    private:
        NumberType type_;

    };

    
    class BcFloatConstant : public BcGenericConstant<float>
    {
    public:
        BcFloatConstant(float v);

        virtual ~BcFloatConstant();
    };

    
    class BcLongConstant : public BcGenericConstant<long long>
    {
    public:
        BcLongConstant(long long v);

        virtual ~BcLongConstant();
    };

    
    class BcDoubleConstant : public BcGenericConstant<double>
    {
    public:
        BcDoubleConstant(double v);

        virtual ~BcDoubleConstant();
    };


    class BcClassConstant : public BcConstant
    {
    public:
        BcClassConstant(int index);

        virtual ~BcClassConstant();

        int index() const;

    private:
        int index_;
    };


    class BcStringConstant : public BcConstant
    {
    public:
        BcStringConstant(int index);

        virtual ~BcStringConstant();

        int index() const;

    private:
        int index_;
    };


    class BcRefConstant : public BcConstant
    {
    public:
        BcRefConstant(int classIndex, int nameTypeIndex);

        virtual ~BcRefConstant();

        int classIndex() const;

        int nameTypeIndex() const;

    private:
        int classIndex_;

        int nameTypeIndex_;
    };


    class BcFieldRefConstant : public BcRefConstant
    {
    public:
        BcFieldRefConstant(int classIndex, int nameTypeIndex);

        virtual ~BcFieldRefConstant();
    };


    class BcMethodRefConstant : public BcRefConstant
    {
    public:
        BcMethodRefConstant(int classIndex, int nameTypeIndex);

        virtual ~BcMethodRefConstant();
    };


    class BcInterfaceMethodRefConstant : public BcRefConstant
    {
    public:
        BcInterfaceMethodRefConstant(int classIndex, int nameTypeIndex);

        virtual ~BcInterfaceMethodRefConstant();
    };


    class BcNameAndTypeConstant : public BcConstant
    {
    public:
        BcNameAndTypeConstant(int nameIndex, int descIndex);

        virtual ~BcNameAndTypeConstant();

        int nameIndex() const;

        int descIndex() const;

    private:
        int nameIndex_;

        int descIndex_;
    };

    template <typename Type >
    BcGenericConstant<Type>::BcGenericConstant(const Type& v) : value_(v)
    {
    }

    template <typename Type >
    BcGenericConstant<Type>::BcGenericConstant()
    {
    }

    template <typename Type>
    BcGenericConstant<Type>::~BcGenericConstant()
    {
    }

    template <typename Type >
    Type BcGenericConstant<Type>::value() const
    {
        return value_;
    }

    template <typename Type >
    string BcGenericConstant<Type>::asString() const
    {
        return StringUtilities::toString(value_);
    }

    template <typename Type>
    BcNumberConstant<Type>::BcNumberConstant(Type v) : BcGenericConstant<Type>(v), type_(CONST_NORMAL)
    {
    }

    template <typename Type>
    BcNumberConstant<Type>::BcNumberConstant(NumberType t) : type_(t)
    {
    }

    template <typename Type>
    BcNumberConstant<Type>::~BcNumberConstant()
    {
    }

    template <typename Type>
    bool BcNumberConstant<Type>::isPositiveInfinity() const
    {
        return type_ == CONST_POSITIVE_INFINITY;
    }

    template <typename Type>
    bool BcNumberConstant<Type>::isNegativeInfinity() const
    {
        return type_ == CONST_NEGATIVE_INFINITY;
    }

    template <typename Type>
    bool BcNumberConstant<Type>::isNotANumber() const
    {
        return type_ == CONST_NOT_A_NUMBER;
    }

    template <typename Type>
    bool BcNumberConstant<Type>::isNormal() const
    {
        return type_ == CONST_NORMAL;
    }


    /**
     * A field extracted from a .class file.
     */
    class BcConstantValue
    {
    public:
        BcConstantValue();
        
        virtual ~BcConstantValue();
        
        /**
         * Returns the value, as a string.
         */
        virtual string asString() const = 0;

    };


    /**
     * A constructor extracted from a .class file. Note that <code>type()</code>
     * (from BcElement) has no significance for a constructor.
     */
    class BcConstructor : public BcMethod
    {
    public:
        BcConstructor(const string& name, // name == class name, not "<init>"
                      int access);
        
        virtual ~BcConstructor();

    };


    /**
     * Returns Java (source) descriptors for a given type. Ref. JVM Spec.,
     * 4.3.2.
     */
    namespace BcDescriptors {
        /**
         * Returns a descriptor for given type. For example, "I" => "int".
         * Handles arrays, e.g., "[[Z" => "boolean[][]". If <code>pos</code> is
         * provided, the string is parsed starting at the given position, and
         * the resulting value of <code>pos</code> will be the next position.
         */
        string toDescriptor(const string& type, int* const pos = NULL);

        /**
         * Returns a Java (.java file) version of the given class name (from a
         * .class file). For example, "Ljava/lang/String" => "java.lang.String".
         */
        string javaType(const string& type);

    };


    /**
     * A field extracted from a .class file.
     */
    template <typename Type>
    class BcGenericConstantValue : public BcConstantValue
    {
    public:
        BcGenericConstantValue(Type v);
        
        virtual ~BcGenericConstantValue();
        
        /**
         * Returns the constant value. This will be invalid if not set.
         */
        Type value() const;

        /**
         * Sets the constant value.
         */
        void setValue(Type v);

        /**
         * Returns the value, as a string.
         */
        virtual string asString() const;

    private:

        /**
         * the constant value.
         */
        Type value_;

    };

    template <typename Type>
    BcGenericConstantValue<Type>::BcGenericConstantValue(Type v) : value_(v)
    {
    }
        
    template <typename Type>
    BcGenericConstantValue<Type>::~BcGenericConstantValue()
    {
    }
     
    template <typename Type>
    Type BcGenericConstantValue<Type>::value() const
    {
        return value_;
    }

    template <typename Type>
    void BcGenericConstantValue<Type>::setValue(Type v)
    {
        value_ = v;
    }

    template <typename Type>
    string BcGenericConstantValue<Type>::asString() const
    {
        return StringUtilities::toString(value_);
    }

    
    /**
     * A class or an interface. Can have methods, fields, inner types, and any
     * number of interfaces " "implemented" (or "extended" in the case of an
     * interface).
    */
    class BcTypeDeclaration : public BcElement
    {
    public:
        BcTypeDeclaration(const string& name, int access, const string& superClass);

        virtual ~BcTypeDeclaration();

        /**
         * Returns the methods within this type declaration.
         */
        vector<BcMethod*> methods() const;

        /**
         * Returns the fields within this type declaration.
         */
        vector<BcField*> fields() const;

        /**
         * Returns the inner types within this type declaration.
         */
        vector<BcTypeDeclaration*> innerTypes() const;

        /**
         * Adds a method to this type declaration. Note that ownership of the
         * method is taken by this object, which will delete it when this goes
         * out of scope.
         */
        void addMethod(BcMethod* const method);

        /**
         * Returns the fields within this type declaration.
         */
        void addField(BcField* const field);

        /**
         * Returns the inner types within this type declaration.
         */
        void addInnerTypes(BcTypeDeclaration* const td);
        
        /**
         * Returns the superclass, in Java format ("java.lang.Object").
         */
        string superClass() const;

        /**
         * Sets the superclass.
         */
        void setSuperClass(const string& sc);

        /**
         * Returns the interfaces implemented, in Java format
         * ("java.lang.Object").
         */
        vector<string> superInterfaces() const;

        /**
         * Returns the interfaces implemented, in Java format
         * ("java.lang.Object").
         */
        void addInterface(const string& i);

        /**
         * For debugging purposes only.
         */
        void print();

        /**
         * Returns whether this type declaration is a class. If not, it is an
         * interface.
         */
        bool isClass() const;

    private:

        /**
         * The methods within this type declaration.
         */
        vector<BcMethod*> methods_;

        /**
         * The fields within this type declaration.
         */
        vector<BcField*> fields_;
       
        /**
         * Returns the inner types within this type declaration.
         */
        vector<BcTypeDeclaration*> innerTypes_;

        /**
         * The superclass, in Java format ("java.lang.Object").
         */
        string superClass_;

        /**
         * The interfaces implemented, in Java format ("java.lang.Object").
         */
        vector<string> superInterfaces_;

    };

}    

#endif //! Bc_h
