#ifndef Parser_h
#define Parser_h

#ifndef AST_h
#include "AST.h"
#endif

#ifndef string_h
#define string_h
#include <string>
#endif

using namespace std;

namespace doctorj
{
    class AstCompilationUnit;
    
    /**
     * Does all the hard work. Encapsulates the common functionality between the
     * parser and the scanner.
     */
    class Parser
    {
    public:
        /**
         * Sets up the parser.
         */
        Parser();

        /**
         * Does not delete any objects.
         */
        virtual ~Parser();

        /**
         * Sets the expected Java version of the source, such as 1.3, 1.4, etc.
         */
        void setJavaVersion(double version);

        /**
         * Returns the expected Java version of the source, such as 1.3, 1.4,
         * etc.
         */
        double javaVersion() const;
        
        /**
         * The file being operated on.
         */
        File* currentFile() const;

        /**
         * The current character position in the line.
         */
        char* position() const; 

        /**
         * Sets the position.
         */
        void setPosition(char* pos);

        /**
         * Adds to the the position.
         */
        void incrementPosition(int pos);

        /**
         * Puts a caret at the current position and writes the token type.
         */
        void point(const string& tokenType) const;

        /**
         * Massive debugging output.
         */
        void display(const string& tokenType) const;

        /**
         * Adds the current noncode to the given newly created leaf and resets the
         * relevant counters.
         */
        void reset(AstLeaf* const leaf, int len);

        /**
         * Parses the file, and passes back the resulting compilation unit.
         */
        void parse(AstCompilationUnit*& compUnit, const string& fileName);

        /**
         * Parses the file, and passes back the resulting compilation unit.
         */
        void parse(AstCompilationUnit*& compUnit, File* const file);

        /**
         * Reports the error.
         */
        void reportError(const string& message);

        /**
         * Reports the warning.
         */
        void reportWarning(const string& message);

        /**
         * Reports the error.
         */
        void report(const string& errType, const string& message);

        /**
         * Sets the tab width (usually 8).
         */
        void setTabWidth(int tabWidth);

        /**
         * Returns the tab width.
         */
        int tabWidth() const;

    protected:

        /**
         * Creates and returns the current noncode.
         */
        AstNoncode* makeNoncode() const;

    private:
        /**
         * The file currently being parsed.
         */
        File* currentFile_;

        char* lastWsPosition_;

        char* position_;

        int tabWidth_;

        vector<File*> files_;

        double javaVersion_;

        /**
         * Whether the current file had an error during parsing.
         */
        bool hadError_;

    };
}

#endif //! Parser_h
