#ifndef StatisticsAnalyzer_h
#include "StatisticsAnalyzer.h"
#endif

#ifndef AST_h
#include "AST.h"
#endif

#ifndef File_h
#include "File.h"
#endif

#ifndef Java_h
#include "Java.h"
#endif

#ifndef InterfaceStatistics_h
#include "InterfaceStatistics.h"
#endif

#ifndef AbstractClassStatistics_h
#include "AbstractClassStatistics.h"
#endif

#ifndef CompilationUnitStatistics_h
#include "CompilationUnitStatistics.h"
#endif

#ifndef ConcreteClassStatistics_h
#include "ConcreteClassStatistics.h"
#endif

#ifndef ProjectStatistics_h
#include "ProjectStatistics.h"
#endif

using namespace doctorj;

StatisticsAnalyzer::StatisticsAnalyzer() : typeStats_(NULL), cuStats_(NULL)
{
    projectStats_ = new ProjectStatistics();
}

StatisticsAnalyzer::~StatisticsAnalyzer()
{
    delete projectStats_;
}

ProjectStatistics* StatisticsAnalyzer::getProjectStatistics()
{
    return projectStats_;
}

void StatisticsAnalyzer::process(AstClassDeclaration* const cd,
                                 AstModifierList* const modifiers,
                                 AstName* const extended,
                                 AstImplementsInterfaceList* const implemented)
{
    InnerTypeStatistics* outerStats = typeStats_;
    
    string cname  = cd->getIdentifier()->text();
    int    nLines = JavaItem::lineCount(cd);
    if (modifiers && JavaModifierList::getAbstractModifier(modifiers)) {
        typeStats_ = new AbstractClassStatistics(cname, nLines);
    }
    else {
        typeStats_ = new ConcreteClassStatistics(cname, nLines);
    }

    if (outerStats) {
        outerStats->addInnerType(typeStats_);
    }
    else {
        cuStats_->addType(typeStats_);
    }

    traverse(cd);

    typeStats_ = outerStats;
}

void StatisticsAnalyzer::process(AstClassDeclarationModsBaseless* const c)
{
    process(c, c->getModifierList(), NULL, NULL);
}

void StatisticsAnalyzer::process(AstClassDeclarationModsExtends* const c)
{
    process(c, c->getModifierList(), c->getName(), NULL);
}

void StatisticsAnalyzer::process(AstClassDeclarationModsExtendsImplements* const c)
{
    process(c, c->getModifierList(), c->getName(), c->getImplementsInterfaceList());
}

void StatisticsAnalyzer::process(AstClassDeclarationModsImplements* const c)
{
    process(c, c->getModifierList(), NULL, c->getImplementsInterfaceList());
}

void StatisticsAnalyzer::process(AstClassDeclarationNoModsBaseless* const c)
{
    process(c, NULL, NULL, NULL);
}

void StatisticsAnalyzer::process(AstClassDeclarationNoModsExtends* const c)
{
    process(c, NULL, c->getName(), NULL);
}

void StatisticsAnalyzer::process(AstClassDeclarationNoModsExtendsImplements* const c)
{
    process(c, NULL, c->getName(), c->getImplementsInterfaceList());
}

void StatisticsAnalyzer::process(AstClassDeclarationNoModsImplements* const c)
{
    process(c, NULL, NULL, c->getImplementsInterfaceList());
}

void StatisticsAnalyzer::process(AstClassMemberDeclaration* const cmd,
                                 AstModifierList* const modifiers,
                                 AstName* const extended,
                                 AstImplementsInterfaceList* const implemented)
{
    InnerTypeStatistics* outerStats = typeStats_;
    
    string cname  = cmd->getIdentifier()->text();
    int    nLines = JavaItem::lineCount(cmd);
    if (modifiers && JavaModifierList::getAbstractModifier(modifiers)) {
        typeStats_ = new AbstractClassStatistics(cname, nLines);
    }
    else {
        typeStats_ = new ConcreteClassStatistics(cname, nLines);
    }

    outerStats->addInnerType(typeStats_);

    traverse(cmd);

    typeStats_ = outerStats;
}

void StatisticsAnalyzer::process(AstClassMemberDeclarationModsExtendsImplements* const c)
{
    process(c, c->getModifierList(), c->getName(), c->getImplementsInterfaceList());
}

void StatisticsAnalyzer::process(AstClassMemberDeclarationModsExtendsNoImplements* const c)
{
    process(c, c->getModifierList(), c->getName(), NULL);
}

void StatisticsAnalyzer::process(AstClassMemberDeclarationModsNoExtendsImplements* const c)
{
    process(c, c->getModifierList(), NULL, c->getImplementsInterfaceList());
}

void StatisticsAnalyzer::process(AstClassMemberDeclarationModsNoExtendsNoImplements* const c)
{
    process(c, c->getModifierList(), NULL, NULL);
}

void StatisticsAnalyzer::process(AstClassMemberDeclarationNoModsExtendsImplements* const c)
{
    process(c, NULL, c->getName(), c->getImplementsInterfaceList());
}

void StatisticsAnalyzer::process(AstClassMemberDeclarationNoModsExtendsNoImplements* const c)
{
    process(c, NULL, c->getName(), NULL);
}

void StatisticsAnalyzer::process(AstClassMemberDeclarationNoModsNoExtendsImplements* const c)
{
    process(c, NULL, NULL, c->getImplementsInterfaceList());
}

void StatisticsAnalyzer::process(AstClassMemberDeclarationNoModsNoExtendsNoImplements* const c)
{
    process(c, NULL, NULL, NULL);
}

void StatisticsAnalyzer::process(AstCompilationUnit* const cu,
                                 AstTypeDeclarationList* const types,
                                 AstImportDeclarationList* const imports)
{
    AstItem* head = cu->leadingNoncode();
    if (!head) {
        head = cu;
    }

    AstItem* tail = cu->trailingNoncode();
    if (!tail) {
        tail = cu;
    }

    int nLines   = JavaItem::lineCount(head, tail);
    int nImports = imports ? imports->getImportDeclarationCount() : 0;

    cuStats_ = new CompilationUnitStatistics(cu->sourceFile()->name(), nLines, nImports);
    projectStats_->addCompilationUnit(cuStats_);

    traverse(cu);
}

void StatisticsAnalyzer::process(AstCompilationUnitEmpty* const cue)
{
    process(cue, NULL, NULL);
}

void StatisticsAnalyzer::process(AstCompilationUnitImp* const cui)
{
    process(cui, NULL, cui->getImportDeclarationList());
}

void StatisticsAnalyzer::process(AstCompilationUnitImpTypes* const cuit)
{
    process(cuit, cuit->getTypeDeclarationList(), cuit->getImportDeclarationList());
}

void StatisticsAnalyzer::process(AstCompilationUnitPkg* const cup)
{
    process(cup, NULL, NULL);
}

void StatisticsAnalyzer::process(AstCompilationUnitPkgImp* const cupi)
{
    process(cupi, NULL, cupi->getImportDeclarationList());
}

void StatisticsAnalyzer::process(AstCompilationUnitPkgImpTypes* const cupit)
{
    process(cupit, cupit->getTypeDeclarationList(), cupit->getImportDeclarationList());
}

void StatisticsAnalyzer::process(AstCompilationUnitPkgTypes* const cupt)
{
    process(cupt, cupt->getTypeDeclarationList(), NULL);
}

void StatisticsAnalyzer::process(AstCompilationUnitTypes* const cut)
{
    process(cut, cut->getTypeDeclarationList(), NULL);
}

void StatisticsAnalyzer::process(AstFieldDeclaration* const fd,
                                 AstModifierList* const modifiers)
{
    AstVariableDeclaratorList* vars = fd->getVariableDeclaratorList();
    int count = vars->getVariableDeclaratorCount();
    if (modifiers && JavaModifierList::getStaticModifier(modifiers)) {
        typeStats_->addStaticFields(count);
    }
    else {
        typeStats_->addInstanceFields(count);
    }
    traverse(fd);
}

void StatisticsAnalyzer::process(AstFieldDeclarationMods* const fdm)
{
    process(fdm, fdm->getModifierList());
}

void StatisticsAnalyzer::process(AstFieldDeclarationNoMods* const fdnm)
{
    process(fdnm, NULL);
}

void StatisticsAnalyzer::process(AstInterfaceDeclaration* const id,
                                 AstModifierList* const modifiers,
                                 AstExtendsInterfaceList* const extended)
{
    InnerTypeStatistics* outerStats = typeStats_;

    string iname  = id->getIdentifier()->text();
    int    nLines = JavaItem::lineCount(id);

    typeStats_ = new InterfaceStatistics(iname, nLines);

    if (outerStats) {
        outerStats->addInnerType(typeStats_);
    }
    else {
        cuStats_->addType(typeStats_);
    }

    traverse(id);

    typeStats_ = outerStats;
}

void StatisticsAnalyzer::process(AstInterfaceDeclarationModsExtends* const idme)
{
    process(idme, idme->getModifierList(), idme->getExtendsInterfaceList());
}

void StatisticsAnalyzer::process(AstInterfaceDeclarationModsNoExtends* const idmne)
{
    process(idmne, idmne->getModifierList(), NULL);
}

void StatisticsAnalyzer::process(AstInterfaceDeclarationNoModsExtends* const idnme)
{
    process(idnme, NULL, idnme->getExtendsInterfaceList());
}

void StatisticsAnalyzer::process(AstInterfaceDeclarationNoModsNoExtends* const idnmne)
{
    process(idnmne, NULL, NULL);
}

void StatisticsAnalyzer::process(AstMethodDeclaration* const md)
{
    JavaModifierList mods(md);
    if (mods.containsStatic()) {
        typeStats_->addStaticMethod(JavaItem::lineCount(md));
    }
    else {
        typeStats_->addInstanceMethod(JavaItem::lineCount(md));
    }
    traverse(md);
}

void StatisticsAnalyzer::process(AstMethodDeclarationBlock* const mdb)
{
    process((AstMethodDeclaration*)mdb);
}

void StatisticsAnalyzer::process(AstMethodDeclarationSemicolon* const mds)
{
    process((AstMethodDeclaration*)mds);
}
