#ifndef StatisticsReporterXML_h
#include "StatisticsReporterXML.h"
#endif

#ifndef ProjectStatistics_h
#include "ProjectStatistics.h"
#endif

#ifndef CompilationUnitStatistics_h
#include "CompilationUnitStatistics.h"
#endif

using namespace std;
using namespace doctorj;


XMLNode::XMLNode(const string& label, StatisticsReporterXML* const reporter) :
        label_(label), reporter_(reporter)
{
    reporter_->writeLabel(label);
    reporter_->incrementLevel();
}

XMLNode::~XMLNode()
{
    reporter_->decrementLevel();
    reporter_->writeLabel("/" + label_);
}

StatisticsReporterXML::StatisticsReporterXML(ostream& os) : StatisticsReporter(os)
{
    os_ << "<?xml version=\"1.0\"?>" << endl;
}

StatisticsReporterXML::~StatisticsReporterXML()
{
}

void StatisticsReporterXML::writeLabel(const string& label)
{
    indent();
    os_ << "<" << label << ">" << endl;
}

void StatisticsReporterXML::writeString(const string& label, const string& str)
{
    indent();
    os_ << "<" << label << ">" << str << "</" << label << ">" << endl;
}

void StatisticsReporterXML::writeTotalLines(int nLines)
{
    write("TotalLines", nLines);
}

void StatisticsReporterXML::writeImportCount(int nImports)
{
    write("Imports", nImports);
}

void StatisticsReporterXML::writeTypesCount(int nConcreteClasses, 
                                            int nAbstractClasses, int nInterfaces,
                                            int sum, double mean, double standardDeviation, 
                                            int max, int min)
{
    int nTypes = nConcreteClasses + nAbstractClasses + nInterfaces;

    {
        XMLNode itc("InnerTypeCounts", this);
        write("Total",           nTypes);
        write("ConcreteClasses", nConcreteClasses);
        write("AbstractClasses", nAbstractClasses);
        write("Interfaces",      nInterfaces);
        
        {
            XMLNode lines("Lines", this);
            write("Sum",               sum);
            write("Mean",              mean);
            write("StandardDeviation", standardDeviation);
            write("Highest",           max);
            write("Lowest",            min);
        }

    }
    blankLine();
}


void StatisticsReporterXML::writeMethodCount(int nStaticMethods, int nInstanceMethods,
                                             int sum, 
                                             double mean, double standardDeviation,
                                             int max, int min)
{
    int nMethods = nStaticMethods + nInstanceMethods;

    {
        XMLNode methods("Methods", this);
        write("Total", nMethods);
        write("Static",    nStaticMethods);
        write("Instance",  nInstanceMethods);
        
        {
            XMLNode lines("Lines", this);
            write("Sum",               sum);
            write("Mean",              mean);
            write("StandardDeviation", standardDeviation);
            write("Highest",           max);
            write("Lowest",            min);
        }   
    }
    blankLine();
}

void StatisticsReporterXML::outputSummary(int nLines,
                                          int nTypes,
                                          int nConcretes,
                                          int nAbstracts,
                                          int nInterfaces,
                                          const vector<int>& typeLines,
                                          int nStaticMethods,
                                          int nInstanceMethods,
                                          const vector<int>& methodLines,
                                          int nStaticFields,
                                          int nInstanceFields)
{
    XMLNode summary("Summary", this);

    write("FileCount",  nTypes);
    write("TotalLines", nLines);
    blankLine();

    {
        XMLNode innerTypes("InnerTypeCounts", this);
        write("Total",           nConcretes + nAbstracts + nInterfaces);
        write("ConcreteClasses", nConcretes);
        write("AbstractClasses", nAbstracts);
        write("Interfaces",      nInterfaces);
        writeStats(typeLines);
    }
    blankLine();

    {
        XMLNode methods("Methods", this);
        write("Total",    nStaticMethods + nInstanceMethods);
        write("Static",   nStaticMethods);
        write("Instance", nInstanceMethods);
        writeStats(methodLines);
    }
    blankLine();

    {
        XMLNode fields("Fields", this);
        write("Total",    nStaticFields + nInstanceFields);
        write("Static",   nStaticFields);
        write("Instance", nInstanceFields);
    }
    blankLine();
}

void StatisticsReporterXML::outputInnerType(const string& type, 
                                            const string& name,
                                            const vector<InnerTypeStatistics*>& innerTypes,
                                            int nLines,
                                            int nStaticMethods,
                                            int nInstanceMethods,
                                            const vector<int>& methods,
                                            int nStaticFields,
                                            int nInstanceFields)
{
    incrementLevel();
    string ts = type;
    ts[0] = toupper(ts[0]);     // class => Class
    string label = ts + " name=\"" + name + "\"";
    writeLabel(label);
    incrementLevel();

    if (innerTypes.size() > 0) {
        XMLNode inner("InnerTypes", this);
        vector<InnerTypeStatistics*>::const_iterator tsit   = innerTypes.begin();
        vector<InnerTypeStatistics*>::const_iterator tsstop = innerTypes.end();
        while (tsit != tsstop) {
            InnerTypeStatistics* type = *tsit;
            writeInnerType(type);
            ++tsit;
        }
    }

    write("TotalLines", nLines);
    blankLine();

    {
        XMLNode meth("Methods", this);
        write("Total",    nStaticMethods + nInstanceMethods);
        write("Static",   nStaticMethods);
        write("Instance", nInstanceMethods);
        writeStats(methods);
    }
    blankLine();

    int nFields = nStaticFields + nInstanceFields;

    {
        XMLNode fields("Fields", this);
        write("Total",    nFields);
        write("Static",   nStaticFields);
        write("Instance", nInstanceFields);
    }        
    blankLine();

    decrementLevel();

    writeLabel("/" + ts);
    decrementLevel();
}

void StatisticsReporterXML::writeFullReport(ProjectStatistics* const ps)
{
    XMLNode top("DoctorJ", this);
    {
        XMLNode proj("Project", this);

        const vector<CompilationUnitStatistics*>& cus = ps->getCompilationUnitsStatistics();
        vector<CompilationUnitStatistics*>::const_iterator cit   = cus.begin();
        vector<CompilationUnitStatistics*>::const_iterator cstop = cus.end();
        while (cit != cstop) {
            const CompilationUnitStatistics* cus = *cit;
        
            string flabel = "File name=\"" + cus->name() + "\"";
            writeLabel(flabel);
            incrementLevel();
            write("Imports", cus->getImportCount());

            writeLabel("Types");
            blankLine();

            const vector<InnerTypeStatistics*>& types = cus->types();

            vector<InnerTypeStatistics*>::const_iterator tsit   = types.begin();
            vector<InnerTypeStatistics*>::const_iterator tsstop = types.end();
            while (tsit != tsstop) {
                InnerTypeStatistics* type = *tsit;
                writeInnerType(type);
                ++tsit;
            }
        
            blankLine();
            ++cit;

            writeLabel("/Types");

            decrementLevel();
            writeLabel("/File");
            blankLine();
        }
    
        if (cus.size() > 1) {
            writeSummary(ps);
        }

    }
}
