#ifndef StringUtilities_h
#define StringUtilities_h

#ifndef std_string
#define std_string
#include <string>
#endif

#ifdef HAVE_SSTREAM
    #ifndef std_sstream
    #define std_sstream
    #include <sstream>
    #endif
#else
    #ifndef std_strstream
    #define std_strstream
    #include <strstream>
    #endif
#endif

#ifndef doctorj_iostream
#define doctorj_iostream
#include <iostream>
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif


using namespace std;

namespace doctorj
{
    namespace StringUtilities
    {

        /**
         * Joins the array of parts, using the delimiter.
         */
        string join(const string& delim, string parts[], int nparts);

        /**
         * A more flexible version of the above, showing that I've been reading
         * some STL books.
         */
        template <typename InputIterator>
        string join(const string& delim, InputIterator sourceBegin, InputIterator sourceEnd);

        /**
         * Returns v as a string.
         */
        template <class T>
        static string toString(T v);

        /**
         * Passes back v converted from a string.
         */
        void fromString(const string& str, bool& v);
        void fromString(const string& str, char& v);
        void fromString(const string& str, char* v);
        void fromString(const string& str, double& v);
        void fromString(const string& str, float& v);
        void fromString(const string& str, int& v);
        void fromString(const string& str, long double& v);
        void fromString(const string& str, long& v);
        void fromString(const string& str, short& v);
        void fromString(const string& str, signed char& v);
        void fromString(const string& str, signed char* v);
        void fromString(const string& str, unsigned char& v);
        void fromString(const string& str, unsigned char* v);
        void fromString(const string& str, unsigned int& v);
        void fromString(const string& str, unsigned long& v);
        void fromString(const string& str, unsigned short& v);
        void fromString(const string& str, string& str);

        /**
         * Returns a copy of the string as lowercase.
         */
        string toLower(const string& str);

        /**
         * Returns a copy of the string as uppercase.
         */
        string toUpper(const string& str);

        /**
         * Same as str =~ s/rhs/lhs/g.
         */
        void subst(string* const str, const string& lhs, const string& rhs);

        /**
         * Removes the leading whitespace.
         */
        void trimFront(string* const str);

        /**
         * Removes the trailing whitespace.
         */
        void trimRear(string* const str);

        /**
         * Removes the trailing and leading whitespace.
         */
        void trim(string* const str);

        /**
         * Returns the number of occurances of the given character in the
         * string.
         */
        int count(const string& str, char ch);

        /**
         * Returns the number of words, i.e., sequences of characters with
         * whitespace between them.
         */
        int countWords(const string& str);

        /**
         * Splits on the given delimiter string.
         */
        vector<string> split(const string& str, const string& del, int limit = -1);

        /**
         * Splits the given string on whitespace, and puts the words into the
         * given vector.
         */
        void getWords(const string& str, vector<string>* const words);

        /**
         * Returns whether s starts with t.
         */
        bool startsWith(const string& s, const string& t);

        /**
         * Returns whether s ends with t.
         */
        bool endsWith(const string& s, const string& t);

    }
    
}

template <class T>
string doctorj::StringUtilities::toString(T v)
{
#ifdef HAVE_SSTREAM
    ostringstream oss;
    oss << v;
    string sstr = oss.str();
#else
    ostrstream oss;
    oss << v << ends;
    string sstr = oss.str();

    // free the char* returned
    oss.freeze(false);
#endif

    return sstr;
}

template <typename InputIterator>
string doctorj::StringUtilities::join(const string& delim, InputIterator sourceBegin, InputIterator sourceEnd)
{
    string joined = *sourceBegin;
    for (InputIterator it = sourceBegin + 1; it != sourceEnd; ++it) {
        joined += delim + *it;
    }
    return joined;
}


#endif //! StringUtilities_h
