// --- BEGIN COPYRIGHT BLOCK ---
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; version 2 of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
//
// (C) 2007 Red Hat, Inc.
// All rights reserved.
// --- END COPYRIGHT BLOCK ---
package com.netscape.cms.servlet.base;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Locale;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang3.StringEscapeUtils;
import org.dogtagpki.server.authentication.AuthManager;
import org.dogtagpki.server.authorization.AuthzToken;
import org.dogtagpki.server.ca.ICertificateAuthority;
import org.mozilla.jss.netscape.security.pkcs.ContentInfo;
import org.mozilla.jss.netscape.security.pkcs.PKCS7;
import org.mozilla.jss.netscape.security.pkcs.SignerInfo;
import org.mozilla.jss.netscape.security.util.Utils;
import org.mozilla.jss.netscape.security.x509.AlgorithmId;
import org.mozilla.jss.netscape.security.x509.CRLExtensions;
import org.mozilla.jss.netscape.security.x509.CRLReasonExtension;
import org.mozilla.jss.netscape.security.x509.CertificateChain;
import org.mozilla.jss.netscape.security.x509.RevocationReason;
import org.mozilla.jss.netscape.security.x509.RevokedCertImpl;
import org.mozilla.jss.netscape.security.x509.X509CertImpl;
import org.w3c.dom.Node;

import com.netscape.certsrv.authentication.IAuthToken;
import com.netscape.certsrv.authority.IAuthority;
import com.netscape.certsrv.base.EBaseException;
import com.netscape.certsrv.base.IArgBlock;
import com.netscape.certsrv.base.SessionContext;
import com.netscape.certsrv.kra.IKeyRecoveryAuthority;
import com.netscape.certsrv.logging.ILogger;
import com.netscape.certsrv.logging.LogEvent;
import com.netscape.certsrv.logging.LogSource;
import com.netscape.certsrv.logging.event.AuthEvent;
import com.netscape.certsrv.logging.event.AuthzEvent;
import com.netscape.certsrv.logging.event.RoleAssumeEvent;
import com.netscape.cms.logging.Logger;
import com.netscape.cms.logging.SignedAuditLogger;
import com.netscape.cms.servlet.common.AuthCredentials;
import com.netscape.cms.servlet.common.CMSFileLoader;
import com.netscape.cms.servlet.common.CMSGateway;
import com.netscape.cms.servlet.common.CMSLoadTemplate;
import com.netscape.cms.servlet.common.CMSRequest;
import com.netscape.cms.servlet.common.CMSTemplate;
import com.netscape.cms.servlet.common.CMSTemplateParams;
import com.netscape.cms.servlet.common.ECMSGWException;
import com.netscape.cms.servlet.common.GenErrorTemplateFiller;
import com.netscape.cms.servlet.common.GenPendingTemplateFiller;
import com.netscape.cms.servlet.common.GenRejectedTemplateFiller;
import com.netscape.cms.servlet.common.GenSuccessTemplateFiller;
import com.netscape.cms.servlet.common.GenSvcPendingTemplateFiller;
import com.netscape.cms.servlet.common.GenUnexpectedErrorTemplateFiller;
import com.netscape.cms.servlet.common.ICMSTemplateFiller;
import com.netscape.cms.servlet.common.ServletUtils;
import com.netscape.cmscore.apps.CMS;
import com.netscape.cmscore.apps.CMSEngine;
import com.netscape.cmscore.apps.CommandQueue;
import com.netscape.cmscore.apps.EngineConfig;
import com.netscape.cmscore.authorization.AuthzSubsystem;
import com.netscape.cmscore.base.ArgBlock;
import com.netscape.cmscore.base.ConfigStore;
import com.netscape.cmscore.dbs.CertRecord;
import com.netscape.cmscore.dbs.CertificateRepository;
import com.netscape.cmscore.request.Request;
import com.netscape.cmscore.request.RequestQueue;
import com.netscape.cmscore.request.RequestRepository;
import com.netscape.cmscore.security.JssSubsystem;
import com.netscape.cmscore.usrgrp.Group;
import com.netscape.cmscore.usrgrp.UGSubsystem;
import com.netscape.cmsutil.xml.XMLObject;

/**
 * This is the base class of all CS servlet.
 *
 * @version $Revision$, $Date$
 */
public abstract class CMSServlet extends HttpServlet {

    public static org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(CMSServlet.class);
    protected static Logger signedAuditLogger = SignedAuditLogger.getLogger();

    private static final long serialVersionUID = -3886300199374147160L;
    // servlet init params
    // xxxx todo:Should enforce init param value checking!
    public final static String SUCCESS = "0";
    public final static String FAILURE = "1";
    public final static String AUTH_FAILURE = "2";

    public final static String PROP_ID = "ID";
    public final static String PROP_AUTHORITY = "authority";
    public final static String PROP_AUTHORITYID = "authorityId";
    public final static String PROP_AUTHMGR = "AuthMgr";
    public final static String PROP_CLIENTAUTH = "GetClientCert";
    public final static String PROP_RESOURCEID = "resourceID";

    public final static String AUTHZ_SRC_LDAP = "ldap";
    public final static String AUTHZ_SRC_TYPE = "sourceType";

    public final static String AUTHZ_SRC_XML = "web.xml";
    public final static String PROP_AUTHZ_MGR = "AuthzMgr";
    public final static String PROP_ACL = "ACLinfo";
    public final static String AUTHZ_MGR_BASIC = "BasicAclAuthz";
    public final static String AUTHZ_MGR_LDAP = "DirAclAuthz";
    private final static String HDR_LANG = "accept-language";

    // final error message - if error and exception templates don't work
    // send out this text string directly to output.

    public final static String PROP_FINAL_ERROR_MSG = "finalErrorMsg";
    public final static String ERROR_MSG_TOKEN = "$ERROR_MSG";
    public final static String FINAL_ERROR_MSG =
            "<HTML>\n" +
                    "<BODY BGCOLOR=white>\n" +
                    "<P>\n" +
                    "The Certificate System has encountered " +
                    "an unrecoverable error.\n" +
                    "<P>\n" +
                    "Error Message:<BR>\n" +
                    "<I>$ERROR_MSG</I>\n" +
                    "<P>\n" +
                    "Please contact your local administrator for assistance.\n" +
                    "</BODY>\n" +
                    "</HTML>\n";

    // properties from configuration.

    protected final static String PROP_UNAUTHORIZED_TEMPLATE = "unauthorizedTemplate";
    protected final static String UNAUTHORIZED_TEMPLATE = "/GenUnauthorized.template";
    protected final static String PROP_SUCCESS_TEMPLATE = "successTemplate";
    protected final static String SUCCESS_TEMPLATE = "/GenSuccess.template";
    protected final static String PROP_PENDING_TEMPLATE = "pendingTemplate";
    protected final static String PENDING_TEMPLATE = "/GenPending.template";
    protected final static String PROP_SVC_PENDING_TEMPLATE = "svcpendingTemplate";
    protected final static String SVC_PENDING_TEMPLATE = "/GenSvcPending.template";
    protected final static String PROP_REJECTED_TEMPLATE = "rejectedTemplate";
    protected final static String REJECTED_TEMPLATE = "/GenRejected.template";
    protected final static String PROP_ERROR_TEMPLATE = "errorTemplate";
    protected final static String ERROR_TEMPLATE = "/GenError.template";
    protected final static String PROP_EXCEPTION_TEMPLATE = "unexpectedErrorTemplate";
    protected final static String EXCEPTION_TEMPLATE = "/GenUnexpectedError.template";

    private final static String PROP_UNAUTHOR_TEMPLATE_FILLER = "unauthorizedTemplateFiller";
    protected final static String PROP_SUCCESS_TEMPLATE_FILLER = "successTemplateFiller";
    private final static String PROP_ERROR_TEMPLATE_FILLER = "errorTemplateFiller";
    private final static String PROP_PENDING_TEMPLATE_FILLER = "pendingTemplateFiller";
    private final static String PROP_SVC_PENDING_TEMPLATE_FILLER = "svcpendingTemplateFiller";
    private final static String PROP_REJECTED_TEMPLATE_FILLER = "rejectedTemplateFiller";
    private final static String PROP_EXCEPTION_TEMPLATE_FILLER = "exceptionTemplateFiller";

    protected final static String RA_AGENT_GROUP = "Registration Manager Agents";
    protected final static String CA_AGENT_GROUP = "Certificate Manager Agents";
    protected final static String KRA_AGENT_GROUP = "Data Recovery Manager Agents";
    protected final static String OCSP_AGENT_GROUP = "Online Certificate Status Manager Agents";
    protected final static String TRUSTED_RA_GROUP = "Trusted Managers";
    protected final static String ADMIN_GROUP = "Administrators";

    // default http params NOT to save in request.(config values added to list )
    private static final String PROP_DONT_SAVE_HTTP_PARAMS = "dontSaveHttpParams";
    private static final String[] DONT_SAVE_HTTP_PARAMS = { "pwd", "password", "passwd",
            "challengePassword", "confirmChallengePassword" };

    // default http headers to save in request. (config values added to list)
    private static final String PROP_SAVE_HTTP_HEADERS = "saveHttpHeaders";
    private static final String[] SAVE_HTTP_HEADERS = { "accept-language", "user-agent", };

    // request prefixes to distinguish from other request attributes.
    public static final String PFX_HTTP_HEADER = "HTTP_HEADER";
    public static final String PFX_HTTP_PARAM = "HTTP_PARAM";
    public static final String PFX_AUTH_TOKEN = "AUTH_TOKEN";

    /* input http params */
    protected final static String AUTHMGR_PARAM = "authenticator";

    public static final String CERT_ATTR =
            "javax.servlet.request.X509Certificate";

    // members.

    protected ServletConfig servletConfig;

    protected boolean mRenderResult = true;
    protected String mFinalErrorMsg = FINAL_ERROR_MSG;
    protected Hashtable<Integer, CMSLoadTemplate> mTemplates = new Hashtable<>();

    protected ServletConfig mServletConfig = null;
    protected ServletContext mServletContext = null;
    private CMSFileLoader mFileLoader = null;

    protected Vector<String> mDontSaveHttpParams = new Vector<>();
    protected Vector<String> mSaveHttpHeaders = new Vector<>();

    protected String mId = null;
    protected ConfigStore mConfig;

    // the authority, RA, CA, KRA this servlet is serving.
    protected IAuthority mAuthority = null;
    protected ICertificateAuthority certAuthority;
    protected RequestRepository requestRepository;
    protected RequestQueue mRequestQueue;

    // system logger.
    protected LogSource mLogCategory = ILogger.S_OTHER;
    private MessageDigest mSHADigest = null;

    protected String mGetClientCert = "false";
    protected String mAuthMgr = null;
    protected AuthzSubsystem mAuthz;

    protected String mAclMethod = null;
    protected String mAuthzResourceName = null;

    protected String mOutputTemplatePath = null;
    protected CMSEngine engine = CMS.getCMSEngine();
    private UGSubsystem mUG = engine.getUGSubsystem();

    public CMSServlet() {
    }

    public static Hashtable<String, String> toHashtable(HttpServletRequest req) {
        Hashtable<String, String> httpReqHash = new Hashtable<>();
        Enumeration<?> names = req.getParameterNames();

        while (names.hasMoreElements()) {
            String name = (String) names.nextElement();

            httpReqHash.put(name, req.getParameter(name));
        }
        return httpReqHash;
    }

    @Override
    public void init(ServletConfig sc) throws ServletException {
        super.init(sc);

        this.servletConfig = sc;

        CMSEngine engine = CMS.getCMSEngine();
        EngineConfig cs = engine.getConfig();

        mAuthz = engine.getAuthzSubsystem();
        mId = sc.getInitParameter(PROP_ID);

        try {
            mAclMethod = ServletUtils.initializeAuthz(sc, mAuthz, mId);
        } catch (ServletException e) {
            logger.error("CMSServlet: " + e.getMessage(), e);
            throw e;
        }

        mConfig = cs.getSubStore(CMSGateway.PROP_CMSGATEWAY, ConfigStore.class);
        mServletConfig = sc;
        mServletContext = sc.getServletContext();
        mFileLoader = new CMSFileLoader();

        mGetClientCert = sc.getInitParameter(PROP_CLIENTAUTH);
        mAuthMgr = sc.getInitParameter(PROP_AUTHMGR);
        mAuthzResourceName = sc.getInitParameter(PROP_RESOURCEID);
        mOutputTemplatePath = sc.getInitParameter("templatePath");

        String authority = sc.getInitParameter(PROP_AUTHORITY);
        if (authority == null) {
            authority = sc.getInitParameter(PROP_AUTHORITYID);
        }

        if (authority != null) {
            mAuthority = (IAuthority) engine.getSubsystem(authority);
            if (mAuthority instanceof ICertificateAuthority)
                certAuthority = (ICertificateAuthority) mAuthority;
        }

        requestRepository = engine.getRequestRepository();

        if (mAuthority != null)
            mRequestQueue = engine.getRequestQueue();

        // set default templates.
        setDefaultTemplates(sc);

        // for logging to the right authority category.
        if (mAuthority == null) {
            mLogCategory = ILogger.S_OTHER;
        } else {
            if (mAuthority instanceof ICertificateAuthority)
                mLogCategory = ILogger.S_CA;
            else if (mAuthority instanceof IKeyRecoveryAuthority)
                mLogCategory = ILogger.S_KRA;
            else
                mLogCategory = ILogger.S_OTHER;
        }

        try {
            // get final error message.
            // used when templates can't even be loaded.
            String eMsg =
                    sc.getInitParameter(PROP_FINAL_ERROR_MSG);

            if (eMsg != null)
                mFinalErrorMsg = eMsg;

            // get any configured templates.
            Enumeration<CMSLoadTemplate> templs = mTemplates.elements();

            while (templs.hasMoreElements()) {
                CMSLoadTemplate templ = templs.nextElement();

                if (templ == null || templ.mPropName == null) {
                    continue;
                }
                String tName =
                        sc.getInitParameter(templ.mPropName);

                if (tName != null)
                    templ.mTemplateName = tName;
                String fillerName =
                        sc.getInitParameter(templ.mFillerPropName);

                if (fillerName != null) {
                    ICMSTemplateFiller filler = newFillerObject(fillerName);

                    if (filler != null)
                        templ.mFiller = filler;
                }
            }

            // get http params NOT to store in a Request and
            // get http headers TO store in a Request.
            getDontSaveHttpParams(sc);
            getSaveHttpHeaders(sc);
        } catch (Exception e) {
            logger.error(CMS.getLogMessage("CMSGW_ERR_CONF_TEMP_PARAMS", e.toString()), e);
            throw new ServletException(e);
        }

        try {
            mSHADigest = MessageDigest.getInstance("SHA-256");
        } catch (NoSuchAlgorithmException e) {
            logger.error(CMS.getLogMessage("CMSGW_ERR_CONF_TEMP_PARAMS", e.toString()), e);
            throw new ServletException(e);
        }
    }

    public String getId() {
        return mId;
    }

    public String getAuthMgr() {
        return mAuthMgr;
    }

    public boolean isClientCertRequired() {
        return mGetClientCert != null && mGetClientCert.equals("true");
    }

    public void outputHttpParameters(HttpServletRequest httpReq) {
        logger.debug("CMSServlet:service() uri: " + httpReq.getRequestURI());
        Enumeration<?> paramNames = httpReq.getParameterNames();
        while (paramNames.hasMoreElements()) {
            String pn = (String) paramNames.nextElement();
            // added this facility so that password can be hidden,
            // all sensitive parameters should be prefixed with
            // __ (double underscores); however, in the event that
            // a security parameter slips through, we perform multiple
            // additional checks to insure that it is NOT displayed
            String value = CMS.isSensitive(pn) ? "(sensitive)" : httpReq.getParameter(pn);
            logger.debug("CMSServlet::service() param name='{}' value='{}'", pn, value);
        }
    }

    @Override
    public void service(HttpServletRequest httpReq,
            HttpServletResponse httpResp)
            throws ServletException, IOException {

        CMSEngine engine = CMS.getCMSEngine();
        EngineConfig cs = engine.getConfig();

        boolean running_state = engine.isInRunningState();

        if (!running_state) {
            throw new IOException("CS server is not ready to serve.");
        }

        try {
            if (cs.getBoolean("useThreadNaming", false)) {
                String currentName = Thread.currentThread().getName();
                Thread.currentThread().setName(currentName + "-" + httpReq.getServletPath());
            }
        } catch (Exception e) {
        }

        httpReq.setCharacterEncoding("UTF-8");

        if (logger.isDebugEnabled()) {
            outputHttpParameters(httpReq);
        }
        logger.debug("CMSServlet: " + mId + " start to service.");
        /* cfu test
        logger.debug("CMSServlet: content length: " +httpReq.getContentLength());
        Enumeration<String> reqParams = httpReq.getParameterNames();
        while (reqParams.hasMoreElements()) {
            String reqName = reqParams.nextElement();
            String reqValue = httpReq.getParameter(reqName);

            logger.debug("CMSServlet: process() cfu debug: HttpServletRequest "+
               reqName+"="+reqValue);
        }
        */

        // get a cms request
        CMSRequest cmsRequest = newCMSRequest();

        // set argblock
        cmsRequest.setHttpParams(new ArgBlock("http-request-params", toHashtable(httpReq)));

        // set http request
        cmsRequest.setHttpReq(httpReq);

        // set http response
        cmsRequest.setHttpResp(httpResp);

        // set servlet config.
        cmsRequest.setServletConfig(mServletConfig);

        // set servlet context.
        cmsRequest.setServletContext(mServletContext);

        IArgBlock httpArgs = cmsRequest.getHttpParams();

        // authenticator value from http overrides the value in web.xml.
        String authMgr_http = httpArgs.getValueAsString(AUTHMGR_PARAM, null);

        if (authMgr_http != null) {
            mAuthMgr = authMgr_http;
        } else {
            mAuthMgr = mServletConfig.getInitParameter(PROP_AUTHMGR);
        }

        // process request.
        CommandQueue iCommandQueue = new CommandQueue();

        try {
            if (iCommandQueue.registerProcess(cmsRequest, this) == false) {
                cmsRequest.setStatus(CMSRequest.ERROR);
                renderResult(cmsRequest);
                SessionContext.releaseContext();
                return;
            }
            long startTime = new Date().getTime();
            process(cmsRequest);
            renderResult(cmsRequest);
            Date endDate = new Date();
            long endTime = endDate.getTime();
            logger.debug("CMSServlet: curDate: " + endDate + " id: " + mId + " time: " + (endTime - startTime));
            iCommandQueue.unRegisterProccess(cmsRequest, this);

        } catch (EBaseException e) {
            iCommandQueue.unRegisterProccess(cmsRequest, this);
            // ByteArrayOutputStream os = new ByteArrayOutputStream(); for debugging only
            // PrintStream ps = new PrintStream(os);
            //e.printStackTrace(ps);
            log(e.toString());
            renderException(cmsRequest, e);
        } catch (Exception ex) {
            iCommandQueue.unRegisterProccess(cmsRequest, this);
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            PrintStream ps = new PrintStream(os);

            ex.printStackTrace(ps);
            log(os.toString());
            renderFinalError(cmsRequest, ex);
        }

        // destroy SessionContext
        SessionContext.releaseContext();

        return;
    }

    /**
     * Create a new CMSRequest object. This should be overriden by servlets
     * implementing different types of request
     *
     * @return a new CMSRequest object
     */
    protected CMSRequest newCMSRequest() {
        return new CMSRequest();
    }

    /**
     * process an HTTP request. Servlets must override this with their
     * own implementation
     *
     * @throws EBaseException if the servlet was unable to satisfactorily
     *             process the request
     */
    protected void process(CMSRequest cmsRequest)
            throws Exception {
    }

    /**
     * Output a template.
     * If an error occurs while outputing the template the exception template
     * is used to display the error.
     *
     * @param cmsReq the CS request
     */
    protected void renderResult(CMSRequest cmsReq)
            throws IOException {

        if (!mRenderResult)
            return;
        Integer status = cmsReq.getStatus();

        CMSLoadTemplate ltempl = mTemplates.get(status);

        if (ltempl == null || ltempl.mTemplateName == null) {
            // result is previously outputed.
            return;
        }
        ICMSTemplateFiller filler = ltempl.mFiller;

        renderTemplate(cmsReq, ltempl.mTemplateName, filler);
    }

    private static final String PRESERVED = "preserved";
    public static final String TEMPLATE_NAME = "templateName";

    protected void outputArgBlockAsXML(XMLObject xmlObj, Node parent,
                                       String argBlockName, IArgBlock argBlock) {
        Node argBlockContainer = xmlObj.createContainer(parent, argBlockName);

        if (argBlock != null) {
            Enumeration<String> names = argBlock.getElements();
            while (names.hasMoreElements()) {
                String name = names.nextElement();
                String val = argBlock.get(name).toString();
                val = val.trim();
                xmlObj.addItemToContainer(argBlockContainer, name, val);
            }
        }
    }

    protected void outputXML(HttpServletResponse httpResp, CMSTemplateParams params) {
        XMLObject xmlObj = null;
        try {
            xmlObj = new XMLObject();

            Node root = xmlObj.createRoot("xml");
            outputArgBlockAsXML(xmlObj, root, "header", params.getHeader());
            outputArgBlockAsXML(xmlObj, root, "fixed", params.getFixed());

            Enumeration<IArgBlock> records = params.queryRecords();
            Node recordsNode = xmlObj.createContainer(root, "records");
            if (records != null) {
                while (records.hasMoreElements()) {
                    IArgBlock record = records.nextElement();
                    outputArgBlockAsXML(xmlObj, recordsNode, "record", record);
                }
            }

            byte[] cb = xmlObj.toByteArray();
            OutputStream os = httpResp.getOutputStream();
            httpResp.setContentType("application/xml");
            httpResp.setContentLength(cb.length);
            os.write(cb);
            os.flush();
        } catch (Exception e) {
            logger.warn("Failed in outputing XML: " + e.getMessage(), e);
        }
    }

    protected void renderTemplate(
            CMSRequest cmsReq, String templateName, ICMSTemplateFiller filler)
            throws IOException {
        try {
            IArgBlock httpParams = cmsReq.getHttpParams();

            Locale[] locale = new Locale[1];
            CMSTemplate template =
                    getTemplate(templateName, cmsReq.getHttpReq(), locale);
            CMSTemplateParams templateParams = null;

            if (filler != null) {
                templateParams = filler.getTemplateParams(
                            cmsReq, mAuthority, locale[0], null);
            }

            // just output arg blocks as XML
            logger.debug("CMSServlet.java: renderTemplate");
            String xmlOutput = cmsReq.getHttpReq().getParameter("xml");
            if (xmlOutput != null && xmlOutput.equals("true")) {
                logger.debug("CMSServlet.java: xml parameter detected, returning xml");
                outputXML(cmsReq.getHttpResp(), templateParams);
                return;
            }

            if (httpParams != null) {
                String httpTemplateName =
                        httpParams.getValueAsString(
                                TEMPLATE_NAME, null);

                if (httpTemplateName != null) {
                    templateName = httpTemplateName;
                }
            }

            if (templateParams == null)
                templateParams = new CMSTemplateParams(null, null);

            // #359630
            // inject preserved http parameter into the template
            if (httpParams != null) {
                String preserved = httpParams.getValueAsString(
                        PRESERVED, null);

                if (preserved != null) {
                    IArgBlock fixed = templateParams.getFixed();

                    if (fixed != null) {
                        fixed.set(PRESERVED, preserved);
                    }
                }
            }

            ByteArrayOutputStream bos = new ByteArrayOutputStream();

            template.renderOutput(bos, templateParams);
            cmsReq.getHttpResp().setContentType("text/html");
            cmsReq.getHttpResp().setContentLength(bos.size());
            bos.writeTo(cmsReq.getHttpResp().getOutputStream());
        } catch (Exception e) {
            logger.error(CMS.getLogMessage("CMSGW_ERR_OUT_TEMPLATE", templateName, e.toString()), e);
            renderException(cmsReq, new ECMSGWException(CMS.getLogMessage("CMSGW_ERROR_DISPLAY_TEMPLATE"), e));
        }
    }

    /**
     * Output exception (unexpected error) template
     * This is different from other templates in that if an exception occurs
     * while rendering the exception a message is printed out directly.
     * If the message gets an error an IOException is thrown.
     * In others if an exception occurs while rendering the template the
     * exception template (this) is called.
     * <p>
     *
     * @param cmsReq the CS request to pass to template filler if any.
     * @param e the unexpected exception
     */
    protected void renderException(CMSRequest cmsReq, EBaseException e)
            throws IOException {
        try {
            Locale[] locale = new Locale[1];
            CMSLoadTemplate loadTempl =
                    mTemplates.get(CMSRequest.EXCEPTION);
            CMSTemplate template = getTemplate(loadTempl.mTemplateName,
                    cmsReq.getHttpReq(), locale);
            ICMSTemplateFiller filler = loadTempl.mFiller;
            CMSTemplateParams templateParams = null;

            // When an exception occurs the exit is non-local which probably
            // will leave the requestStatus value set to something other
            // than CMSRequest.EXCEPTION, so force the requestStatus to
            // EXCEPTION since it must be that if we're here.
            cmsReq.setStatus(CMSRequest.EXCEPTION);

            if (filler != null) {
                templateParams = filler.getTemplateParams(
                            cmsReq, mAuthority, locale[0], e);
            }
            if (templateParams == null) {
                templateParams = new CMSTemplateParams(null, new ArgBlock());
            }
            if (e != null) {
                templateParams.getFixed().set(
                        ICMSTemplateFiller.EXCEPTION, e.toString(locale[0]));
            }

            // just output arg blocks as XML
            logger.debug("CMSServlet.java: renderTemplate");
            String xmlOutput = cmsReq.getHttpReq().getParameter("xml");
            if (xmlOutput != null && xmlOutput.equals("true")) {
                logger.debug("CMSServlet.java: xml parameter detected, returning xml");
                outputXML(cmsReq.getHttpResp(), templateParams);
                return;
            }

            ByteArrayOutputStream bos = new ByteArrayOutputStream();

            template.renderOutput(bos, templateParams);
            cmsReq.getHttpResp().setContentType("text/html");
            cmsReq.getHttpResp().setContentLength(bos.size());
            bos.writeTo(cmsReq.getHttpResp().getOutputStream());
        } catch (Exception ex) {
            renderFinalError(cmsReq, ex);
        }
    }

    public void renderFinalError(CMSRequest cmsReq, Exception ex)
            throws IOException {
        logger.debug("Caught exception in renderFinalError: " + ex.getMessage(), ex);
        // this template is the last resort for all other unexpected
        // errors in other templates so we can only output text.
        HttpServletResponse httpResp = cmsReq.getHttpResp();

        httpResp.setContentType("text/html");
        ServletOutputStream out = httpResp.getOutputStream();

        // replace $ERRORMSG with exception message if included.
        String finalErrMsg = mFinalErrorMsg;
        int tokenIdx = mFinalErrorMsg.indexOf(ERROR_MSG_TOKEN);

        if (tokenIdx != -1) {
            finalErrMsg =
                    mFinalErrorMsg.substring(0, tokenIdx) +
                            StringEscapeUtils.escapeHtml4(ex.toString()) +
                            mFinalErrorMsg.substring(
                                    tokenIdx + ERROR_MSG_TOKEN.length());
        }
        out.println(finalErrMsg);
    }

    /**
     * Invalidates a SSL Session. So client auth will happen again.
     */
    protected static void invalidateSSLSession(HttpServletRequest httpReq) {

        /*
         try {
         s = (SSLSocket) ((HTTPRequest) httpReq).getConnection().getSocket();
         } catch (ClassCastException e) {
            logger.warn(CMS.getLogMessage("CMSGW_SSL_NO_INVALIDATE"), e);
         // ignore.
         return;
         }
         try {
         s.invalidateSession();
         s.resetHandshake();
         }catch (SocketException se) {
         }
         */
    }

    /**
     * construct a authentication credentials to pass into authentication
     * manager.
     */
    public static AuthCredentials getAuthCreds(
            AuthManager authMgr, IArgBlock argBlock, X509Certificate clientCert)
            throws EBaseException {
        // get credentials from http parameters.
        String[] reqCreds = authMgr.getRequiredCreds();
        AuthCredentials creds = new AuthCredentials();

        for (int i = 0; i < reqCreds.length; i++) {
            String reqCred = reqCreds[i];

            if (reqCred.equals(AuthManager.CRED_SSL_CLIENT_CERT)) {
                // cert could be null;
                creds.set(reqCred, new X509Certificate[] { clientCert }
                        );
            } else {
                String value = argBlock.getValueAsString(reqCred);

                creds.set(reqCred, value); // value could be null;
            }
        }
        // Inserted by bskim
        creds.setArgBlock(argBlock);
        // Insert end
        return creds;
    }

    /**
     * get ssl client authenticated certificate
     */
    protected X509Certificate getSSLClientCertificate(HttpServletRequest httpReq) throws EBaseException {
        return getSSLClientCertificate(httpReq, true);
    }

    protected X509Certificate getSSLClientCertificate(HttpServletRequest httpReq, boolean clientCertRequired) throws EBaseException {

        X509Certificate cert = null;

        logger.info(CMS.getLogMessage("CMSGW_GETTING_SSL_CLIENT_CERT"));

        // iws60 support Java Servlet Spec V2.2, attribute
        // javax.servlet.request.X509Certificate now contains array
        // of X509Certificates instead of one X509Certificate object
        X509Certificate[] allCerts = (X509Certificate[]) httpReq.getAttribute(CERT_ATTR);

        if (allCerts == null || allCerts.length == 0) {
            if (clientCertRequired) {
                throw new EBaseException("You did not provide a valid certificate for this operation");
            }
            return null;
        }

        cert = allCerts[0];

        if (cert == null) {
            // just don't have a cert.
            logger.error(CMS.getLogMessage("CMSGW_SSL_CL_CERT_FAIL"));
            return null;
        }

        // convert to sun's x509 cert interface.
        try {
            byte[] certEncoded = cert.getEncoded();

            cert = new X509CertImpl(certEncoded);
        } catch (CertificateEncodingException e) {
            logger.error(CMS.getLogMessage("CMSGW_SSL_CL_CERT_FAIL_ENCODE", e.getMessage()), e);
            return null;
        } catch (CertificateException e) {
            logger.error(CMS.getLogMessage("CMSGW_SSL_CL_CERT_FAIL_DECODE", e.getMessage()), e);
            return null;
        }
        return cert;
    }

    /**
     * get a template based on result status.
     */
    protected CMSTemplate getTemplate(
            String templateName, HttpServletRequest httpReq, Locale[] locale)
            throws EBaseException, IOException {
        // this converts to system dependent file seperator char.
        if (mServletConfig == null) {
            logger.error("CMSServlet:getTemplate() - mServletConfig is null!");
            return null;
        }
        String realpath =
                mServletConfig.getServletContext().getRealPath("/" + templateName);

        if (realpath == null) {
            logger.error(CMS.getLogMessage("CMSGW_NO_FIND_TEMPLATE", templateName));
            throw new ECMSGWException(CMS.getLogMessage("CMSGW_ERROR_DISPLAY_TEMPLATE"));
        }

        File realpathFile = new File(realpath);
        File templateFile =
                getLangFile(httpReq, realpathFile, locale);
        String charSet = httpReq.getCharacterEncoding();

        if (charSet == null) {
            charSet = "UTF8";
        }
        return (CMSTemplate) mFileLoader.getCMSFile(templateFile, charSet);
    }

    /**
     * get http parameters not to save from configuration.
     */
    protected void getDontSaveHttpParams(ServletConfig sc) {
        String dontSaveParams = null;

        try {
            for (int i = 0; i < DONT_SAVE_HTTP_PARAMS.length; i++) {
                mDontSaveHttpParams.addElement(DONT_SAVE_HTTP_PARAMS[i]);
            }
            dontSaveParams = sc.getInitParameter(
                        PROP_DONT_SAVE_HTTP_PARAMS);
            if (dontSaveParams != null) {
                StringTokenizer params =
                        new StringTokenizer(dontSaveParams, ",");

                while (params.hasMoreTokens()) {
                    String param = params.nextToken();

                    mDontSaveHttpParams.addElement(param);
                }
            }
        } catch (Exception e) {
            logger.warn(CMS.getLogMessage("CMSGW_NO_CONFIG_VALUE", PROP_DONT_SAVE_HTTP_PARAMS, e.toString()), e);
            // default just in case.
            for (int i = 0; i < DONT_SAVE_HTTP_PARAMS.length; i++) {
                mDontSaveHttpParams.addElement(DONT_SAVE_HTTP_PARAMS[i]);
            }
        }
    }

    /**
     * get http headers to save from configuration.
     */
    protected void getSaveHttpHeaders(ServletConfig sc) {
        try {
            // init save http headers. default will always be saved.
            for (int i = 0; i < SAVE_HTTP_HEADERS.length; i++) {
                mSaveHttpHeaders.addElement(SAVE_HTTP_HEADERS[i]);
            }

            // now get from config file if there's more.
            String saveHeaders =
                    sc.getInitParameter(PROP_SAVE_HTTP_HEADERS);

            if (saveHeaders != null) {
                StringTokenizer headers =
                        new StringTokenizer(saveHeaders, ",");

                while (headers.hasMoreTokens()) {
                    String hdr = headers.nextToken();

                    mSaveHttpHeaders.addElement(hdr);
                }
            }
        } catch (Exception e) {
            logger.warn(CMS.getLogMessage("CMSGW_NO_CONFIG_VALUE", PROP_SAVE_HTTP_HEADERS, e.toString()), e);
        }
    }

    /**
     * save http headers in a Request.
     */
    protected void saveHttpHeaders(
            HttpServletRequest httpReq, Request req)
            throws EBaseException {
        Hashtable<String, String> headers = new Hashtable<>();
        Enumeration<String> hdrs = mSaveHttpHeaders.elements();

        while (hdrs.hasMoreElements()) {
            String hdr = hdrs.nextElement();
            String val = httpReq.getHeader(hdr);

            if (val != null) {
                headers.put(hdr, val);
            }
        }
        req.setExtData(Request.HTTP_HEADERS, headers);
    }

    /**
     * save http headers in a Request.
     */
    protected void saveHttpParams(
            IArgBlock httpParams, Request req) {
        Hashtable<String, String> saveParams = new Hashtable<>();

        Enumeration<String> names = httpParams.elements();

        while (names.hasMoreElements()) {
            String name = names.nextElement();
            Enumeration<String> params = mDontSaveHttpParams.elements();
            boolean dosave = true;

            while (params.hasMoreElements()) {
                String param = params.nextElement();

                if (name.equalsIgnoreCase(param)) {
                    dosave = false;
                    break;
                }
            }
            if (dosave) {
                // kmccarth
                // fear not - service() calls toHashtable() which only
                // retrieves string values.
                // TODO - when we can use JDK5 features we should typecast
                // the params until they get here
                saveParams.put(name, (String) httpParams.get(name));
            }
        }
        req.setExtData(Request.HTTP_PARAMS, saveParams);
    }

    /**
     * handy routine for getting a cert record given a serial number.
     */
    protected CertRecord getCertRecord(BigInteger serialNo) {
        if (!(mAuthority instanceof ICertificateAuthority)) {
            logger.error(CMS.getLogMessage("CMSGW_NON_CERT_AUTH"));
            return null;
        }
        CertificateRepository certdb = ((ICertificateAuthority) mAuthority).getCertificateRepository();

        if (certdb == null) {
            logger.error(CMS.getLogMessage("CMSGW_CERT_DB_NULL", mAuthority.toString()));
            return null;
        }
        CertRecord certRecord = null;

        try {
            certRecord = certdb.readCertificateRecord(serialNo);
        } catch (EBaseException e) {
            logger.error(CMS.getLogMessage("CMSGW_NO_CERT_REC", serialNo.toString(16), e.toString()), e);
            return null;
        }
        return certRecord;
    }

    /**
     * handy routine for validating if a cert is from this CA.
     * mAuthority must be a CA.
     */
    protected boolean isCertFromCA(X509Certificate cert) {
        BigInteger serialno = cert.getSerialNumber();
        X509CertImpl certInDB = (X509CertImpl) getX509Certificate(serialno);

        return certInDB != null && certInDB.equals(cert);
    }

    /**
     * handy routine for checking if a list of certs is from this CA.
     * mAuthortiy must be a CA.
     */
    protected boolean areCertsFromCA(X509Certificate[] certs) {
        for (int i = certs.length - 1; i >= 0; i--) {
            if (!isCertFromCA(certs[i]))
                return false;
        }
        return true;
    }

    /**
     * handy routine for getting a certificate from the certificate
     * repository. mAuthority must be a CA.
     */
    protected X509Certificate getX509Certificate(BigInteger serialNo) {
        if (!(mAuthority instanceof ICertificateAuthority)) {
            logger.error(CMS.getLogMessage("CMSGW_NOT_CERT_AUTH"));
            return null;
        }
        CertificateRepository certdb = ((ICertificateAuthority) mAuthority).getCertificateRepository();

        if (certdb == null) {
            logger.error(CMS.getLogMessage("CMSGW_CERT_DB_NULL", mAuthority.toString()));
            return null;
        }
        X509Certificate cert = null;

        try {
            cert = certdb.getX509Certificate(serialNo);
        } catch (EBaseException e) {
            logger.error(CMS.getLogMessage("CMSGW_NO_CERT_REC", serialNo.toString(16), e.toString()), e);
            return null;
        }
        return cert;
    }

    /**
     * instantiate a new filler from a class name,
     *
     * @return null if can't be instantiated, new instance otherwise.
     */
    protected ICMSTemplateFiller newFillerObject(String fillerClass) {
        ICMSTemplateFiller filler = null;

        try {
            filler = (ICMSTemplateFiller) Class.forName(fillerClass).getDeclaredConstructor().newInstance();
        } catch (Exception e) {
            if ((e instanceof RuntimeException)) {
                throw (RuntimeException) e;
            }
            logger.error(CMS.getLogMessage("CMSGW_CANT_LOAD_FILLER", fillerClass, e.toString()), e);
            return null;
        }
        return filler;
    }

    /**
     * set default templates.
     * subclasses can override, and should override at least the success
     * template
     */
    protected void setDefaultTemplates(ServletConfig sc) {
        // Subclasses should override these for diff templates and params in
        // their constructors.
        // Set a template name to null to not use these standard ones.
        // When template name is set to null nothing will be displayed.
        // Servlet is assumed to have rendered its own output.
        // The only exception is the unexpected error template where the
        // default one will always be used if template name is null.
        String successTemplate = null;
        String errorTemplate = null;
        String unauthorizedTemplate = null;
        String pendingTemplate = null;
        String svcpendingTemplate = null;
        String rejectedTemplate = null;
        String unexpectedErrorTemplate = null;

        String gateway = sc.getInitParameter("interface");
        String authority = sc.getInitParameter(PROP_AUTHORITY);
        if (authority == null) {
            authority = sc.getInitParameter("authorityId");
        }

        try {
            successTemplate = sc.getInitParameter(
                        PROP_SUCCESS_TEMPLATE);
            if (successTemplate == null) {
                successTemplate = SUCCESS_TEMPLATE;
                if (gateway != null)
                    //successTemplate = "/"+gateway+successTemplate;
                    successTemplate = "/" + gateway + successTemplate;
            }

            errorTemplate = sc.getInitParameter(
                        PROP_ERROR_TEMPLATE);
            if (errorTemplate == null) {
                errorTemplate = ERROR_TEMPLATE;
                if (gateway != null)
                    //errorTemplate = "/"+gateway+errorTemplate;
                    errorTemplate = "/" + gateway + errorTemplate;
            }

            unauthorizedTemplate = sc.getInitParameter(
                        PROP_UNAUTHORIZED_TEMPLATE);
            if (unauthorizedTemplate == null) {
                unauthorizedTemplate = UNAUTHORIZED_TEMPLATE;
                if (gateway != null)
                    //unauthorizedTemplate = "/"+gateway+unauthorizedTemplate;
                    unauthorizedTemplate = "/" + gateway + unauthorizedTemplate;
            }

            pendingTemplate = sc.getInitParameter(
                        PROP_PENDING_TEMPLATE);
            if (pendingTemplate == null) {
                pendingTemplate = PENDING_TEMPLATE;
                if (gateway != null)
                    //pendingTemplate = "/"+gateway+pendingTemplate;
                    pendingTemplate = "/" + gateway + pendingTemplate;
            }

            svcpendingTemplate = sc.getInitParameter(
                        PROP_SVC_PENDING_TEMPLATE);
            if (svcpendingTemplate == null) {
                svcpendingTemplate = SVC_PENDING_TEMPLATE;
                if (gateway != null)
                    //svcpendingTemplate = "/"+gateway+svcpendingTemplate;
                    svcpendingTemplate = "/" + gateway + svcpendingTemplate;
            }

            rejectedTemplate = sc.getInitParameter(
                        PROP_REJECTED_TEMPLATE);
            if (rejectedTemplate == null) {
                rejectedTemplate = REJECTED_TEMPLATE;
                if (gateway != null)
                    //rejectedTemplate = "/"+gateway+rejectedTemplate;
                    rejectedTemplate = "/" + gateway + rejectedTemplate;
            }

            unexpectedErrorTemplate = sc.getInitParameter(
                        PROP_EXCEPTION_TEMPLATE);
            if (unexpectedErrorTemplate == null) {
                unexpectedErrorTemplate = EXCEPTION_TEMPLATE;
                if (gateway != null)
                    //unexpectedErrorTemplate = "/"+gateway+unexpectedErrorTemplate;
                    unexpectedErrorTemplate = "/" + gateway + unexpectedErrorTemplate;
            }
        } catch (Exception e) {
            logger.warn(CMS.getLogMessage("CMSGW_IMP_INIT_SERV_ERR", e.toString(), mId), e);
        }

        mTemplates.put(
                CMSRequest.UNAUTHORIZED,
                new CMSLoadTemplate(
                        PROP_UNAUTHORIZED_TEMPLATE, PROP_UNAUTHOR_TEMPLATE_FILLER,
                        unauthorizedTemplate, null));
        mTemplates.put(
                CMSRequest.SUCCESS,
                new CMSLoadTemplate(
                        PROP_SUCCESS_TEMPLATE, PROP_SUCCESS_TEMPLATE_FILLER,
                        successTemplate, new GenSuccessTemplateFiller()));
        mTemplates.put(
                CMSRequest.PENDING,
                new CMSLoadTemplate(
                        PROP_PENDING_TEMPLATE, PROP_PENDING_TEMPLATE_FILLER,
                        pendingTemplate, new GenPendingTemplateFiller()));
        mTemplates.put(
                CMSRequest.SVC_PENDING,
                new CMSLoadTemplate(
                        PROP_SVC_PENDING_TEMPLATE, PROP_SVC_PENDING_TEMPLATE_FILLER,
                        svcpendingTemplate, new GenSvcPendingTemplateFiller()));
        mTemplates.put(
                CMSRequest.REJECTED,
                new CMSLoadTemplate(
                        PROP_REJECTED_TEMPLATE, PROP_REJECTED_TEMPLATE_FILLER,
                        rejectedTemplate, new GenRejectedTemplateFiller()));
        mTemplates.put(
                CMSRequest.ERROR,
                new CMSLoadTemplate(
                        PROP_ERROR_TEMPLATE, PROP_ERROR_TEMPLATE_FILLER,
                        errorTemplate, new GenErrorTemplateFiller()));
        mTemplates.put(
                CMSRequest.EXCEPTION,
                new CMSLoadTemplate(
                        PROP_EXCEPTION_TEMPLATE, PROP_EXCEPTION_TEMPLATE_FILLER,
                        unexpectedErrorTemplate, new GenUnexpectedErrorTemplateFiller()));
    }

    /**
     * handy routine to check if client is navigator based on user-agent.
     */
    public static boolean clientIsNav(HttpServletRequest httpReq) {
        String useragent = httpReq.getHeader("user-agent");

        return useragent.startsWith("Mozilla") && useragent.indexOf("MSIE") == -1;
    }

    /**
     * handy routine to check if client is msie based on user-agent.
     */
    public static boolean clientIsMSIE(HttpServletRequest httpReq) {
        String useragent = httpReq.getHeader("user-agent");

        return useragent != null && useragent.indexOf("MSIE") != -1;
    }

    /**
     * handy routine to check if client is cartman based on hidden http input
     * set using cartman JS. (no other way to tell)
     */
    private static String CMMF_RESPONSE = "cmmfResponse";

    public static boolean doCMMFResponse(IArgBlock httpParams) {
        return httpParams.getValueAsBoolean(CMMF_RESPONSE, false);
    }

    private static final String IMPORT_CERT = "importCert";
    private static final String IMPORT_CHAIN = "importCAChain";
    private static final String IMPORT_CERT_MIME_TYPE = "importCertMimeType";
    // default mime type
    private static final String NS_X509_USER_CERT = "application/x-x509-user-cert";
    private static final String NS_X509_EMAIL_CERT = "application/x-x509-email-cert";

    // CMC mime types
    public static final String SIMPLE_ENROLLMENT_REQUEST = "application/pkcs10";
    public static final String SIMPLE_ENROLLMENT_RESPONSE = "application/pkcs7-mime";
    public static final String FULL_ENROLLMENT_REQUEST = "application/pkcs7-mime";
    public static final String FULL_ENROLLMENT_RESPONSE = "application/pkcs7-mime";

    /**
     * handy routine to check if client want full enrollment response
     */
    public static String FULL_RESPONSE = "fullResponse";

    public static boolean doFullResponse(IArgBlock httpParams) {
        return httpParams.getValueAsBoolean(FULL_RESPONSE, false);
    }

    /**
     * @return false if import cert directly set to false.
     * @return true if import cert directly is true and import cert.
     */
    protected boolean checkImportCertToNav(
            HttpServletResponse httpResp, IArgBlock httpParams, X509CertImpl cert)
            throws EBaseException {
        if (!httpParams.getValueAsBoolean(IMPORT_CERT, false)) {
            return false;
        }
        boolean importCAChain =
                httpParams.getValueAsBoolean(IMPORT_CHAIN, true);
        // XXX Temporary workaround because of problem with passing Mime type
        boolean emailCert =
                httpParams.getValueAsBoolean("emailCert", false);
        String importMimeType = (emailCert) ?
                httpParams.getValueAsString(IMPORT_CERT_MIME_TYPE, NS_X509_EMAIL_CERT) :
                httpParams.getValueAsString(IMPORT_CERT_MIME_TYPE, NS_X509_USER_CERT);

        //		String importMimeType =
        //			httpParams.getValueAsString(
        //				IMPORT_CERT_MIME_TYPE, NS_X509_USER_CERT);
        importCertToNav(httpResp, cert, importMimeType, importCAChain);
        return true;
    }

    /**
     * handy routine to import cert to old navigator in nav mime type.
     */
    public void importCertToNav(
            HttpServletResponse httpResp, X509CertImpl cert,
            String contentType, boolean importCAChain)
            throws EBaseException {
        ServletOutputStream out = null;
        byte[] encoding = null;

        logger.debug("CMSServlet: importCertToNav " +
                       "contentType=" + contentType + " " +
                       "importCAChain=" + importCAChain);
        try {
            out = httpResp.getOutputStream();
            // CA chain.
            if (importCAChain) {
                CertificateChain caChain = null;
                X509Certificate[] caCerts = null;
                PKCS7 p7 = null;

                caChain = ((ICertificateAuthority) mAuthority).getCACertChain();
                caCerts = caChain.getChain();

                // set user + CA cert chain in pkcs7
                X509CertImpl[] userChain =
                        new X509CertImpl[caCerts.length + 1];

                userChain[0] = cert;
                int m = 1, n = 0;

                for (; n < caCerts.length; m++, n++) {
                    userChain[m] = (X509CertImpl) caCerts[n];

                    /*
                     System.out.println(
                     m+"th Cert "+userChain[m].toString());
                     */
                }
                p7 = new PKCS7(new AlgorithmId[0],
                            new ContentInfo(new byte[0]),
                            userChain,
                            new SignerInfo[0]);
                ByteArrayOutputStream bos = new ByteArrayOutputStream();

                p7.encodeSignedData(bos, false);
                encoding = bos.toByteArray();
                logger.debug("CMServlet: return P7 " + Utils.base64encode(encoding, true));
            } else {
                encoding = cert.getEncoded();
                logger.debug("CMServlet: return Certificate " + Utils.base64encode(encoding, true));
            }
            httpResp.setContentType(contentType);
            out.write(encoding);
        } catch (IOException e) {
            logger.error(CMS.getLogMessage("CMSGW_RET_CERT_IMPORT_ERR", e.toString()), e);
            throw new ECMSGWException(CMS.getLogMessage("CMSGW_ERROR_RETURNING_CERT"), e);
        } catch (CertificateEncodingException e) {
            logger.error(CMS.getLogMessage("CMSGW_NO_ENCODED_IMP_CERT", e.toString()), e);
            throw new ECMSGWException(CMS.getLogMessage("CMSGW_ERROR_ENCODING_ISSUED_CERT"), e);
        }
    }

    protected static void saveAuthToken(IAuthToken token, Request req) {
        if (token != null && req != null)
            req.setExtData(Request.AUTH_TOKEN, token);

        // # 56230 - expose auth token parameters to the policy predicate
        if (token != null && req != null) {
            Enumeration<String> e = token.getElements();
            while (e.hasMoreElements()) {
                String n = e.nextElement();
                String[] x1 = token.getInStringArray(n);
                if (x1 != null) {
                    for (int i = 0; i < x1.length; i++) {
                        logger.debug("Setting " + Request.AUTH_TOKEN + "-" + n +
                                "(" + i + ")=" + x1[i]);
                        req.setExtData(Request.AUTH_TOKEN + "-" + n + "(" + i + ")",
                                x1[i]);
                    }
                } else {
                    String x = token.getInString(n);
                    if (x != null) {
                        logger.debug("Setting " + Request.AUTH_TOKEN + "-" + n + "=" + x);
                        req.setExtData(Request.AUTH_TOKEN + "-" + n, x);
                    }
                }
            } // while
        } // if
    }

    protected IAuthToken getAuthToken(Request req) {
        return req.getExtDataInAuthToken(Request.AUTH_TOKEN);
    }

    protected static boolean connectionIsSSL(HttpServletRequest httpReq) {
        return httpReq.isSecure();
    }

    /**
     * handy routine for getting agent's relative path
     */
    protected String getRelPath(IAuthority authority) {
        if (authority instanceof ICertificateAuthority)
            return "ca/";
        else if (authority instanceof IKeyRecoveryAuthority)
            return "kra/";
        else
            return "/";
    }

    /**
     * A system certificate such as the CA signing certificate
     * should not be allowed to delete.
     * The main purpose is to avoid revoking the self signed
     * CA certificate accidentially.
     */
    protected boolean isSystemCertificate(BigInteger serialNo) throws EBaseException {
        if (!(mAuthority instanceof ICertificateAuthority)) {
            return false;
        }
        X509Certificate caCert =
                ((ICertificateAuthority) mAuthority).getCACert();
        if (caCert != null) {
            /* only check this if we are self-signed */
            if (caCert.getSubjectDN().equals(caCert.getIssuerDN())) {
                if (caCert.getSerialNumber().equals(serialNo)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * make a CRL entry from a serial number and revocation reason.
     *
     * @return a RevokedCertImpl that can be entered in a CRL.
     */
    protected RevokedCertImpl formCRLEntry(
            BigInteger serialNo, RevocationReason reason)
            throws EBaseException {
        CRLReasonExtension reasonExt = new CRLReasonExtension(reason);
        CRLExtensions crlentryexts = new CRLExtensions();

        try {
            crlentryexts.set(CRLReasonExtension.NAME, reasonExt);
        } catch (IOException e) {
            logger.error(CMS.getLogMessage("CMSGW_ERR_CRL_REASON", reason.toString(), e.toString()), e);
            throw new ECMSGWException(CMS.getLogMessage("CMSGW_ERROR_SETTING_CRLREASON"), e);
        }
        return new RevokedCertImpl(serialNo, new Date(), crlentryexts);
    }

    /**
     * check if a certificate (serial number) is revoked on a CA.
     *
     * @return true if cert is marked revoked in the CA's database.
     * @return false if cert is not marked revoked.
     */
    protected boolean certIsRevoked(BigInteger serialNum)
            throws EBaseException {
        CertRecord certRecord = getCertRecord(serialNum);

        if (certRecord == null) {
            logger.error(CMS.getLogMessage("CMSGW_BAD_CERT_SER_NUM", String.valueOf(serialNum)));
            throw new ECMSGWException(CMS.getLogMessage("CMSGW_INVALID_CERT"));
        }
        return certRecord.getStatus().equals(CertRecord.STATUS_REVOKED);
    }

    public static String generateSalt() {
        CMSEngine engine = CMS.getCMSEngine();
        JssSubsystem jssSubsystem = engine.getJSSSubsystem();
        SecureRandom rnd = jssSubsystem.getRandomNumberGenerator();
        return Integer.toString(rnd.nextInt());
    }

    protected String hashPassword(String pwd) {
        String salt = generateSalt();
        byte[] pwdDigest = mSHADigest.digest((salt + pwd).getBytes());
        String b64E = Utils.base64encode(pwdDigest, true);

        return "{SHA-256}" + salt + ";" + b64E;
    }

    /**
     * @param req http servlet request
     * @param realpathFile the file to get.
     * @param locale array of at least one to be filled with locale found.
     */
    public static File getLangFile(
            HttpServletRequest req, File realpathFile, Locale[] locale)
            throws IOException {
        File file = null;
        String acceptLang = req.getHeader("accept-language");

        if (acceptLang != null && !acceptLang.equals("")) {
            StringTokenizer tokenizer = new StringTokenizer(acceptLang, ",");
            int numLangs = tokenizer.countTokens();

            if (numLangs > 0) {
                // languages are searched in order.
                String parent = realpathFile.getParent();

                if (parent == null) {
                    parent = "." + File.separatorChar;
                }
                String name = realpathFile.getName();

                if (name == null) { // filename should never be null.
                    throw new IOException("file has no name");
                }
                int i;

                for (i = 0; i < numLangs; i++) {
                    String lang = null;
                    String token = tokenizer.nextToken();

                    int semicolon = token.indexOf(';');

                    if (semicolon == -1) {
                        lang = token.trim();
                    } else {
                        if (semicolon < 2)
                            continue; // protocol error.
                        lang = token.substring(0, semicolon).trim();
                    }
                    // if browser locale is the same as default locale,
                    // use the default form. (is this the right thing to do ?)
                    Locale l = getLocale(lang);

                    if (Locale.getDefault().equals(l)) {
                        locale[0] = l;
                        file = realpathFile;
                        break;
                    }

                    String langfilepath =
                            parent + File.separatorChar +
                                    lang + File.separatorChar + name;

                    file = new File(langfilepath);
                    if (file.exists()) {
                        locale[0] = getLocale(lang);
                        break;
                    }
                }
                // if no file for lang was found use default
                if (i == numLangs) {
                    file = realpathFile;
                    locale[0] = Locale.getDefault();
                }
            }
        } else {
            // use default if accept-language is not availabe
            file = realpathFile;
            locale[0] = Locale.getDefault();
        }
        return file;
    }

    public static Locale getLocale(String lang) {
        int dash = lang.indexOf('-');

        if (dash == -1) {
            return new Locale(lang, "");
        }
        return new Locale(lang.substring(0, dash), lang.substring(dash + 1));
    }

    public IAuthToken authenticate(CMSRequest req)
            throws EBaseException {
        return authenticate(req, mAuthMgr);
    }

    public IAuthToken authenticate(HttpServletRequest httpReq)
            throws EBaseException {
        return authenticate(httpReq, mAuthMgr);
    }

    public IAuthToken authenticate(CMSRequest req, String authMgrName)
            throws EBaseException {
        IAuthToken authToken = authenticate(req.getHttpReq(),
                authMgrName);

        saveAuthToken(authToken, req.getRequest());
        return authToken;
    }

    /**
     * Authentication
     * <P>
     *
     * <ul>
     * <li>signed.audit LOGGING_SIGNED_AUDIT_AUTH_FAIL used when authentication fails (in case of SSL-client auth, only
     * webserver env can pick up the SSL violation; CS authMgr can pick up cert mis-match, so this event is used)
     * <li>signed.audit LOGGING_SIGNED_AUDIT_AUTH_SUCCESS used when authentication succeeded
     * </ul>
     *
     * @exception EBaseException an error has occurred
     */
    public IAuthToken authenticate(HttpServletRequest httpReq, String authMgrName)
            throws EBaseException {

        String auditSubjectID = ILogger.UNIDENTIFIED;
        String auditAuthMgrID = ILogger.UNIDENTIFIED;
        String auditUID = ILogger.UNIDENTIFIED;

        // ensure that any low-level exceptions are reported
        // to the signed audit log and stored as failures
        try {
            String getClientCert = mGetClientCert;

            ArgBlock httpArgs = new ArgBlock(toHashtable(httpReq));
            SessionContext ctx = SessionContext.getContext();
            String ip = httpReq.getRemoteAddr();
            logger.debug("IP: " + ip);

            if (ip != null) {
                ctx.put(SessionContext.IPADDRESS, ip);
            }
            if (authMgrName != null) {
                logger.debug("AuthMgrName: " + authMgrName);
                ctx.put(SessionContext.AUTH_MANAGER_ID, authMgrName);
            }
            // put locale into session context
            ctx.put(SessionContext.LOCALE, getLocale(httpReq));

            //
            // check ssl client authentication if specified.
            //
            X509Certificate clientCert = null;

            if (getClientCert != null && getClientCert.equals("true")) {
                logger.debug("CMSServlet: retrieving SSL certificate");
                clientCert = getSSLClientCertificate(httpReq);
            }

            //
            // check authentication by auth manager if any.
            //
            if (authMgrName == null) {

                // Fixed Blackflag Bug #613900:  Since this code block does
                // NOT actually constitute an authentication failure, but
                // rather the case in which a given servlet has been correctly
                // configured to NOT require an authentication manager, the
                // audit message called LOGGING_SIGNED_AUDIT_AUTH_FAIL has
                // been removed.

                logger.debug("CMSServlet: no authMgrName");
                return null;
            }
            // save the "Subject DN" of this certificate in case it
            // must be audited as an authentication failure
            if (clientCert == null) {
                logger.debug("CMSServlet: no client certificate found");
            } else {
                String certUID = clientCert.getSubjectDN().getName();
                logger.debug("CMSServlet: certUID=" + certUID);

                if (certUID != null) {
                    certUID = certUID.trim();

                    if (!(certUID.equals(""))) {
                        // reset the "auditUID"
                        auditUID = certUID;
                    }
                }
            }

            // reset the "auditAuthMgrID"
            auditAuthMgrID = authMgrName;
            IAuthToken authToken = CMSGateway.checkAuthManager(httpReq,
                    httpArgs,
                    clientCert,
                    authMgrName);
            if (authToken == null) {
                return null;
            }
            String userid = authToken.getInString(IAuthToken.USER_ID);

            logger.debug("CMSServlet: userid=" + userid);

            if (userid != null) {
                ctx.put(SessionContext.USER_ID, userid);
            }

            // reset the "auditSubjectID"
            auditSubjectID = auditSubjectID();

            audit(AuthEvent.createSuccessEvent(
                        auditSubjectID,
                        auditAuthMgrID));

            return authToken;
        } catch (EBaseException eAudit1) {

            audit(AuthEvent.createFailureEvent(
                        auditSubjectID,
                        auditAuthMgrID,
                        auditUID));

            // rethrow the specific exception to be handled later
            throw eAudit1;
        }
    }

    public AuthzToken authorize(String authzMgrName, String resource, IAuthToken authToken,
            String exp) throws EBaseException {
        AuthzToken authzToken = null;

        logger.debug("CMSServlet.authorize(" + authzMgrName + ", " + resource + ")");

        String auditSubjectID = auditSubjectID();
        String auditGroupID = auditGroupID();
        String auditACLResource = resource;
        String auditOperation = "enroll";

        try {
            authzToken = mAuthz.authorize(authzMgrName, authToken, exp);
            if (authzToken != null) {

                audit(AuthzEvent.createSuccessEvent(
                            auditSubjectID,
                            auditACLResource,
                            auditOperation));

                audit(RoleAssumeEvent.createSuccessEvent(
                            auditSubjectID,
                            auditGroupID));

            } else {

                audit(AuthzEvent.createFailureEvent(
                            auditSubjectID,
                            auditACLResource,
                            auditOperation));

                audit(RoleAssumeEvent.createFailureEvent(
                            auditSubjectID,
                            auditGroupID));
            }
            return authzToken;
        } catch (Exception e) {

            audit(AuthzEvent.createFailureEvent(
                        auditSubjectID,
                        auditACLResource,
                        auditOperation));

            audit(RoleAssumeEvent.createFailureEvent(
                        auditSubjectID,
                        auditGroupID));

            throw new EBaseException(e.toString());
        }
    }

    /**
     * Authorize must occur after Authenticate
     * <P>
     *
     * <ul>
     * <li>signed.audit LOGGING_SIGNED_AUDIT_AUTHZ_FAIL used when authorization has failed
     * <li>signed.audit LOGGING_SIGNED_AUDIT_AUTHZ_SUCCESS used when authorization is successful
     * <li>signed.audit LOGGING_SIGNED_AUDIT_ROLE_ASSUME used when user assumes a role (in current CS that's when one
     * accesses a role port)
     * </ul>
     *
     * @param authzMgrName string representing the name of the authorization
     *            manager
     * @param authToken the authentication token
     * @param resource a string representing the ACL resource id as defined in
     *            the ACL resource list
     * @param operation a string representing one of the operations as defined
     *            within the ACL statement (e. g. - "read" for an ACL statement containing
     *            "(read,write)")
     * @exception EBaseException an error has occurred
     * @return the authorization token
     */
    public AuthzToken authorize(String authzMgrName, IAuthToken authToken,
            String resource, String operation)
            throws EBaseException {

        logger.debug("CMSServlet.authorize(" + authzMgrName + ")");

        String auditSubjectID = auditSubjectID();
        String auditGroupID = auditGroupID();
        String auditID = auditSubjectID;
        String auditACLResource = resource;
        String auditOperation = operation;

        SessionContext auditContext = SessionContext.getExistingContext();
        String authManagerId = null;

        if (auditContext != null) {
            authManagerId = (String) auditContext.get(SessionContext.AUTH_MANAGER_ID);

            if (authManagerId != null && authManagerId.equals("TokenAuth")) {
                if (auditSubjectID.equals(ILogger.NONROLEUSER) ||
                        auditSubjectID.equals(ILogger.UNIDENTIFIED)) {
                    logger.debug("CMSServlet: in authorize... TokenAuth auditSubjectID unavailable, changing to auditGroupID");
                    auditID = auditGroupID;
                }
            }
        }

        // "normalize" the "auditACLResource" value
        if (auditACLResource != null) {
            auditACLResource = auditACLResource.trim();
        }

        // "normalize" the "auditOperation" value
        if (auditOperation != null) {
            auditOperation = auditOperation.trim();
        }

        if (authzMgrName == null) {
            // Fixed Blackflag Bug #613900:  Since this code block does
            // NOT actually constitute an authorization failure, but
            // rather the case in which a given servlet has been correctly
            // configured to NOT require an authorization manager, the
            // audit message called LOGGING_SIGNED_AUDIT_AUTHZ_FAIL and
            // the audit message called LOGGING_SIGNED_AUDIT_ROLE_ASSUME
            // (marked as a failure) have been removed.

            return null;
        }

        String roles = auditGroups(auditSubjectID);

        try {
            AuthzToken authzTok = mAuthz.authorize(authzMgrName,
                    authToken,
                    resource,
                    operation);

            if (authzTok != null) {

                audit(AuthzEvent.createSuccessEvent(
                            auditSubjectID,
                            auditACLResource,
                            auditOperation));

                if (roles != null) {
                    audit(RoleAssumeEvent.createSuccessEvent(
                                auditID,
                                roles));
                }

            } else {

                audit(AuthzEvent.createFailureEvent(
                            auditSubjectID,
                            auditACLResource,
                            auditOperation));

                if (roles != null) {
                    audit(RoleAssumeEvent.createFailureEvent(
                                auditID,
                                roles));
                }
            }

            return authzTok;
        } catch (EBaseException eAudit1) {

            audit(AuthzEvent.createFailureEvent(
                        auditSubjectID,
                        auditACLResource,
                        auditOperation));

            if (roles != null) {
                audit(RoleAssumeEvent.createFailureEvent(
                            auditID,
                            roles));
            }

            return null;
        } catch (Exception eAudit1) {

            audit(AuthzEvent.createFailureEvent(
                        auditSubjectID,
                        auditACLResource,
                        auditOperation));

            if (roles != null) {
                audit(RoleAssumeEvent.createFailureEvent(
                            auditSubjectID,
                            roles));
            }

            return null;
        }
    }

    /**
     * Signed Audit Log
     *
     * This method is inherited by all extended "CMSServlet"s,
     * and is called to store messages to the signed audit log.
     * <P>
     *
     * @param msg signed audit log message
     */
    protected void audit(String msg) {
        signedAuditLogger.log(msg);
    }

    protected void audit(LogEvent event) {
        signedAuditLogger.log(event);
    }

    /**
     * Signed Audit Log Subject ID
     *
     * This method is inherited by all extended "CMSServlet"s,
     * and is called to obtain the "SubjectID" for
     * a signed audit log message.
     * <P>
     *
     * @return id string containing the signed audit log message SubjectID
     */
    protected String auditSubjectID() {

        SessionContext auditContext = SessionContext.getExistingContext();

        if (auditContext == null) {
            return ILogger.UNIDENTIFIED;
        }

        String subjectID = (String) auditContext.get(SessionContext.USER_ID);

        if (subjectID == null) {
            return ILogger.NONROLEUSER;
        }

        return subjectID.trim();
    }

    /**
     * Signed Audit Log Group ID
     *
     * This method is inherited by all extended "CMSServlet"s,
     * and is called to obtain the "gid" for
     * a signed audit log message.
     * <P>
     *
     * @return id string containing the signed audit log message SubjectID
     */
    protected String auditGroupID() {

        logger.debug("CMSServlet: in auditGroupID");
        String groupID = null;

        // Initialize groupID
        SessionContext auditContext = SessionContext.getExistingContext();

        // logger.debug("CMSServlet: auditGroupID auditContext " + auditContext);
        if (auditContext != null) {
            groupID = (String)
                    auditContext.get(SessionContext.GROUP_ID);

            logger.debug("CMSServlet auditGroupID: groupID: " + groupID);
            if (groupID != null) {
                groupID = groupID.trim();
            } else {
                groupID = ILogger.NONROLEUSER;
            }
        } else {
            groupID = ILogger.UNIDENTIFIED;
        }

        return groupID;
    }

    /**
     * Signed Audit Groups
     *
     * This method is called to extract all "groups" associated
     * with the "auditSubjectID()".
     * <P>
     *
     * @param SubjectID string containing the signed audit log message SubjectID
     * @return a delimited string of groups associated
     *         with the "auditSubjectID()"
     */
    private String auditGroups(String SubjectID) {

        if (SubjectID == null || SubjectID.equals(ILogger.UNIDENTIFIED)) {
            return null;
        }

        Enumeration<Group> groups = null;

        try {
            groups = mUG.findGroups("*");
        } catch (Exception e) {
            return null;
        }

        StringBuffer membersString = new StringBuffer();

        while (groups.hasMoreElements()) {
            Group group = groups.nextElement();

            if (group.isMember(SubjectID)) {
                if (membersString.length() != 0) {
                    membersString.append(", ");
                }

                membersString.append(group.getGroupID());
            }
        }

        if (membersString.length() == 0) {
            return null;
        }

        return membersString.toString();
    }

    /**
     * Retrieves locale based on the request.
     */
    protected Locale getLocale(HttpServletRequest req) {
        Locale locale = null;
        String lang = req.getHeader(HDR_LANG);

        if (lang == null) {
            // use server locale
            locale = Locale.getDefault();
        } else {
            locale = new Locale(UserInfo.getUserLanguage(lang),
                        UserInfo.getUserCountry(lang));
        }
        return locale;
    }

    protected void outputResult(HttpServletResponse httpResp,
            String contentType, byte[] content) {
        try {
            OutputStream os = httpResp.getOutputStream();

            httpResp.setContentType(contentType);
            httpResp.setContentLength(content.length);
            os.write(content);
            os.flush();
        } catch (IOException e) {
            logger.warn(CMS.getLogMessage("CMSGW_ERR_BAD_SERV_OUT_STREAM", "", e.toString()), e);
            return;
        }
    }

    protected void outputError(HttpServletResponse httpResp, String errorString) {
        outputError(httpResp, FAILURE, errorString, null);
    }

    protected void outputError(HttpServletResponse httpResp, String errorString, String requestId) {
        outputError(httpResp, FAILURE, errorString, null);
    }

    protected void outputError(HttpServletResponse httpResp, String status, String errorString, String requestId) {
        XMLObject xmlObj = null;
        try {
            xmlObj = new XMLObject();
            Node root = xmlObj.createRoot("XMLResponse");
            xmlObj.addItemToContainer(root, "Status", status);
            xmlObj.addItemToContainer(root, "Error", errorString);
            if (requestId != null) {
                xmlObj.addItemToContainer(root, "RequestId", requestId);
            }
            byte[] cb = xmlObj.toByteArray();

            OutputStream os = httpResp.getOutputStream();
            httpResp.setContentType("application/xml");
            httpResp.setContentLength(cb.length);
            os.write(cb);
            os.flush();
        } catch (Exception ee) {
            logger.warn(CMS.getLogMessage("CMSGW_ERR_BAD_SERV_OUT_STREAM", "", ee.toString()), ee);
        }
    }
}
